;
; rcsID=$Id: Convert.lisp,v 1.16 2008/04/29 16:25:15 tmatsugaki Exp $
;
(proclaim '(inline brailleEncoder
                   brailleDecoder
                   seek
                   seeksym
                   conv
                   conv_g2))
(proclaim '(optimize speed))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 brailleEncoder
;; 【機能　】 変換元の言語連想配列を使用して墨字文字列を数値化する。
;;            その数値が変換先の言語連想配列の cdr部 と合致するコンスセルの car部 を
;;            文字列化して墨字を生成する。
;; 【入力　】 sym  : 墨字シンボル（※intern する文字列に '|' が含まれないこと）
;;          slst : 変換元の言語連想配列
;; 【出力　】 点字文字列
;; 【例外　】 なし
;; 【使用例】 (brailleEncoder (intern "あ") *kana*) → "A"
;;            (brailleEncoder (intern "íː") *pronunciation*) → "I3"
;;            (brailleEncoder (intern "but") *us_grade2*) → "B "
;;            (brailleEncoder (intern "about") *us_grade2*) → "AB "
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun brailleEncoder (sym slist
         &optional (wholeWord nil) (initWord nil) (midWord nil) (finWord nil))
  (let ((ret nil) val cval (forceG2 nil) (cell nil))
;(format t "~S~%" (length (string sym)))
;(format t "~%~S, ~S, ~S, ~S, ~S" sym wholeWord initWord midWord finWord)
    ; 【エンコード前処理】
    ; 同じ墨字で複数の点字がある場合に主処理で自動的に処理するためのに即値を設定する。ugly
    ; 【凡例】外字符 vs セミコロンや括弧等
;(format t "~%~S [~3S] [~3S] [~3S] [~3S]" sym wholeWord initWord midWord finWord)
    (if (= *grade* 2)
      (progn
        ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
        ; 結合ありの場合の例外処理
        ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
        (if wholeWord
          (setf cell (assoc sym *us_grade2_excp_whole*))
          (if initWord
            (setf cell (assoc sym *us_grade2_excp_initial*))
            (if midWord
              (setf cell (assoc sym *us_grade2_excp_medial*))
              (if finWord
                (setf cell (assoc sym *us_grade2_excp_final*))
                ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
                ; 結合なしの場合の例外処理
                ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
                (progn
                  (setf cell (assoc sym *us_grade2_excp_none*))
                  (if cell
                    (setf forceG2 t)
                    nil)))))))
      nil)
    ;; 英語グレード２の例外ではない。
    (if (null cell)
      (setf cell (assoc sym slist))
      nil)
    ;; 該当の語のドット対が見つかった。
    (if cell
      (setf val (cdr cell))
       nil)
;(format t "~%#b~48,'0b" val)
    ; 【エンコード主処理】
    (if (numberp val)
      (if (or wholeWord initWord midWord finWord forceG2)
        ; 結合があるか、または強制グレード２の場合
        (progn
          ; ドット対の cdr のフラグをオフにする。
          (setf val (logand _AllClearBits_ val))
          (cond
            ; 英語グレード２の Whole-Word は cdr の MSB が立っているので適合させる。
            (wholeWord (setf cval (logior val _wholeWordMask_)))
            ; 英語グレード２の finWord-Word は cdr の MSB-3 が立っているので適合させる。
            (finWord (setf cval (logior val _finalWordMask_)))
            ; 英語グレード２の midWord-Word は cdr の MSB-2 が立っているので適合させる。
            (midWord (setf cval (logior val _medialWordMask_)))
            ; 英語グレード２の initWord-Word は cdr の MSB-1 が立っているので適合させる。
            (initWord (setf cval (logior val _initialWordMask_))))
          ; 1.先ず、結合ありの候補を検索する。
          (setf ret (conv_g2 sym cval slist *represent*))
          ; 2.最適な物がない場合は、結合なしの候補を検索する。
          (if (and (null ret) (= (length (string sym)) 1))
            (setf ret (conv_g2 sym val slist *represent*))
            nil))
        ; 結合がなく、かつ強制グレード２英語でもない場合
        (setf ret (conv sym slist *represent*)))
      nil)
;(format t "~%~A #b~48,'0b" sym ret)
;(format t "~%#b~48,'0b" cval)
    ret))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 brailleDecoder
;; 【機能　】 表示用の言語連想配列を使用して点字文字列を数値化する。
;;            その数値が変換先の言語連想配列の cdr部 と合致するコンスセルの car部 を
;;            文字列化して墨字を生成する。
;; 【入力　】 str  : 点字文字列
;;            dlst : 変換先の言語連想配列
;; 【出力　】 墨字文字列
;; 【例外　】 なし
;; 【使用例】 (brailleDecoder "A" *kana*) → ア
;;          (brailleDecoder "I3" *pronunciation*) → íː
;;          (brailleDecoder "B" *us_grade2* t) → but
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun brailleDecoder (str dlist
         &optional (wholeWord nil) (initWord nil) (midWord nil) (finWord nil))
  (let ((ret nil) (capital nil) (val 0) s)
    ; 【デコード前処理】
    (do ((i 0 (1+ i)))
      ((= i (length str)) nil)
      (let (cc)
        (setf cc (assoc (intern (subseq str i (1+ i))) *represent*))
        (if (and (consp cc) (numberp (cdr cc)))
          (setf val (+ val (* (cdr cc) (nth (- (length str) (1+ i)) _power_of_64_))))
          nil)))
    ; 【デコード主処理】
    (if (or wholeWord initWord midWord finWord)
      ; 結合がある場合
      (let (flags)
        (cond
          (wholeWord (setf flags _wholeWordMask_))
          (initWord  (setf flags _initialWordMask_))
          (midWord   (setf flags _medialWordMask_))
          (finWord   (setf flags _finalWordMask_))
          (t         (setf flags 0)))
        (setf s (car (rassoc (logior val flags) dlist)))
        (if (and (null s) wholeWord)
          (progn
            ; wholeWord で見つからない場合は、英語グレード２の大文字符の可能性もある。（現在は、his[,8] のみ）
            (setf s (car (rassoc (logior val flags _capitalMask_) dlist)))
            (if s
              (setf capital t)
              nil))
          nil)
        ; 見つからない場合は、結合なしを検索する。
        (if (null s)
          (setf s (car (rassoc val dlist)))
          nil))
      ; 結合がない場合
      (setf s (car (rassoc val dlist))))
    (if s
      (setf ret (string s))
      nil)
    (values (substitute-string ret "|" "") capital)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 seek
;; 【機能　】 言語連想配列に対して点字１文字分の値を指定し検索する。
;;            指定した値で見つからない場合は、右シフトをした値で検索を継続する。
;;            最終結果のみをドット対（シフト回数・値）で出力する。
;; 【入力　】 ilst : 検索対象の言語連想配列
;;            unit : 点字１区画の値（点字の１マスは、2^6=64）
;;            val  : 点字１文字分の値
;;            nth  : 初期値（0 を指定する）
;;            olst : 出力用リスト（nil を指定する）
;; 【出力　】 リスト（シフト回数・値）
;; 【例外　】 なし
;; 【使用例】 (seek *represent* 64 357116544 0 nil)  → (4 21)
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun seek (ilst unit val nth olst)
  (if (cdr (rassoc val ilst))
    (list nth (cdr (rassoc val ilst)))
    (if (> nth 7)
      (list nth -1)
      (if (and (numberp val) (numberp unit))
        (let (q)
          (setf q (truncate val unit))
          (seek ilst unit q (1+ nth) olst))
        (list nth -2)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 seeksym
;; 【機能　】 言語連想配列に対して点字１文字分の値を指定し検索し、結果をシンボルの
;;            リストで出力する。
;; 【入力　】 ilst : 検索対象の言語連想配列
;;            unit : 点字１区画の値（点字の１マスは、2^6=64）
;;            val  : 点字１文字分の値
;; 【出力　】 シンボルまたは数値のリスト
;; 【例外　】 なし
;; 【使用例】 (seeksym *represent* (cdr (assoc '☆ *common*))) → (9 9 | |)
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun seeksym (ilst val)
  (let (l olst)
    ; （英語グレード２に関わらず）言語連想配列の cdr のフラグをオフにする。
    (if (numberp val)
      (setf val (logand _AllClearBits_ val))
      nil)
    (setf l (seek ilst 64 val 0 nil))
    (setf olst (cons (car (rassoc (car (cdr l)) ilst)) nil))
    (do ((i (car l) (1- i)))
      ((= i 0) nil)
      (progn
        (setf l (seek ilst 64 (mod val (nth i _power_of_64_)) 0 nil))
        (if (listp l)
          (setf olst (cons (car (rassoc (car (cdr l)) ilst)) olst))
          nil)))
    (reverse olst)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 conv
;; 【機能　】 変換元の言語連想配列にて点字１文字を検索し数値化し、その数値を変換先の
;;            言語連想配列を使用して文字列化する。
;; 【入力　】 sym  : 検索対象の文字列
;;            slst : 変換元の言語連想配列
;;            dlst : 変換先の言語連想配列
;; 【出力　】 文字列
;; 【例外　】 なし
;; 【使用例】 (conv 'あ *kana* *represent*) → "A"
;;            (conv 'a *represent* *kana*) → "ア"
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun conv (sym slst dlst)
    (let ((s nil))
      (dolist (x (seeksym dlst (cdr (assoc sym slst))))
        (if x
          (setf s (concatenate 'string s (princ-to-string x)))
          nil))
      (substitute-string s "|" "")))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 conv_g2（*us_grade2* → *represent* 専用）
;; 【機能　】 変換元の言語連想配列にあると想定される数値の点字を、変換先の
;;            言語連想配列を使用して文字列化する。
;; 【入力　】 sym  : 検索対象の文字列
;;            val  : 変換元の言語連想配列にあると想定される点字の数値
;;            slst : 変換元の言語連想配列
;;            dlst : 変換先の言語連想配列
;; 【出力　】 文字列
;; 【例外　】 なし
;; 【使用例】 (conv_g2 (intern "con") 70368744177682 *us_grade2* *represent*) → "3"
;;            (conv_g2 (intern "con") 35184372088850 *us_grade2* *represent*) → "3"
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun conv_g2 (sym val slst dlst)
  (let ((s nil))
    (if (equal (car (rassoc val slst)) sym)
      (dolist (x (seeksym dlst val))
        (if x
          (setf s (concatenate 'string s (princ-to-string x)))
          nil))
      nil)
    (substitute-string s "|" "")))
