;
; rcsID=$Id: Hyphenation.lisp,v 1.16 2008/04/29 16:25:15 tmatsugaki Exp $
;
(proclaim '(inline
;                   genHypenationDictionary
                   genHyphenationDicInfo
))
(proclaim '(optimize speed))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 getHyphenateCandidates
;; 【機能　】 文字列をハイフネーションした結果のリストを返す。
;; 【入力　】 dic : ハイフネーション辞書
;;            buff : 1文字以上の文字列
;; 【出力　】 文字列リストのリスト、エラー情報のリスト
;; 【例外　】 なし
;; 【使用例】 (getHyphenateCandidates *us_hyphen_dict* "aaaaaaaaaa")
;;           → (("aaaaaaaaaa")) ;
;;              NIL
;;            (getHyphenateCandidates *us_hyphen_dict* "immdiately")
;;           → (("im" "mdiately")) ;
;;              NIL
;;            (getHyphenateCandidates *us_hyphen_dict* "technology")
;;           → (("tech" "nology") ("te" "chnology")) ;
;;              NIL
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun getHyphenateCandidates (dic buff)
  (let ((ret nil) (errList nil) (len (length buff))
       (lst nil) (lstr nil) (mstr nil) (rstr nil) s)
    (do ((left 0 (1+ left))) ; 左のインデクスのループ
      ((or lst (>= left len)) nil)
      (do ((right len (1- right))) ; 右のインデクスのループ
        ((or lst (>= left right)) nil)
        (setf s (subseq buff left right))
        (if (null lst)
          (setf lst (assoc (intern s) dic))
          nil)
        (if lst
          (progn
;(format t "~%~S ~S ~S" lst left right)
            (if (and (= left 0) (string= (subseq (nth 2 lst) 0 1) "-"))
              ; 検出したが不適切なので、差し戻す。
              (setf lst nil)
              ; 検出した。
              (progn
                (setf mstr (nth 2 lst))
                (setf rstr (concatenate 'string rstr (subseq buff right))))))))
      (if (null lst)
        (setf lstr (concatenate 'string lstr (subseq buff left (1+ left))))
        nil))
    ; ハイフネーションの候補リストを作成する。
    (progn
      (setf s (join-string (list lstr mstr rstr) ""))
      (setf lst (split-string s "-"))
      (cond
        ; ハイフネーション不可(候補は1つで、トークンは 1つ)
        ((= (length lst) 1) (setf ret (list lst)))
        ; ハイフネーション可能(候補は1つで、トークンは 2つ)
        ((= (length lst) 2) (setf ret (list lst)))
        ; ハイフネーション可能(候補が複数存在し、トークンは 2つ)
        (t
          (let ((candidates nil) left-list right-list)
            (do ((i (1- (length lst)) (1- i)))
              ((= i 0) nil)
                (setf right-list (nthcdr i lst))
                (setf left-list (ldiff lst right-list))
                (setf candidates
                  (append candidates
                    (list (list (join-string left-list "")
                    (join-string right-list ""))))))
            (setf ret candidates)))))
    (values ret errList)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 genHypenationDictionary
;; 【機能　】 ファイルのデータからハイフネーション(米国)辞書の連想配列を作成する。
;; 【入力　】 infn : ハイフネーション辞書データファイル（入力）のパス
;; 【出力　】 エラー情報のリスト
;; 【例外　】 なし
;; 【使用例】 (genHyphenationDictionary "../dic/hyph_en_US.dic")
;;           → nil
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun genHyphenationDictionary (infn)
  (let ((dic nil) buff (lineNo 1) (errList nil))
    ;; 辞書入力ファイルをオープンする。
    (with-open-file (instream infn :direction :input)
      (do ()
        ((null (setf buff (read-line instream nil))) nil)
        ; 1行エンコードする。
        (if (> (length buff) 1)
          (progn
            ; 1行目を読み飛ばす
            (if (> lineNo 1)
              (let (hyph e)
                (multiple-value-setq (hyph e) (genHyphenationDicInfo buff))
                (if e (push e errList) nil)
                (push hyph dic))
              nil)
            (incf lineNo))
          nil))
      ;; 辞書入力ファイルをクローズする。
      (close instream))
    (values dic errList)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; 【関数名】 genHyphenationDicInfo
;; 【機能　】 1行の文字列からハイフネーション(米国)辞書の連想配列のメンバー情報を作成する。
;; 【入力　】 buff   : 文字列
;; 【出力　】 ハイフネーション情報、エラー情報のリスト
;; 【例外　】 なし
;; 【使用例】 (genHyphenationDicInfo ".ab4i") → (|abi| ".ab4i" "-ab-i") ;
;;                                             NIL
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun genHyphenationDicInfo (buff)
  (let ((ret nil) s c (sym nil) (hyphenated nil) (len (length buff)) errList)
    (do ((i 0 (1+ i)))
      ((>= i len) nil)
      (setf s (subseq buff i (1+ i)))
      (setf c (char s 0))
      ; 検索用シンボル作成
      (if (both-case-p c)
        (setf sym (concatenate 'string sym s))
        nil)
      ; 置換用文字列作成
      (cond
        ((char= c #\.) (setf hyphenated (concatenate 'string hyphenated "-")))
        ((digit-char-p c) (setf hyphenated (concatenate 'string hyphenated "-")))
        (t (setf hyphenated (concatenate 'string hyphenated s)))))
    (setf ret (list (intern sym) buff hyphenated))
    (values ret errList)))
