/*
 *  Breeze  --  An application launcher with command-line style
 *  Copyright (C) 2005, 2006, 2008 Hironao Komatsu
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <sys/wait.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>

#include <glib/gi18n.h>
#include <gtk/gtk.h>

#include "commands.h"
#include "mainwin.h"

/*
 *  Edit here to grab another key.
 *
 *  Default is "Windows key + space" (PC/AT keyboard)
 */
#define DEFAULT_KEY XK_space
#define DEFAULT_KEY_MASK Mod4Mask

volatile gboolean continue_loop_flag = TRUE;

static int key_to_grab;
static unsigned int modifiers_to_grab;
static unsigned int NumLockMask, CapsLockMask, ScrollLockMask;

static Display *dpy = NULL;
static Window root;

static void signal_handler(int sig);
static void init_keyboard(void);
static void grab_key(int keycode, unsigned int modifiers, Window w);
static unsigned int get_modifiers_to_grab(const char *str);
static void initialize(int argc, char **argv);
static void finalize(void);

static void signal_handler(int sig)
{
  switch (sig) {
  case SIGTERM:
  case SIGINT:
  case SIGHUP:
    finalize();
    exit(0);
  case SIGSEGV:
    finalize();
    exit(1);
  case SIGCHLD:
    wait(NULL);
    break;
  default:
    ;
  }
}

/*
 *  Functions init_keyboard and grab_key are originally from keylaunch.c;
 *  written by Ken Lynch and Stefan Pfetzing.
 */
static void init_keyboard(void)
{
  XModifierKeymap *xmk = NULL;
  KeyCode *map;
  int m, k;

  xmk = XGetModifierMapping(dpy);
  if (xmk) {
    map = xmk->modifiermap;
    for (m = 0; m < 8; m++)
      for (k = 0; k < xmk->max_keypermod; k++, map++) {
        if (*map == XKeysymToKeycode(dpy, XK_Num_Lock))
          NumLockMask = (1 << m);
        if (*map == XKeysymToKeycode(dpy, XK_Caps_Lock))
          CapsLockMask = (1 << m);
        if (*map == XKeysymToKeycode(dpy, XK_Scroll_Lock))
          ScrollLockMask = (1 << m);
      }
    XFreeModifiermap(xmk);
  }
}

static void grab_key(int keycode, unsigned int modifiers, Window w)
{
  const unsigned int mods[] = {
    0,
    NumLockMask,
    CapsLockMask,
    ScrollLockMask,
    NumLockMask | CapsLockMask,
    CapsLockMask | ScrollLockMask,
    NumLockMask | CapsLockMask | ScrollLockMask,
  };

  if (keycode) {
    int i;

    for (i = 0; i < sizeof(mods) / sizeof(*mods); i++) {
      XGrabKey(dpy, keycode, modifiers | mods[i],
	       w, False, GrabModeAsync, GrabModeAsync); 
    }
  }
}

static unsigned int get_modifiers_to_grab(const char *str)
{
  const char *modifiers_str[] = {
    "Control", "Shift", "Mod1", "Mod2", "Mod3", "Mod4", "Mod5" };
  const int modifiers[] = {
    ControlMask, ShiftMask, Mod1Mask, Mod2Mask, Mod3Mask, Mod4Mask, Mod5Mask }; 
  const int num = sizeof(modifiers_str) / sizeof(*modifiers_str);
  int i;

  for (i = 0; i < num; i++)
    if (strcmp(str, modifiers_str[i]) == 0)
      return modifiers[i];

  return 0;
}

static void initialize(int argc, char **argv)
{
  gchar *keysym_str = NULL;
  gchar **modifier_str_v = NULL;

  GOptionContext *context;
  const GOptionEntry entries[] = {
    { "key", 'k', 0, G_OPTION_ARG_STRING, &keysym_str,
      _("set KEY as the key to grab"), "KEY" },
    { "modifier", 'm', 0, G_OPTION_ARG_STRING_ARRAY, &modifier_str_v,
      _("set MOD as the modifier of KEY"), "MOD" },
    { NULL },
  };
  const gchar *desc = _("MOD is one of:\n"
			"  Control, Shift, Mod1, Mod2, Mod3, Mod4, Mod5\n"
			"\n"
			"If no KEY specified, default KEY and MOD (Mod4 + spece) will be used.\n");
  GError *err = NULL;

  KeySym keysym = 0L;
  unsigned int modifiers = 0;
  struct sigaction sa;

  context = g_option_context_new("breeze");
  g_option_context_add_main_entries(context, entries, NULL);
  g_option_context_add_group(context, gtk_get_option_group(TRUE));
  g_option_context_set_help_enabled(context, TRUE);
  g_option_context_set_description(context, desc);
  g_option_context_parse(context, &argc, &argv, &err);

  if (err) {
    g_critical(_("Error handling option: %s\n"), err->message);
    g_error_free(err);
    exit(1);
  }

  if (keysym_str) {
    keysym = XStringToKeysym(keysym_str);
    if (keysym == NoSymbol)
      g_warning(_("unknown key: %s. ignored.\n"), keysym_str);
  }
  if (modifier_str_v) {
    gchar** p;

    for (p = modifier_str_v; *p; p++) {
      unsigned int mod = get_modifiers_to_grab(*p);

      if (mod == 0)
	g_warning(_("unknown modifier key: %s. ignored.\n"), *p);
      else
	modifiers |= mod;
    }
  }

  sa.sa_handler = signal_handler;
  sa.sa_flags = 0;

  sigaction(SIGTERM, &sa, NULL);
  sigaction(SIGINT, &sa, NULL);
  sigaction(SIGSEGV, &sa, NULL);
  sigaction(SIGHUP, &sa, NULL);
  sigaction(SIGCHLD, &sa, NULL);

  if ((dpy = XOpenDisplay(NULL)) == NULL) {
    g_critical(_("Can't open display.\n"));
    exit(1);
  }

  if (initialize_commands()) {
    g_critical(_("Can't initialize commands table.\n"));
    XCloseDisplay(dpy);
    exit(1);
  }

  root = XDefaultRootWindow(dpy);

  init_keyboard();

  if (keysym)
    key_to_grab = XKeysymToKeycode(dpy, keysym);
  else {
    key_to_grab = XKeysymToKeycode(dpy, DEFAULT_KEY);
    modifiers = DEFAULT_KEY_MASK;
  }

  modifiers_to_grab = modifiers;

#ifdef DEBUG
  g_debug("(key, modifier) = (%d, %d)\n", key_to_grab, modifiers_to_grab);
#endif

  grab_key(key_to_grab, modifiers_to_grab, root);

  gtk_init(&argc, &argv);
}

static void finalize(void)
{
  finalize_commands();

  XUngrabKey(dpy, AnyKey, AnyModifier, root);
  XCloseDisplay(dpy);
}

int main(int argc, char **argv)
{
#ifdef ENABLE_NLS
  bindtextdomain(GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
  bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
  textdomain(GETTEXT_PACKAGE);
#endif

  initialize(argc, argv);

  while (continue_loop_flag) {
    if (XPending(dpy) > 0) {
      XEvent ev;
      XNextEvent(dpy, &ev);

      if (ev.type == KeyPress &&
	  ev.xkey.keycode == key_to_grab &&
	  (modifiers_to_grab == 0 || ev.xkey.state & modifiers_to_grab)) {
	GtkWidget *win = main_win_new();
	main_win_run(MAIN_WIN(win));
      }
    }
    while (XPending(dpy) > 0) {
      XEvent ev;
      XNextEvent(dpy, &ev);
    }
    usleep(250);
  }

  finalize();

  return 0;
}
