/*
 *  Breeze  --  An application launcher with command-line style
 *  Copyright (C) 2005, 2006, 2008 Hironao Komatsu
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <sys/wait.h>

#include <glib/gi18n.h>
#include <gdk/gdkx.h>
#include <gtk/gtk.h>

#include "commands.h"
#include "mainwin.h"
#include "keyconfig.h"

volatile gboolean continue_loop_flag = TRUE;
static KeyConfig *kconf = NULL;

static void signal_handler(int sig);
static void initialize(int argc, char **argv);
static void finalize(void);

static void signal_handler(int sig)
{
  switch (sig) {
  case SIGTERM:
  case SIGINT:
  case SIGHUP:
    finalize();
    exit(0);
  case SIGSEGV:
    finalize();
    exit(1);
  case SIGCHLD:
    wait(NULL);
    break;
  default:
    ;
  }
}

static void initialize(int argc, char **argv)
{
  gchar *keysym_str = NULL;
  gchar **modifier_str_v = NULL;

  GOptionContext *context;
  const GOptionEntry entries[] = {
    { "key", 'k', 0, G_OPTION_ARG_STRING, &keysym_str,
      _("set KEY as the key to grab"), "KEY" },
    { "modifier", 'm', 0, G_OPTION_ARG_STRING_ARRAY, &modifier_str_v,
      _("set MOD as the modifier of KEY"), "MOD" },
    { NULL },
  };
  const gchar *desc = _("MOD is one of:\n"
			"  Control, Shift, Mod1, Mod2, Mod3, Mod4, Mod5\n"
			"\n"
			"If no KEY specified, default KEY and MOD (Mod4 + spece) will be used.\n");
  GError *err = NULL;

  struct sigaction sa;

  Display *dpy;
  Window root;

  context = g_option_context_new("breeze");
  g_option_context_add_main_entries(context, entries, NULL);
  g_option_context_add_group(context, gtk_get_option_group(TRUE));
  g_option_context_set_help_enabled(context, TRUE);
  g_option_context_set_description(context, desc);
  g_option_context_parse(context, &argc, &argv, &err);

  if (err) {
    g_critical(_("Error handling option: %s\n"), err->message);
    g_error_free(err);
    exit(1);
  }

  sa.sa_handler = signal_handler;
  sa.sa_flags = 0;

  sigaction(SIGTERM, &sa, NULL);
  sigaction(SIGINT, &sa, NULL);
  sigaction(SIGSEGV, &sa, NULL);
  sigaction(SIGHUP, &sa, NULL);
  sigaction(SIGCHLD, &sa, NULL);

  gtk_init(&argc, &argv);

  if (initialize_commands()) {
    g_critical(_("Can't initialize commands table.\n"));
    exit(1);
  }

  dpy = GDK_DISPLAY();
  root = GDK_ROOT_WINDOW();

  kconf = malloc(sizeof(*kconf));
  memset(kconf, 0, sizeof(*kconf));

  kconf_initialize(kconf, dpy, root, keysym_str, modifier_str_v);
}

static void finalize(void)
{
  Display *dpy = GDK_DISPLAY();
  Window root = GDK_ROOT_WINDOW();

  finalize_commands();

  XUngrabKey(dpy, AnyKey, AnyModifier, root);

  if (kconf) {
    free(kconf);
    kconf = NULL;
  }
}

int main(int argc, char **argv)
{
  Display *dpy;

#ifdef ENABLE_NLS
  bindtextdomain(GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
  bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
  textdomain(GETTEXT_PACKAGE);
#endif

  initialize(argc, argv);

  dpy = GDK_DISPLAY();

  while (continue_loop_flag) {
    if (XPending(dpy) > 0) {
      XEvent ev;
      XNextEvent(dpy, &ev);

      if (ev.type == KeyPress &&
	  ev.xkey.keycode == kconf->key_to_grab &&
	  (kconf->modifiers_to_grab == 0 ||
	   ev.xkey.state & kconf->modifiers_to_grab)) {
	GtkWidget *win = main_win_new();
	guint32 timestamp = ev.xkey.time;

	main_win_run(MAIN_WIN(win), timestamp);
      }
    }
    while (XPending(dpy) > 0) {
      XEvent ev;
      XNextEvent(dpy, &ev);
    }
    usleep(250);
  }

  finalize();

  return 0;
}
