
#include "BRbBase.h"
#include "BRbInputBuffer.h"
#include "BRbOutputBuffer.h"
#include "BRbReadNode.h"
#include "BRbWriteNode.h"


extern "C" {
	VALUE ruby_top_self;         // object.c
	VALUE rb_eval(VALUE, NODE*); // eval.c
}


static VALUE brb_read_file(VALUE filename);
static NODE* brb_compile(VALUE source, VALUE filename);
static VALUE rb_s_dump(VALUE obj, VALUE source, VALUE filename);
static VALUE rb_s_dumpfile(VALUE obj, VALUE filename);
static VALUE rb_s_load(VALUE obj, VALUE bytecode);
static VALUE rb_s_loadfile(VALUE obj, VALUE filename);
extern "C" void Init_bruby();


bool  g_debug         = false;
VALUE rb_mBRb          = 0;
VALUE rb_mBRbException = 0;


static VALUE
brb_read_file(VALUE filename)
{
	VALUE mode = rb_str_new2("rb");
	VALUE file = rb_funcall(rb_cFile, rb_intern("open"), 2, filename, mode);
	VALUE text = rb_funcall(file, rb_intern("read"), 0);
	rb_funcall(file, rb_intern("close"), 0);
	return text;
}

static NODE*
brb_compile(VALUE source, VALUE filename)
{
	const char* basename = STR2CSTR(rb_funcall(rb_cFile, rb_intern("basename"), 1, filename));
	const int   lineno   = 1;
	return rb_compile_string(basename, source, lineno);
}

static VALUE
rb_s_dump(VALUE obj, VALUE source, VALUE filename)
{
	Check_Type(source,   T_STRING);
	Check_Type(filename, T_STRING);

	NODE* root = brb_compile(source, filename);
	BRbOutputBuffer buffer;
	BRbWriteNode writer(buffer, g_debug);

	try {
		writer.write(root);
	} catch (char* e) {
		rb_raise(rb_mBRbException, e);
	}

	return buffer.to_string();
}

static VALUE
rb_s_dumpfile(VALUE obj, VALUE filename)
{
	Check_Type(filename, T_STRING);

	VALUE source   = brb_read_file(filename);
	VALUE bytecode = rb_s_dump(obj, source, filename);

	return bytecode;
}

static VALUE
rb_s_load(VALUE obj, VALUE bytecode)
{
	Check_Type(bytecode, T_STRING);

	try {
		BRbInputBuffer buffer(bytecode);
		BRbReadNode reader(buffer, g_debug);
		NODE* root = reader.read();
		rb_eval(ruby_top_self, root);
	} catch (char* e) {
		rb_raise(rb_mBRbException, e);
	}

	return Qnil;
}

static VALUE
rb_s_loadfile(VALUE obj, VALUE filename)
{
	Check_Type(filename, T_STRING);

	VALUE bytecode = brb_read_file(filename);
	rb_s_load(obj, bytecode);

	return Qnil;
}

static VALUE
rb_s_debug_get(VALUE obj)
{
	return ( g_debug ? Qtrue : Qfalse );
}

static VALUE
rb_s_debug_set(VALUE obj, VALUE debug)
{
	g_debug = ( RTEST(debug) ? true : false);
	return debug;
}

extern "C" void
Init_bruby()
{
	rb_mBRb = rb_define_module("BRb");
	rb_define_singleton_method(rb_mBRb, "dump", (VALUE(*)(...))rb_s_dump, 2);
	rb_define_singleton_method(rb_mBRb, "dumpfile", (VALUE(*)(...))rb_s_dumpfile, 1);
	rb_define_singleton_method(rb_mBRb, "load", (VALUE(*)(...))rb_s_load, 1);
	rb_define_singleton_method(rb_mBRb, "loadfile", (VALUE(*)(...))rb_s_loadfile, 1);
	rb_define_singleton_method(rb_mBRb, "debug", (VALUE(*)(...))rb_s_debug_get, 0);
	rb_define_singleton_method(rb_mBRb, "debug=", (VALUE(*)(...))rb_s_debug_set, 1);

	rb_mBRbException = rb_define_class("BRbException", rb_eException);
}
