// $Id: BRb.cpp,v 1.12 2002/08/18 15:39:58 yuya Exp $

#include "BRbBase.h"
#include "BRbInputBuffer.h"
#include "BRbOutputBuffer.h"
#include "BRbReadNode.h"
#include "BRbWriteNode.h"

////////////////////////////////////////////////////////////////////////////////

extern "C" {
	VALUE ruby_top_self;         // object.c
	VALUE rb_eval(VALUE, NODE*); // eval.c
	void top_local_init();       // parse.c
	void top_local_setup();      // parse.c
}

////////////////////////////////////////////////////////////////////////////////

static VALUE bruby_read_file(VALUE filename);
static NODE* bruby_compile(VALUE source, VALUE filename);
static VALUE bruby_dumpnode(NODE* node);
static NODE* bruby_loadnode(VALUE bytecode);

static VALUE rb_s_dump(VALUE obj, VALUE source, VALUE filename);
static VALUE rb_s_dumpfile(VALUE obj, VALUE filename);
static VALUE rb_s_load(VALUE obj, VALUE bytecode);
static VALUE rb_s_loadfile(VALUE obj, VALUE filename);
static VALUE rb_s_verify(VALUE obj, VALUE source, VALUE filename);
static VALUE rb_s_verifyfile(VALUE obj, VALUE filename);
static VALUE rb_s_debug_get(VALUE obj);
static VALUE rb_s_debug_set(VALUE obj, VALUE debug);

////////////////////////////////////////////////////////////////////////////////

extern "C" void Init_bruby();

////////////////////////////////////////////////////////////////////////////////

bool  g_debug          = false;
VALUE rb_mBRb          = 0;
VALUE rb_mBRbException = 0;

////////////////////////////////////////////////////////////////////////////////

static VALUE
bruby_read_file(VALUE filename)
{
	VALUE mode = rb_str_new2("rb");
	VALUE file = rb_funcall(rb_cFile, rb_intern("open"), 2, filename, mode);
	VALUE text = rb_funcall(file, rb_intern("read"), 0);
	rb_funcall(file, rb_intern("close"), 0);
	return text;
}

static NODE*
bruby_compile(VALUE source, VALUE filename)
{
	const char* basename = STR2CSTR(rb_funcall(rb_cFile, rb_intern("basename"), 1, filename));
	const int   lineno   = 1;
	return rb_compile_string(basename, source, lineno);
}

static VALUE
bruby_dumpnode(NODE* node)
{
	BRbOutputBuffer buffer;
	BRbWriteNode writer(buffer, g_debug);

	try {
		writer.write(node);
	} catch (const char* e) {
		rb_raise(rb_mBRbException, e);
	}

	return buffer.to_string();
}

static NODE*
bruby_loadnode(VALUE bytecode)
{
	BRbInputBuffer buffer(bytecode);
	BRbReadNode reader(buffer, g_debug);

	try {
		return reader.read();
	} catch (const char* e) {
		rb_raise(rb_mBRbException, e);
	}
}

////////////////////////////////////////////////////////////////////////////////

static VALUE
rb_s_dump(VALUE obj, VALUE source, VALUE filename)
{
	Check_Type(source,   T_STRING);
	Check_Type(filename, T_STRING);

	NODE* root     = bruby_compile(source, filename);
	VALUE bytecode = bruby_dumpnode(root);

	return bytecode;
}

static VALUE
rb_s_dumpfile(VALUE obj, VALUE filename)
{
	Check_Type(filename, T_STRING);

	VALUE source   = bruby_read_file(filename);
	VALUE bytecode = rb_s_dump(obj, source, filename);

	return bytecode;
}

static VALUE
rb_s_load(VALUE obj, VALUE bytecode)
{
	Check_Type(bytecode, T_STRING);

	NODE* root = bruby_loadnode(bytecode);
	top_local_init();
	top_local_setup();
	rb_eval(ruby_top_self, root);

	return Qnil;
}

static VALUE
rb_s_loadfile(VALUE obj, VALUE filename)
{
	Check_Type(filename, T_STRING);

	VALUE bytecode = bruby_read_file(filename);
	rb_s_load(obj, bytecode);

	return Qnil;
}

static VALUE
rb_s_verify(VALUE obj, VALUE source, VALUE filename)
{
	Check_Type(source,   T_STRING);
	Check_Type(filename, T_STRING);

	VALUE bytecode1 = rb_s_dump(obj, source, filename);
	NODE* root      = bruby_loadnode(bytecode1);
	VALUE bytecode2 = bruby_dumpnode(root);

	return rb_funcall(bytecode1, rb_intern("=="), 1, bytecode2);
}

static VALUE
rb_s_verifyfile(VALUE obj, VALUE filename)
{
	Check_Type(filename, T_STRING);

	VALUE source = bruby_read_file(filename);

	return rb_s_verify(obj, source, filename);
}

static VALUE
rb_s_debug_get(VALUE obj)
{
	return (g_debug ? Qtrue : Qfalse);
}

static VALUE
rb_s_debug_set(VALUE obj, VALUE debug)
{
	g_debug = (RTEST(debug) ? true : false);
	return debug;
}

////////////////////////////////////////////////////////////////////////////////

extern "C" void
Init_bruby()
{
	rb_mBRb = rb_define_module("BRb");
	rb_define_singleton_method(rb_mBRb, "dump", (VALUE(*)(...))rb_s_dump, 2);
	rb_define_singleton_method(rb_mBRb, "dumpfile", (VALUE(*)(...))rb_s_dumpfile, 1);
	rb_define_singleton_method(rb_mBRb, "load", (VALUE(*)(...))rb_s_load, 1);
	rb_define_singleton_method(rb_mBRb, "loadfile", (VALUE(*)(...))rb_s_loadfile, 1);
	rb_define_singleton_method(rb_mBRb, "verify", (VALUE(*)(...))rb_s_verify, 2);
	rb_define_singleton_method(rb_mBRb, "verifyfile", (VALUE(*)(...))rb_s_verifyfile, 1);
	rb_define_singleton_method(rb_mBRb, "debug", (VALUE(*)(...))rb_s_debug_get, 0);
	rb_define_singleton_method(rb_mBRb, "debug=", (VALUE(*)(...))rb_s_debug_set, 1);

	rb_mBRbException = rb_define_class("BRbException", rb_eException);
}

////////////////////////////////////////////////////////////////////////////////
