/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bt_coverage.c - coverage information display program                     */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2006                         */
/*             Authors: Yumiko Sugita (sugita@sdl.hitachi.co.jp),            */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include "bfd_if.h"
#include "bt.h"
#include <libgen.h>

#define BT_COVERAGE_VER	BT_VER
#define	COPYRIGHT	"Copyright (c) Hitachi, Ltd. 2005-2006"

#define	chk_next(argc, i)	if ((argc) <= (i)+1) { usage(); err_exit(); }
#define chk_html_opt_idx(i)	if ((i) >= 0) { usage(); err_exit(); }

struct range_to_name {
	unsigned long	begin;
	unsigned long	end;
	unsigned long	offset;
	char		name[MAX_NAME_LEN];
	char		*dirname;
	char		*basename;

	struct bfd_if	bi;
	struct path	**pt;
	long		cnt;
};

/*-----------------------------------------------------------------------------
 *  build range to name structure
 *-----------------------------------------------------------------------------
 */
struct range_to_name **all_r2n;
int r2n_max = 128;

void dump_r2n(void)
{
	int i;
	struct range_to_name *r2n;

	if (!all_r2n)
		return;
	for (i = 0; i < r2n_max; i++) {
		r2n = all_r2n[i];
		if (r2n) {
			printf("====== 0x%08lx:0x%08lx\t%s ======\n",
			       r2n->begin, r2n->end, r2n->basename);
			dump_bfd_symbols(&r2n->bi);
			dump_path_tree(r2n->pt, r2n->cnt);
		}
	}
}

void free_r2n(void)
{
	int i;
	struct range_to_name *r2n;

	if (!all_r2n)
		return;
	for (i = 0; i < r2n_max; i++) {
		r2n = all_r2n[i];
		if (r2n) {
			//close_mmapfile(r2n->fd, r2n->p_map, r2n->mapped_size);
			//free(r2n->fnames);
			free(r2n);
		}
	}
	free(all_r2n);
	all_r2n = NULL;
}

struct range_to_name* addr_to_r2n(unsigned long addr)
{
	int i;
	struct range_to_name *r2n;

	for (i = 0; i < r2n_max; i++) {
		r2n = all_r2n[i];
		if (r2n && addr >= r2n->begin && addr <= r2n->end)
			return r2n;
	}
	return NULL;
}

int prepare_obj_file(struct range_to_name *r2n, char *kallsyms)
{
	char path[MAX_LINE_LEN];

	snprintf(path, MAX_LINE_LEN, "%s/%s", r2n->dirname, r2n->basename);
	printf("checking %s...\n", path);
	if (init_bfd_if(&r2n->bi, strdup(path), kallsyms) < 0)
		return -1;
	if (!r2n->begin && !r2n->end) {
		struct addr_range *range = get_pid_addr_range(ALL_PID);
		bfd_vma min, max;
		struct bt_record rec;

		get_min_max_addr(&r2n->bi, &min, &max);
		rec.from = min;
		rec.to = max;
		if (is_addr_match(&rec, range)) {
			r2n->begin = min;
			r2n->end = max;
		} else
			return 1;
	} else {
		r2n->offset = get_offset_addr(&r2n->bi, r2n->begin);
		/* for DEBUG
		printf("OFFSET: ");
		printf_bfd_vma(r2n->offset);
		printf("\n");
		*/
	}
	return chk_path_tree(&r2n->bi, &r2n->pt, &r2n->cnt, r2n->offset);
}

int is_same_name_registered(char *name)
{
	int i, len;
	struct range_to_name *r2n;

	if (!all_r2n)
		return 0;
	for (i = 0; i < r2n_max; i++) {
		r2n = all_r2n[i];
		if (!r2n)
			continue;
		len = strlen(r2n->dirname);
		if (strstr(name, r2n->dirname) == name && name[len] == '/' &&
		    strstr(name, r2n->basename) == name + len + 1) {
			return 1;
		}
	}
	return 0;
}

int add_range_to_name(unsigned long begin, unsigned long end, char *name,
		      char *kallsyms, int is_module)
{
	int rc, i, step = 128;
	struct range_to_name *r2n;

	if (is_same_name_registered(name))
		return 0;
	r2n = malloc(sizeof(struct range_to_name));
	if (!r2n) {
		fprintf(stderr, "malloc failed.(%s)\n", strerror(errno));
		return -1;
	}
	memset(r2n, 0, sizeof(struct range_to_name));
	r2n->begin = begin;
	r2n->end = end;
	snprintf(r2n->name, MAX_NAME_LEN, "%s", name);
	r2n->basename = basename(r2n->name);
	r2n->dirname = dirname(r2n->name);

	for (i = 0; i < r2n_max && all_r2n[i]; i++);
	if (i >= r2n_max) {
		all_r2n = realloc(all_r2n, (r2n_max + step) * sizeof(r2n));
		if (!all_r2n) {
			free(r2n);
			fprintf(stderr, "realloc failed.(%s)\n",
				strerror(errno));
			return -1;
		}
		memset(&all_r2n[i], 0, step * sizeof(r2n));
		i = r2n_max;
		r2n_max += step;
	}
	all_r2n[i] = r2n;
	rc = prepare_obj_file(r2n, kallsyms);
	if (rc < 0)
		return -1;
	else if (rc > 0) {
		free(r2n);
		all_r2n[i] = NULL;
	}
	return 0;
}

#define INC_CNT(p_cnt)						\
	do {							\
		if (*(p_cnt) != 0x7fffffff) *(p_cnt) += 1;	\
	} while(0)

#define ADD_CNT(p_cnt, cnt)					\
	do {							\
		*(p_cnt) += (cnt);				\
		if (*(p_cnt) < (cnt))				\
			*(p_cnt) = 0x7fffffff;			\
	} while(0)

int inc_tracking_cnt_with_addr(struct path *p, bfd_vma addr)
{
	struct unknown *uk;

	for (uk = (struct unknown *)p->jmp_cnt; uk; uk = uk->next) {
		if (uk->addr == addr) {
			INC_CNT(&uk->cnt);
			return 0;
		}
	}
	uk = calloc(1, sizeof(struct unknown));
	if (!uk) {
		fprintf(stderr, "calloc failed.(%s)\n", strerror(errno));
		return -1;
	}
	uk->addr = addr;
	uk->cnt = 1;
	uk->next = (struct unknown*)p->jmp_cnt;
	p->jmp_cnt = (long)uk;
	return 0;
}

int proc_each_record(struct bt_record *rec, struct addr_range *range,
		     unsigned long *last)
{
	long i, idx_last, idx_from, chk_fallthrough;
	struct range_to_name *r2n, *r2n_to;
	struct path *p, *p_last, *p_from, *p_to = NULL;

	if (is_warn_record(rec)) {
		struct warn_record *warn = (struct warn_record*)rec;
		fprintf(stderr, "WARN: bts left only: %ld\n", warn->left);
		goto EXIT;
	}
	if (is_pid_record(rec) || !is_addr_match(rec, range))
		goto EXIT;
	r2n = addr_to_r2n(rec->from);
	if (!r2n)
		goto EXIT;
	idx_from = r2n->cnt;
	p_from = find_path_from_addr(r2n->pt, &idx_from, rec->from);
	if (!p_from)
		goto EXIT;
	idx_last = r2n->cnt;
	p_last = find_path_from_addr(r2n->pt, &idx_last, *last);

	/*
	printf("DBG:");
	printf_bfd_vma(*last);
	printf("->");
	printf_bfd_vma(rec->from);
	printf("->");
	printf_bfd_vma(rec->to);
	printf(" ");
	printf_path(p_from);
	if (p_to)
		printf_path(p_to);
		*/
	if (p_last) {
		/* check fallthrough */
		chk_fallthrough = 1;
		for (i = idx_last; i < idx_from; i++) {
			p = r2n->pt[i];
			if (p->type != BTYPE_BRANCH && p->type != BTYPE_OTHER) {
				chk_fallthrough = 0;
				/* The error occurs because the address between
				 * from turning off the bts facility to turning
				 * on seems to drop off.
				 * When the error occurs in bt_mod.ko, the
				 * fallthrough check is not done, and the
				 * following processing is done.
				 */
				if (strcmp(r2n->basename, MOD_NAME ".ko") != 0){
					fprintf(stderr,
						"WARN: detect lack of log");
					fprintf(stderr,
						" (0x%08lx) -> 0x%08lx -> 0x%08lx\n",
						*last, rec->from, rec->to);
					break;
					/*
					fprintf(stdout, "!!! NG path tree\n");
					dump_r2n();
					return -1;
					*/
				}
			}
		}
		if (chk_fallthrough)
			for (i = idx_last; i < idx_from; i++) {
				p = r2n->pt[i];
				if (i != idx_last)
					INC_CNT(&p->cnt);
				if (p->type == BTYPE_BRANCH)
					INC_CNT(&p->next_cnt);
			}
	}
	if (p_last != p_from)
		INC_CNT(&p_from->cnt);

	/* check branch */
	if (p_from->base == rec->from) {
		if (p_from->jmp == UNKNOWN_BADDR) {
			if (inc_tracking_cnt_with_addr(p_from, rec->to) < 0)
				return -1;
		} else if (p_from->jmp == rec->to)
			INC_CNT(&p_from->jmp_cnt);
	}
	/* check to */
	r2n_to = addr_to_r2n(rec->to);
	if (r2n_to) {
		i = r2n_to->cnt;
		p_to = find_path_from_addr(r2n_to->pt, &i, rec->to);
		if (p_to)
			INC_CNT(&p_to->cnt);
	}
EXIT:
	*last = rec->to;
	return 0;
}

void printf_funcname(struct bfd_if *bi, bfd_vma addr)
{
	int rc, lno;
	size_t offset;
	const char *src_name, *func_name;

	if (addr == UNKNOWN_BADDR) {
		printf("----------");
		return;
	}
	rc = addr_to_func_name_and_offset(bi, addr, &func_name, &offset);
	if (rc >= 0) {
		printf_func_name(func_name, offset);
		rc = get_source_info(bi, addr, &src_name, &func_name, &lno);
		if (rc >= 0 && src_name)
			printf(":%s,%d", src_name, lno);
	} else
		printf_bfd_vma(addr);
}

void printf_srcname_and_lno(struct bfd_if *bi, bfd_vma addr)
{
	int rc, lno;
	const char *src_name, *func_name;

	if (addr == UNKNOWN_BADDR) {
		printf("----------");
		return;
	}
	rc = get_source_info(bi, addr, &src_name, &func_name, &lno);
	if (rc >= 0 && src_name)
		printf("%s,%d", src_name, lno);
	else
		printf_bfd_vma(addr);
		//printf_funcname(bi, addr);
}

/*-----------------------------------------------------------------------------
 *  display function coverage
 *-----------------------------------------------------------------------------
 */
struct func_chk {
	bfd_vma		addr;
	unsigned long	cnt;
};

/*
struct fc_r2n_pack {
	node			*fc_list;
	struct range_to_name	*r2n;
};
*/

static int f_cmp_fc(void *__target, void *__dt)
{
	bfd_vma addr = *((bfd_vma*)__target);
	struct func_chk *fc = (struct func_chk*)__dt;

	if (addr < fc->addr)
		return -1;
	if (addr > fc->addr)
		return 1;
	return 0;
}

int __check_func(node **fc_list, bfd_vma addr, unsigned long cnt)
{
	struct func_chk *fc;

	fc = search_tree(&addr, *fc_list, f_cmp_fc);
	if (fc) {
		ADD_CNT(&fc->cnt, cnt);
		return 0;
	}
	fc = malloc(sizeof(struct func_chk));
	if (!fc) {
		fprintf(stderr, "malloc failed.(%s)\n", strerror(errno));
		return -1;
	}
	fc->addr = addr;
	fc->cnt = cnt;

	*fc_list = insert_tree(fc, *fc_list, f_cmp_fc);
	if (!(*fc_list))
		return -1;
	return 0;
}

int __check_func_by_jmp_addr(node **fc_list, bfd_vma addr, unsigned long cnt)
{
	struct func_chk *fc;

	if (!cnt)
		return 0;
	fc = search_tree(&addr, *fc_list, f_cmp_fc);
	if (fc)
		ADD_CNT(&fc->cnt, cnt);
	return 0;
}

int check_func(node **fc_list, struct path *p)
{
	struct unknown *uk;

	if (p->jmp == UNKNOWN_BADDR) {
		for (uk = (struct unknown*)p->jmp_cnt; uk; uk = uk->next) {
			if (__check_func(fc_list, uk->addr, uk->cnt) < 0)
				return -1;
		}
	} else if (__check_func(fc_list, p->jmp, p->jmp_cnt) < 0)
			return -1;
	return 0;
}

int chk_all_func_syms(node **fc_list, struct range_to_name *r2n)
{
	unsigned long i;
	struct bfd_if *bi = &r2n->bi;
	asymbol *sym;

	for (i = 0; i < bi->n_fsyms; i++) {
		sym = bi->p_fsyms[i];
		if (sym->flags & BSF_FUNCTION) {
			if (__check_func(fc_list,
					 bfd_asymbol_value(sym) + r2n->offset,
					 0) < 0)
				return -1;
		}
	}
	return 0;
}

static void f_fc_free(void *__dt)
{
	struct func_chk *fc = (struct func_chk*)__dt;

	free(fc);
}

int chk_func_coverage(node **fc_list)
{
	unsigned long i, j;
	struct range_to_name *r2n;
	struct path *p;

	*fc_list = NULL;
	for (i = 0; i < r2n_max; i++) {
		r2n = all_r2n[i];
		if (r2n) {
			if (chk_all_func_syms(fc_list, r2n) < 0)
				goto ERR_EXIT;
			for (j = 0; j < r2n->cnt; j++) {
				p = r2n->pt[j];
				if (p->type == BTYPE_CALL) {
					if (check_func(fc_list, p) < 0)
						goto ERR_EXIT;
				} else if (p->type == BTYPE_JMP) {
					if (__check_func_by_jmp_addr(fc_list,
								     p->jmp,
								     p->jmp_cnt)
					    < 0)
						goto ERR_EXIT;
				}
			}
		}
	}
	return 0;

ERR_EXIT:
	free_tree(*fc_list, f_fc_free);
	return -1;
}

struct call_cnt_pack {
	unsigned long		called;
	unsigned long		all;
	struct range_to_name	*r2n;
};

static int f_cnt_each_fc(void *__dt, void *user_data)
{
	struct call_cnt_pack *pack = (struct call_cnt_pack*)user_data;
	struct range_to_name *r2n = pack->r2n;
	struct func_chk *cc = (struct func_chk*)__dt;

	if (cc->addr >= r2n->begin && cc->addr <= r2n->end) {
		pack->all++;
		if (cc->cnt)
			pack->called++;
	} else if (cc->addr > r2n->end)
		return 1;	/* break */
	return 0;
}

static int f_dump_each_fc(void *__dt, void *user_data)
{
	struct range_to_name *r2n = (struct range_to_name*)user_data;
	struct func_chk *cc = (struct func_chk*)__dt;

	if (cc->addr >= r2n->begin && cc->addr <= r2n->end) {
		printf(cc->cnt ? "(OK) " : "(NT) ");
		printf_funcname(&r2n->bi, cc->addr - r2n->offset);
		if (cc->cnt)
			printf("\t(%ld)\n", cc->cnt);
		else
			printf("\n");
	} else if (cc->addr > r2n->end)
		return 1;	/* break */
	return 0;
}

void dump_func_coverage(struct range_to_name *r2n, node *cc_list)
{
	struct call_cnt_pack pack = { 0, 0, r2n };

	for_each_node(cc_list, f_cnt_each_fc, &pack);
	printf("------ function coverage (%ld/%ld) ------\n",
	       pack.called, pack.all);
	for_each_node(cc_list, f_dump_each_fc, r2n);
}

/*-----------------------------------------------------------------------------
 *  display branch coverage
 *-----------------------------------------------------------------------------
 */
#define IS_SWITCH_JMP(p)	\
	((p)->type == BTYPE_JMP \
	 && (p)->jmp == UNKNOWN_BADDR && (p)->next_cnt < 0)

static long get_unknown_bcnt(struct path *p)
{
	long i = 0;
	struct unknown *uk;

	for (uk = (struct unknown*)p->jmp_cnt; uk; uk = uk->next)
		i++;
	return i;
}

void dump_one_branch_coverage(struct range_to_name *r2n, struct path *p)
{
	struct unknown *uk;
	struct bfd_if *bi = &r2n->bi;

	if (IS_SWITCH_JMP(p)) {
		/* case of switch case ... */
		if (get_unknown_bcnt(p)) {
			for (uk = (struct unknown*)p->jmp_cnt; uk;
			     uk = uk->next) {
				printf("(UK) ");
				printf_srcname_and_lno(bi,
						       p->base - r2n->offset);
				printf(" [%ld/x] ", uk->cnt);
				printf_srcname_and_lno(bi,
						       uk->addr - r2n->offset);
				printf(":xxxxxxxxxx\n");
			}
		} else {
			printf("(NT) ");
			printf_srcname_and_lno(bi, p->base - r2n->offset);
			printf(" [0/x] ----------:xxxxxxxxxx\n");
		}
	} else if (p->jmp == UNKNOWN_BADDR) {
		/* case of indirect addressing branch */
		if (get_unknown_bcnt(p)) {
			for (uk = (struct unknown*)p->jmp_cnt; uk;
			     uk = uk->next) {
				printf("(UK) ");
				printf_srcname_and_lno(bi,
						       p->base - r2n->offset);
				printf(" [%ld/-] ", uk->cnt);
				printf_srcname_and_lno(bi,
						       uk->addr - r2n->offset);
				printf(":----------\n");
			}
		} else {
			printf("(NT) ");
			printf_srcname_and_lno(bi, p->base - r2n->offset);
			printf(" [0/-] ----------:----------\n");
		}
	} else {
		if (p->jmp_cnt != 0 && p->next_cnt != 0)
			printf("(OK) ");
		else if (p->jmp_cnt == 0 && p->next_cnt == 0)
			printf("(NT) ");
		else
			printf("(HT) ");
		printf_srcname_and_lno(bi, p->base - r2n->offset);
		printf(" [%ld/%ld] ", p->jmp_cnt, p->next_cnt);
		printf_srcname_and_lno(bi, p->jmp - r2n->offset);
		printf(":");
		printf_srcname_and_lno(bi, p->next - r2n->offset);
		printf("\n");
	}
}

struct branch_cnts {
	unsigned long ok;
	unsigned long nt;
	unsigned long ht;
	unsigned long uk_through;
	unsigned long uk_not_through;
};

void dump_branch_coverage(struct range_to_name *r2n)
{
	long i, uk_bcnt;
	struct branch_cnts bcs = {0,0,0,0,0};
	struct path *p;

	for (i = 0; i < r2n->cnt; i++) {
		p = r2n->pt[i];
		if (p->type != BTYPE_BRANCH && !IS_SWITCH_JMP(p))
			continue;
		if (p->jmp == UNKNOWN_BADDR) {
			uk_bcnt = get_unknown_bcnt(p);
			if (!uk_bcnt)
				bcs.uk_not_through++;
			else
				bcs.uk_through++;
		} else if (p->jmp_cnt != 0 && p->next_cnt != 0)
			bcs.ok++;
		else if (p->jmp_cnt == 0 && p->next_cnt == 0)
			bcs.nt++;
		else
			bcs.ht++;
	}
	printf("------ branch coverage (OK:%ld,UK:%ld,HT:%ld,NT:%ld / %ld)\n",
	       bcs.ok, bcs.uk_through, bcs.ht, bcs.nt,
	       (bcs.ok + bcs.ht + bcs.nt) * 2
	       			+ bcs.uk_through + bcs.uk_not_through);
	for (i = 0; i < r2n->cnt; i++) {
		p = r2n->pt[i];
		if (p->type != BTYPE_BRANCH && !IS_SWITCH_JMP(p))
			continue;
		dump_one_branch_coverage(r2n, p);
	}
}

struct state_cnts {
	unsigned long ok;
	unsigned long nt;
};

/*-----------------------------------------------------------------------------
 *  display state coverage
 *-----------------------------------------------------------------------------
 */
void dump_state_coverage(struct range_to_name *r2n)
{
	long i;
	struct state_cnts scs = {0,0};
	struct path *p;

	for (i = 0; i < r2n->cnt; i++) {
		p = r2n->pt[i];
		if (p->cnt > 0)
			scs.ok += 1;
		else
			scs.nt += 1;
	}
	printf("------ state coverage (%ld/%ld) ------\n",
	       scs.ok, scs.ok + scs.nt);
	for (i = 0; i < r2n->cnt; i++) {
		p = r2n->pt[i];
		if (p->cnt > 0)
			printf("(OK) ");
		else
			printf("(NT) ");
		printf_srcname_and_lno(&r2n->bi, p->addr);
		printf("\n");
	}
}

/*-----------------------------------------------------------------------------
 *  display coverage
 *-----------------------------------------------------------------------------
 */
void dump_coverage(void)
{
	int i;
	struct range_to_name *r2n;
	node *cc_list;

	if (!all_r2n)
		return;

	if (chk_func_coverage(&cc_list) < 0)
		return;

	for (i = 0; i < r2n_max; i++) {
		r2n = all_r2n[i];
		if (r2n) {
			printf("====== %s coverage ======\n", r2n->basename);
			dump_func_coverage(r2n, cc_list);
			dump_branch_coverage(r2n);
			dump_state_coverage(r2n);
		}
	}
}

int proc_logfile(char *logfile)
{
	int fd, rc = -1;
	size_t size;
	struct bt_record *top, *rec, *p_max;
	char *dir, *fname;
	struct addr_range *range;
	unsigned long last_addr;

	if (open_mmapfile(logfile, &fd, (void**)&top, &size) < 0)
		goto EXIT;

	rec = top;
	fname = basename(logfile);
	dir = dirname(logfile);
	if (is_exists_maps(dir, fname)) {
		if (parse_maps(add_range_to_name, dir, fname) < 0)
			goto EXIT;
	}
	if (parse_modules(add_range_to_name, dir) < 0)
		goto EXIT;
	printf("start\n");
	range = get_pid_addr_range(ALL_PID);
	last_addr = UNKNOWN_BADDR;
	for (p_max = (struct bt_record*)((char*)rec + size);
	     rec < p_max; rec++) {
		/*
		if (!((p_max - rec) % 1000))
			printf("%d records left...\n", p_max - rec);
			*/
		if (proc_each_record(rec, range, &last_addr) < 0) {
			printf("!!! Error: (0x%08lx) -> 0x%08lx -> 0x%08lx\n",
			       last_addr, rec->from, rec->to);
			goto EXIT;
		}
	}
	rc = 0;
EXIT:
	close_mmapfile(fd, top, size);
	return rc;
}

void err_exit(void)
{
	exit(1);
}

void usage(void)
{
	fprintf(stderr, "bt_coverage %s\n", BT_COVERAGE_VER);
	fprintf(stderr, "    %s\n\n", COPYRIGHT);
	fprintf(stderr, "bt_coverage [-a top:end [...]] [-d top:end [...]] [--usr|--ker|--all]\n");
	fprintf(stderr, "            -f logfile [[-s src_dir] -o html_out_dir]\n");
	fprintf(stderr, "  -a: add address range\n");
	fprintf(stderr, "  -d: delete address range\n");
	fprintf(stderr, "  --usr: alias of '-a 0:0xbfffffff'\n");
	fprintf(stderr, "  --ker: alias of '-a 0xc0000000:0xffffffff'\n");
	fprintf(stderr, "  --all: alias of '-a 0:0xffffffff'\n");
	fprintf(stderr, "  -f: logfile\n");
	fprintf(stderr, "  -s: source directory\n");
	fprintf(stderr, "  -o: html output directory\n");
}

int proc_conv2html(char *outdir, char *srcdir, char *argv[]);

int main(int argc, char *argv[])
{
	int i, html_opt_idx = -1, verbose = 0;
	char *outdir = NULL, *srcdir = NULL, *logfile = NULL;
	unsigned long begin, end;

	if (alloc_pid_range(ALL_PID) < 0)
		err_exit();
	for (i = 1; i < argc;) {
		if (strcmp(argv[i], "--usr") == 0) {
			add_range(ALL_PID, 0, 0xbfffffff);
			i++;
		} else if (strcmp(argv[i], "--ker") == 0) {
			add_range(ALL_PID, 0xc0000000, 0xffffffff);
			i++;
		} else if (strcmp(argv[i], "--all") == 0) {
			add_range(ALL_PID, 0, 0xffffffff);
			i++;
		} else if (strcmp(argv[i], "-a") == 0) {
			chk_next(argc, i);
			chk_html_opt_idx(html_opt_idx);
			i++;
			while (i < argc && argv[i][0] != '-') {
				if (!range2ulongs(argv[i], &begin, &end))
					err_exit();
				add_range(ALL_PID, begin, end);
				i++;
			}
		} else if (strcmp(argv[i], "-d") == 0) {
			chk_next(argc, i);
			chk_html_opt_idx(html_opt_idx);
			i++;
			while (i < argc && argv[i][0] != '-') {
				if (!range2ulongs(argv[i], &begin, &end))
					err_exit();
				printf("del_range\n");
				del_range(ALL_PID, begin, end);
				i++;
			}
		} else if (strcmp(argv[i], "-f") == 0) {
			chk_next(argc, i);
			chk_html_opt_idx(html_opt_idx);
			i++;
			logfile = argv[i];
			i++;
		} else if (strcmp(argv[i], "-o") == 0) {
			chk_next(argc, i);
			if (html_opt_idx < 0)
				html_opt_idx = i;
			i++;
			outdir = argv[i];
			i++;
		} else if (strcmp(argv[i], "-s") == 0) {
			chk_next(argc, i);
			if (html_opt_idx < 0)
				html_opt_idx = i;
			i++;
			srcdir = argv[i];
			i++;
		} else if (strcmp(argv[i], "-v") == 0) {
			verbose = 1;
			i++;
		} else {
			usage();
			err_exit();
		}
	}
	if (!logfile) {
		usage();
		err_exit();
	}
	if (outdir) {
		argv[html_opt_idx] = NULL;
		if (proc_conv2html(outdir, srcdir, argv) < 0)
			err_exit();
	} else {
		all_r2n = calloc(r2n_max, sizeof(struct range_to_name*));
		if (!all_r2n) {
			fprintf(stderr, "calloc failed.(%s)\n",
				strerror(errno));
			err_exit();
		}
		if (proc_logfile(logfile) < 0)
			err_exit();
		if (verbose)	/* hidden option for debug */
			dump_r2n();
		dump_coverage();
	}
	exit(0);
}
