/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bt_hconv.c - coverage output to html converter                           */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2006                         */
/*             Authors: Yumiko Sugita (sugita@sdl.hitachi.co.jp),            */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include "bfd_if.h"
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <signal.h>
#include <libgen.h>

#define MAX_PATH_LEN	256
#define MAX_LINE_LEN	256

#define	chk_next(argc, i)	if ((argc) <= (i)+1) { usage(); err_exit(); }

struct app_dt {
	FILE	*in;
	FILE	*summary_out;
	FILE	*cur_out;
	char	*srcdir;
	char	*outdir;
};

/*
 * When the object contains debugging information, we need to create the source
 * code html.
 * The source line number of the result of the coverage check has the
 * possibility of appearing at random when it compiled with optimization.
 * Moreover, there is a possibility that two or more branches are included in
 * the same source line, too.
 * Therefore, we first create the each source line type structure, then
 * create the source html by merging it with source file.
 */

#define T_NONE		0
#define T_NOT_EXECUTED	1
#define T_EXECUTED	2

struct src_info {
	char	path[MAX_PATH_LEN + 1];
	long	ln_max;
	char	*type;	/* each source line type */

	/* using for source html output */
	int	is_ref;
	char	html_out_path[MAX_PATH_LEN + 1];
};

static struct src_info **src_info;
static long src_info_num;

/*-----------------------------------------------------------------------------
 *  output html
 *-----------------------------------------------------------------------------
 */
#define COLOR_OK	"#00ff00"
#define COLOR_HT	"#ffff00"
#define COLOR_NT	"#ff0000"

static void out_html_header(FILE *f, char *doctype, char *title)
{
	fprintf(f,
		"%s\n" \
		"<html lang=\"en\">\n" \
		"<head>\n" \
		"<meta http-equiv=\"Content-Type\" content=\"text/html; charset=ISO-8859-1\">\n" \
		"<meta http-equiv=\"Content-Style-Type\" content=\"text/css\">\n" \
		"<title>%s</title>\n" \
		"</head>\n",
		doctype, title);
}

static void out_html_frame_header(FILE *f, char *title)
{
	out_html_header(f,
			"<!DOCTYPE html PUBLIC \"-//W3C//DTD HTML 4.01 Frameset//EN\">",
			title);
}

static void out_html_normal_header(FILE *f, char *title)
{
	out_html_header(f,
			"<!DOCTYPE html PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">",
			title);
}

static int out_top_html(struct app_dt *dt)
{
	char buf[MAX_PATH_LEN + 1];
	FILE *f;

	buf[MAX_PATH_LEN] = '\0';
	snprintf(buf, MAX_PATH_LEN, "%s/top.html", dt->outdir);
	f = fopen(buf, "w+");
	if (!f) {
		fprintf(stderr, "%s can't open.(%s)\n", buf, strerror(errno));
		return -1;
	}
	out_html_frame_header(f, "BTRAX - COVERAGE RESULT");
	fprintf(f,
		"<frameset title=\"all\" rows=\"40,*\">\n" \
		"  <frame title=\"title\" src=\"./title.html\" scrolling=\"no\" noresize>\n" \
		"  <frameset title=\"right\" cols=\"50%%,*\">\n" \
		"    <frameset title=\"left\" rows=\"40%%,*\">\n" \
		"      <frame title=\"summary\" name=\"summary\" src=\"./summary.html\">\n" \
		"      <frame title=\"each\" name=\"each\">\n" \
		"    </frameset>\n" \
		"    <frame title=\"src\" name=\"src\">\n" \
		"  </frameset>\n" \
		"  <noframes><body><p>use frame supported browser</p></body></noframes>\n" \
		"</frameset>\n" \
		"</html>\n");
	fclose(f);
	return 0;
}

static int out_title_html(struct app_dt *dt)
{
	char buf[MAX_PATH_LEN + 1];
	FILE *f;

	buf[MAX_PATH_LEN] = '\0';
	snprintf(buf, MAX_PATH_LEN, "%s/title.html", dt->outdir);
	f = fopen(buf, "w+");
	if (!f) {
		fprintf(stderr, "%s can't open.(%s)\n", buf, strerror(errno));
		return -1;
	}
	out_html_normal_header(f, "title");
	fprintf(f,
		"<body style=\"background-color:silver\">\n" \
		"<p style=\"font-size:16pt; font-weight:bold; text-align:center\">\n" \
		"<span style=\"color:#0000ff\">BTRAX</span> Coverage\n" \
		"<span style=\"color:#ff0000\">R</span>e" \
		"<span style=\"color:#ffff00\">s</span>u" \
		"<span style=\"color:#00ff00\">l</span>t\n" \
		"</p>\n" \
		"</body></html>\n");
	fclose(f);
	return 0;
}

static void out_summary_html_start(struct app_dt *dt)
{
	out_html_normal_header(dt->summary_out, "summary");
	fprintf(dt->summary_out,
		"<body>\n" \
		"<table summary=\"summary\" border=\"1\">\n" \
		"  <tr><th rowspan=\"2\">name</th><th colspan=\"3\">coverage</th></tr>\n" \
		"  <tr><th>function</th><th>branch</th><th>state</th></tr>\n");
}

static void out_summary_html_name(struct app_dt *dt, char *name)
{
	fprintf(dt->summary_out, "  <tr><td>%s</td>\n", name);
}

static void out_summary_html_func(struct app_dt *dt, char *path,
				  long n_func, long n_func_all)
{
	fprintf(dt->summary_out,
		"    <td align=\"right\"><a href=\"%s\" target=\"each\" title=\"%s\">%.2f%%</a></td>\n",
		path, path, (double)n_func * 100 / n_func_all);
}

static void out_summary_html_branch(struct app_dt *dt, char *path,
				    long n_br_ok, long n_br_uk,
				    long n_br_ht, long n_br_nt, long n_br_all)
{
	fprintf(dt->summary_out,
		"    <td align=\"right\"><a href=\"%s\" target=\"each\" title=\"%s\">%.2f%%</a></td>\n",
		path, path,
		(double)(n_br_ok * 2 + n_br_uk + n_br_ht) * 100 / n_br_all);
}

static void out_summary_html_state(struct app_dt *dt, char *path,
				   long n_ok, long n_states)
{
	fprintf(dt->summary_out,
		"    <td align=\"right\"><a href=\"%s\" target=\"each\" title=\"%s\">%.2f%%</a></td></tr>\n",
		path, path, (double)n_ok * 100 / n_states);
}

static void out_summary_html_end(struct app_dt *dt)
{
	fprintf(dt->summary_out,
		"</table>\n" \
		"</body></html>\n");
}

static struct src_info* chk_src_line_type(struct app_dt*, char*, long, char);
static char* get_src_html_path(struct app_dt*, struct src_info*, int);

static void out_func_html_start(struct app_dt *dt, char *name)
{
	char title[MAX_LINE_LEN + 1];

	title[MAX_LINE_LEN] = '\0';
	sprintf(title, "%s function coverage", name);
	out_html_normal_header(dt->cur_out, title);
	fprintf(dt->cur_out,
		"<body>\n" \
		"<table summary=\"%s\" border=\"1\"><tr><th>function</th><th>count</th></tr>\n",
		title);
}

static void out_func_html_each(struct app_dt *dt, char *func, long cnt,
			       char *src, long ln)
{
	struct src_info *info;
	char *color, *ref;
	int type;

	if (cnt) {
		type = T_EXECUTED;
		color = COLOR_OK;
	} else {
		type = T_NOT_EXECUTED;
		color = COLOR_NT;
	}
	if (src[0] != '\0' && (info = chk_src_line_type(dt, src, ln, type))) {
		ref = get_src_html_path(dt, info, 1);
		fprintf(dt->cur_out,
			"<tr><td bgcolor=\"%s\"><a href=\"%s#%ld\" target=\"src\">%s</a></td><td align=\"right\">%ld</td></tr>\n",
			color, ref, ln, func, cnt);
	} else {
		fprintf(dt->cur_out,
			"<tr><td bgcolor=\"%s\">%s</td><td align=\"right\">%ld</td></tr>\n",
			color, func, cnt);
	}
}

static void out_func_html_end(struct app_dt *dt)
{
	fprintf(dt->cur_out,
		"</table>\n" \
		"</body></html>\n");
}

static void out_branch_html_start(struct app_dt *dt, char *name)
{
	char title[MAX_LINE_LEN + 1];

	title[MAX_LINE_LEN] = '\0';
	sprintf(title, "%s branch coverage", name);
	out_html_normal_header(dt->cur_out, title);
	fprintf(dt->cur_out,
		"<body>\n" \
		"<table summary=\"%s\" border=\"1\"><tr>\n" \
		"  <th rowspan=\"2\">base</th>\n" \
		"  <th colspan=\"2\">branch</th>\n" \
		"  <th colspan=\"2\">next</th></tr>\n" \
		"  <tr>\n" \
		"  <th>address</th><th>cnt</th>\n" \
		"  <th>address</th><th>cnt</th>\n" \
		"  </tr>\n", title);
}

static int dir_chk_and_create(char*);

static struct src_info* chk_branch_point(struct app_dt *dt, char *src_dt,
					 long cnt, char *fname, long *ln)
{
	char type;
	int num;

	num = sscanf(src_dt, "%[^,],%ld", fname, ln);
	if (num == 2) {
		type = cnt ? T_EXECUTED : T_NOT_EXECUTED;
		return chk_src_line_type(dt, fname, *ln, type);
	}
	return NULL;
}

static void out_branch_html_each(struct app_dt *dt, char *base, char *branch,
				 char *fall, long b_cnt, long f_cnt)
{
	char fname[MAX_LINE_LEN + 1], *color, *ref = "";
	long ln;
	struct src_info *info;

	color = COLOR_NT;
	if (b_cnt && f_cnt > 0)
		color = COLOR_OK;
	else if (b_cnt || f_cnt > 0)
		color = COLOR_HT;
	info = chk_branch_point(dt, base, b_cnt + (f_cnt < 0 ? 0 : f_cnt),
				fname, &ln);
	base = basename(base);
	if (info) {
		ref = get_src_html_path(dt, info, 1);
		fprintf(dt->cur_out,
			"<tr><td bgcolor=\"%s\"><a href=\"%s#%ld\" target=\"src\">%s</a></td>\n",
			color, ref, ln, base);
	} else {
		fprintf(dt->cur_out,
			"<tr><td bgcolor=\"%s\">%s</td>\n",
			color, base);
	}
	chk_branch_point(dt, branch, b_cnt, fname, &ln);
	chk_branch_point(dt, fall, f_cnt, fname, &ln);
	branch = basename(branch);
	if (fall[0] != '\0')
		fall = basename(fall);
	fprintf(dt->cur_out,
		"    <td>%s</td><td align=\"right\">%ld</td>\n",
		branch, b_cnt);
	if (f_cnt < 0)
		fprintf(dt->cur_out,
			"    <td></td><td align=\"right\"></td></tr>\n");
	else
		fprintf(dt->cur_out,
			"    <td>%s</td><td align=\"right\">%ld</td></tr>\n",
			fall, f_cnt);
}

static void out_branch_html_end(struct app_dt *dt)
{
	fprintf(dt->cur_out,
		"</table>\n" \
		"</body></html>\n");
}

static void out_state_html_start(struct app_dt *dt, char *name)
{
	char title[MAX_LINE_LEN + 1];

	title[MAX_LINE_LEN] = '\0';
	sprintf(title, "%s state coverage", name);
	out_html_normal_header(dt->cur_out, title);
	fprintf(dt->cur_out,
		"<body>\n" \
		"<table summary=\"%s\" border=\"1\"><tr><th>state</th><th>exec</th></tr>\n",
		title);
}

static void out_state_html_each(struct app_dt *dt, char *state, long is_exec,
				long ln)
{
	struct src_info *info;
	char *color, *mark, *ref;
	int type;

	if (is_exec) {
		type = T_EXECUTED;
		color = COLOR_OK;
		mark = "Y";
	} else {
		type = T_NOT_EXECUTED;
		color = COLOR_NT;
		mark = "N";
	}
	if (ln > 0 && (info = chk_src_line_type(dt, state, ln, type))) {
		ref = get_src_html_path(dt, info, 1);
		state = basename(state);
		fprintf(dt->cur_out,
			"<tr><td bgcolor=\"%s\"><a href=\"%s#%ld\" target=\"src\">%s,%ld</a></td><td align=\"center\">%s</td></tr>\n",
			color, ref, ln, state, ln, mark);
	} else {
		fprintf(dt->cur_out,
			"<tr><td bgcolor=\"%s\">%s</td><td align=\"center\">%s</td></tr>\n",
			color, state, mark);
	}
}

static char* get_src_html_path(struct app_dt *dt, struct src_info *info,
			       int is_ref)
{
	if (info->html_out_path[0] != '\0' && info->is_ref == is_ref)
		return info->html_out_path;
	if (is_ref)
		sprintf(info->html_out_path, "../src/%s.html",
			info->path[0] == '/' ? info->path + 1 : info->path);
	else
		sprintf(info->html_out_path, "%s/src/%s.html", dt->outdir,
			info->path[0] == '/' ? info->path + 1 : info->path);
	info->is_ref = is_ref;
	return info->html_out_path;
}

static struct src_info* chk_src_line_type(struct app_dt *dt, char *fname,
					  long ln, char type)
{
	FILE *f = NULL;
	char path[MAX_PATH_LEN + 1];
	struct src_info *info = NULL;
	long i;
	char buf[MAX_LINE_LEN + 1];

	if (!ln)
		return NULL;
	if (dt->srcdir) {
		path[MAX_PATH_LEN] = '\0';
		sprintf(path, "%s/%s", dt->srcdir, fname);
	}

	/* search into src_info structures */
	for (i = src_info_num - 1; i >= 0; i--) {
		info = src_info[i];
		if (strcmp(info->path, fname) == 0)
			goto CHK_LINE_TYPE;
		if (dt->srcdir && strcmp(info->path, path) == 0)
			goto CHK_LINE_TYPE;
	}
	info = calloc(1, sizeof(struct src_info));
	if (!info) {
		fprintf(stderr, "memory allocation failure.(%s)\n",
			strerror(errno));
		return NULL;
	}

	/* check source file existance and max line number */
	f = fopen(fname, "r");
	if (!f) {
		if (!dt->srcdir)
			goto ERR_EXIT;
		f = fopen(path, "r");
		if (!f)
			goto ERR_EXIT;
		strcpy(info->path, path);
	}
	if (info->path[0] == '\0')
		strcpy(info->path, fname);
	for (i = 0; fgets(buf, MAX_LINE_LEN, f); i++);
	fclose(f);
	info->ln_max = i;
	info->type = calloc(i, sizeof(char));
	if (!info->type) {
		fprintf(stderr, "memory allocation failure.(%s)\n",
			strerror(errno));
		goto ERR_EXIT;
	}
	if (src_info_num) {
		src_info = realloc(src_info, (src_info_num + 1)
				   		* sizeof(struct src_info*));
	} else {
		src_info = malloc(sizeof(struct src_info*));
	}
	if (!src_info) {
		fprintf(stderr, "memory allocation failure.(%s)\n",
			strerror(errno));
		goto ERR_EXIT;
	}
	src_info[src_info_num++] = info;
CHK_LINE_TYPE:
	if (info->type[ln - 1] == T_NONE ||
	    (info->type[ln - 1] != type && type == T_EXECUTED))
		info->type[ln - 1] = type;
	return info;

ERR_EXIT:
	if (info) {
		if (info->type)
			free(info->type);
		free(info);
	}
	if (f)
		fclose(f);
	return NULL;
}

static char escape_need_chars[] = "\t<>&\"";
static char *escape_htmls[] = {
	"        ",
	"&lt;",
	"&gt;",
	"&amp;",
	"&quot;",
};

static void escape_for_html(char *buf)
{
	int len_max = strlen(buf), len, n;
	char *from, *to, *p, tmp[MAX_LINE_LEN + 1];

	tmp[MAX_LINE_LEN] = '\0';
	len = 0;
	for (from = buf, to = tmp; from - buf < len_max;) {
		p = strchr(escape_need_chars, *from);
		if (p) {
			from++;
			p = escape_htmls[p - escape_need_chars];
			n = strlen(p);
			if (len + n > MAX_LINE_LEN) {
				n = MAX_LINE_LEN - (len + n);
				snprintf(to, n, "%s", p);
			} else
				sprintf(to, "%s", p);
			to += n;
			len += n;
		} else {
			*to++ = *from++;
			len++;
		}
	}
	*to = '\0';
	strcpy(buf, tmp);
}

static int get_ln_cols(struct src_info *info)
{
	int cols = 0;
	long left;

	for (left = info->ln_max + 1; left; left = left / 10)
		cols++;
	return cols;
}

static char* type2color(char type)
{
	switch (type) {
	case T_NOT_EXECUTED:
		return COLOR_NT;
	case T_EXECUTED:
		return COLOR_OK;
	default:
		return "";
	}
}

static void out_src_html(struct app_dt *dt)
{
	long i, ln;
	struct src_info *info;
	int ln_cols, type;
	FILE *r, *w;
	char *dir, *path, buf[MAX_LINE_LEN + 1];

	buf[MAX_LINE_LEN] = '\0';
	for (i = 0; i < src_info_num; i++) {
		info = src_info[i];
		r = fopen(info->path, "r");
		if (!r) {
			fprintf(stdout, "%s can't open.(%s)\n",
				info->path, strerror(errno));
			break;
		}
		path = get_src_html_path(dt, info, 0);
		dir = dirname(strdup(path));
		if (dir_chk_and_create(dir) < 0) {
			fclose(r);
			break;
		}
		w = fopen(path, "w");
		if (!w) {
			fprintf(stdout, "%s can't open.(%s)\n",
				path, strerror(errno));
			break;
		}
		ln_cols = get_ln_cols(info);
		out_html_normal_header(w, "src");
		fprintf(w, "<body style=\"font-size:8pt\"><pre>\n");
		type = T_NONE;
		for (ln = 0; fgets(buf, MAX_LINE_LEN, r); ln++) {
			//printf("%*ld: %d\n", ln_cols, ln + 1, info->type[ln]);
			fprintf(w, "%*ld: ", ln_cols, ln + 1);
			if (info->type[ln] != T_NONE)
				type = info->type[ln];
			buf[strlen(buf) - 1] = '\0';
			escape_for_html(buf);
			fprintf(w, "<a name=\"%ld\" style=\"background:%s\">%s</a>\n",
				ln + 1, type2color(type), buf);
		}
		fprintf(w, "</pre></body></html>\n");
		fclose(r);
		fclose(w);
	}
}

/*-----------------------------------------------------------------------------
 *  parse input
 *-----------------------------------------------------------------------------
 */
static int parse_obj_start(char *buf, char *name)
{
	int num;

	num = sscanf(buf, "====== %s coverage ======\n", name);
	if (num != 1) {
		fprintf(stderr, "name parse error.(%s)\n", buf);
		return -1;
	}
	return 0;
}

static int parse_func_summary(char *buf, long *n_func, long *n_func_all)
{
	int num;

	num = sscanf(buf, "------ function coverage (%ld/%ld) ------\n",
		     n_func, n_func_all);
	if (num != 2) {
		fprintf(stderr, "function coverage parse error.(%s)\n", buf);
		return -1;
	}
	return 0;
}

static int parse_one_func(char *buf, char *func, long *cnt, char *src, long *ln)
{
	int num;

	func[MAX_LINE_LEN] = '\0';
	src[MAX_LINE_LEN] = '\0';
	num = sscanf(buf, "(OK) <%[^>]>:%[^,],%ld\t(%ld)\n", func, src, ln,cnt);
	if (num == 4)
		return 0;
	num = sscanf(buf, "(NT) <%[^>]>:%[^,],%ld\n", func, src, ln);
	if (num == 3) {
		*cnt = 0;
		return 0;
	}
	src[0] = '\0';
	*ln = 0;
	num = sscanf(buf, "(OK) <%[^>]>\t(%ld)\n", func, cnt);
	if (num == 2) {
		return 0;
	}
	num = sscanf(buf, "(NT) <%[^>]>\n", func);
	if (num == 1) {
		*cnt = 0;
		return 0;
	}
	fprintf(stderr, "function coverage parse error.(%s)\n", buf);
	return -1;
}

static int parse_branch_summary(char *buf, long *n_br_ok, long *n_br_uk,
				long *n_br_ht, long *n_br_nt, long *n_br_all)
{
	int num;

	num = sscanf(buf, "------ branch coverage (OK:%ld,UK:%ld,HT:%ld,NT:%ld / %ld)\n",
		     n_br_ok, n_br_uk, n_br_ht, n_br_nt, n_br_all);
	if (num != 5) {
		fprintf(stderr, "branch coverage parse error.(%s)\n", buf);
		return -1;
	}
	return 0;
}

static int parse_one_branch(char *buf, char *base,
			    char *branch, char *fall, long *b_cnt, long *f_cnt)
{
	int num;

	base[MAX_LINE_LEN] = '\0';
	branch[MAX_LINE_LEN] = '\0';
	fall[MAX_LINE_LEN] = '\0';
	num = sscanf(buf, "%*s %s [%ld/%ld] %[^:]:%[^:\n]\n",
		     base, b_cnt, f_cnt, branch, fall);
	if (num == 5)
		return 0;
	num = sscanf(buf, "%*s %s [%ld/x] %[^:]:xxxxxxxxxx\n",
		     base, b_cnt, branch);
	if (num == 3) {
		fall[0] = '\0';
		*f_cnt = -1;
		return 0;
	}
	return -1;
}

static int parse_state_summary(char *buf, long *n_ok, long *n_states)
{
	int num;

	num = sscanf(buf, "------ state coverage (%ld/%ld) ------\n",
		     n_ok, n_states);
	if (num != 2) {
		fprintf(stderr, "state coverage parse error.(%s)\n", buf);
		return -1;
	}
	return 0;
}

static int parse_one_state(char *buf, char *state, long *is_exec, long *ln)
{
	int num;
	char tmp[3];

	state[MAX_LINE_LEN] = '\0';
	num = sscanf(buf, "(%[^)]) %[^,],%ld\n", tmp, state, ln);
	if (num == 3) {
		*is_exec = strcmp(tmp, "OK") == 0 ? 1 : 0;
		return 0;
	}
	*ln = 0;
	num = sscanf(buf, "(%[^)]) %s\n", tmp, state);
	if (num == 2) {
		*is_exec = strcmp(tmp, "OK") == 0 ? 1 : 0;
		return 0;
	}
	state[0] = '\0';
	fprintf(stderr, "state coverage parse error.(%s)\n", buf);
	return -1;
}

#define PARSE_FUNC	0
#define PARSE_BRANCH	1
#define PARSE_STATE	2
#define PARSE_MAX	3

static int parse_input(struct app_dt *dt)
{
	char buf[MAX_LINE_LEN + 1], name[MAX_LINE_LEN + 1];
	char t1[MAX_LINE_LEN + 1], t2[MAX_LINE_LEN + 1], t3[MAX_LINE_LEN + 1];
	long c1, c2, c3, c4, c5;
	char path[MAX_PATH_LEN + 1], ref[MAX_PATH_LEN + 1];
	int started = 0, parse_type = PARSE_STATE;

	buf[MAX_LINE_LEN] = '\0';
	path[MAX_PATH_LEN] = ref[MAX_PATH_LEN] = '\0';
	out_summary_html_start(dt);
	while (fgets(buf, MAX_LINE_LEN, dt->in)) {
		if (!started) {
			if (strstr(buf, "start") == buf)
				started = 1;
			continue;
		}
		switch (buf[0]) {
		case '=':
			if (dt->cur_out) {
				out_branch_html_end(dt);
				fclose(dt->cur_out);
			}
			if (parse_obj_start(buf, name) < 0)
				return -1;
			out_summary_html_name(dt, name);
			break;
		case '-':
			parse_type += 1;
			if (parse_type == PARSE_MAX)
				parse_type = PARSE_FUNC;
			switch (parse_type) {
			case PARSE_FUNC:
				if (parse_func_summary(buf, &c1, &c2) < 0)
					return -1;
				snprintf(ref, MAX_PATH_LEN,"each/%s.f.html",
					 name);
				out_summary_html_func(dt, ref, c1, c2);
				break;
			case PARSE_BRANCH:
				out_func_html_end(dt);
				fclose(dt->cur_out);
				if (parse_branch_summary(buf, &c1, &c2,
							 &c3, &c4, &c5) < 0)
					return -1;
				snprintf(ref, MAX_PATH_LEN,"each/%s.b.html",
					 name);
				out_summary_html_branch(dt, ref,
							c1, c2, c3, c4, c5);
				break;
			case PARSE_STATE:
				out_branch_html_end(dt);
				fclose(dt->cur_out);
				if (parse_state_summary(buf, &c1, &c2) < 0)
					return -1;
				snprintf(ref, MAX_PATH_LEN,"each/%s.s.html",
					 name);
				out_summary_html_state(dt, ref, c1, c2);
				break;
			}
			snprintf(path, MAX_PATH_LEN, "%s/", dt->outdir);
			strncat(path, ref, MAX_PATH_LEN - strlen(ref));
			dt->cur_out = fopen(path, "w+");
			if (!dt->cur_out) {
				fprintf(stderr, "%s can't open.(%s)\n",
					path, strerror(errno));
				return -1;
			}
			switch (parse_type) {
			case PARSE_FUNC:
				out_func_html_start(dt, name);
				break;
			case PARSE_BRANCH:
				out_branch_html_start(dt, name);
				break;
			case PARSE_STATE:
				out_state_html_start(dt, name);
				break;
			}
			break;
		case '(':
			switch (parse_type) {
			case PARSE_FUNC:
				if (parse_one_func(buf, t1, &c1, t2, &c2) < 0)
					return -1;
				out_func_html_each(dt, t1, c1, t2, c2);
				break;
			case PARSE_BRANCH:
				if (parse_one_branch(buf, t1, t2, t3, &c1, &c2)
				    < 0)
					return -1;
				out_branch_html_each(dt, t1, t2, t3, c1, c2);
				break;
			case PARSE_STATE:
				if (parse_one_state(buf, t1, &c1, &c2) < 0)
					return -1;
				out_state_html_each(dt, t1, c1, c2);
				break;
			}
			break;
		}
	}
	if (dt->cur_out) {
		out_branch_html_end(dt);
		fclose(dt->cur_out);
	}
	out_src_html(dt);
	out_summary_html_end(dt);
	return 0;
}

/*-----------------------------------------------------------------------------
 *  main
 *-----------------------------------------------------------------------------
 */
static int dir_chk_and_create(char *path)
{
	struct stat st;
	char buf[MAX_LINE_LEN + 1];

	if (stat(path, &st) < 0) {
		if (errno == ENOENT) {
			buf[MAX_LINE_LEN] = '\0';
			snprintf(buf, MAX_LINE_LEN, "mkdir -p \"%s\"", path);
			if (system(buf) < 0) {
				fprintf(stderr, "%s can't create.\n", path);
				return -1;
			}
			return 0;
		}
		fprintf(stderr, "%s can't get stat.(%s)\n",
			path, strerror(errno));
		return -1;
	}
	if (!S_ISDIR(st.st_mode)) {
		fprintf(stderr, "%s is not directory.\n", path);
		return -1;
	}
	return 0;
}

pid_t c_pid;

void sig_handler(int signo)
{
	switch (signo) {
	case SIGCHLD:
		signal(SIGCHLD, sig_handler);
		break;
	default:
		if (c_pid > 0) {
			kill(c_pid, signo);
			c_pid = 0;
		}
		break;
	}
}

int proc_conv2html(char *outdir, char *srcdir, char *argv[])
{
	char path[MAX_PATH_LEN + 1];
	struct app_dt dt;
	int fds[2];
	int st;

	memset(&dt, 0, sizeof dt);
	dt.outdir = outdir;
	dt.srcdir = srcdir;
	if (dir_chk_and_create(outdir) < 0)
		return -1;
	path[MAX_PATH_LEN] = '\0';
	snprintf(path, MAX_PATH_LEN, "%s/each", outdir);
	if (dir_chk_and_create(path) < 0)
		return -1;

	signal(SIGTERM, sig_handler);
	signal(SIGINT, sig_handler);

	if (pipe(fds) < 0) {
		fprintf(stderr, "pipe failed.(%s)\n", strerror(errno));
		return -1;
	}
	switch ((c_pid = fork())) {
	case -1:
		fprintf(stderr, "fork failed.(%s)\n", strerror(errno));
		return -1;
	case 0:
		close(fds[0]);
		close(1);
		dup(fds[1]);
		close(fds[1]);
		execvp(argv[0], argv);
		break;
	default:
		signal(SIGCHLD, sig_handler);
		close(fds[1]);
		if (!(dt.in = fdopen(fds[0], "r"))) {
			fprintf(stderr, "fdopen failed.(%s)\n",
				strerror(errno));
			goto C_EXIT;
		}
		path[MAX_PATH_LEN] = '\0';
		snprintf(path, MAX_PATH_LEN, "%s/summary.html", outdir);
		dt.summary_out = fopen(path, "w+");
		if (!dt.summary_out) {
			fprintf(stderr, "%s can't open.(%s)\n", path,
				strerror(errno));
			goto C_EXIT;
		}
		if (out_top_html(&dt) < 0)
			goto C_EXIT;
		if (out_title_html(&dt) < 0)
			goto C_EXIT;
		if (parse_input(&dt) < 0)
			goto C_EXIT;
		wait(&st);
		if (WIFEXITED(st) && !WEXITSTATUS(st))
			return 0;
		return -1;
C_EXIT:
		kill(c_pid, SIGTERM);
		return -1;
	}
	return 0;
	/*
EXIT:
	if (dt.summary_out)
		fclose(dt.summary_out);
	return out;
	*/
}

/*
int main(int argc, char *argv[])
{
	int i;
	char *infile = NULL;
	struct app_dt dt;
	char path[MAX_PATH_LEN + 1];

	memset(&dt, 0, sizeof(dt));
	dt.outdir = ".";
	for (i = 1; i < argc;) {
		if (strcmp(argv[i], "-o") == 0) {
			chk_next(argc, i);
			i++;
			dt.outdir = argv[i];
			i++;
		} else if (strcmp(argv[i], "-s") == 0) {
			chk_next(argc, i);
			i++;
			dt.srcdir = argv[i];
			i++;
		} else if (strcmp(argv[i], "-f") == 0) {
			if (infile) {
				fprintf(stderr, "coverage_file duplicate\n");
				usage();
				err_exit();
			}
			chk_next(argc, i);
			i++;
			infile = argv[i];
			i++;
		} else {
			usage();
			err_exit();
		}
	}
	if (!infile) {
		fprintf(stderr, "coverage_file not specified\n");
		usage();
		err_exit();
	}
	dt.in = fopen(infile, "r");
	if (!dt.in) {
		fprintf(stderr, "%s can't open.(%s)\n",
			infile, strerror(errno));
		err_exit();
	}

	if (dir_chk_and_create(dt.outdir) < 0)
		err_exit();
	path[MAX_PATH_LEN] = '\0';
	snprintf(path, MAX_PATH_LEN, "%s/each", dt.outdir);
	if (dir_chk_and_create(path) < 0)
		err_exit();

	path[MAX_PATH_LEN] = '\0';
	snprintf(path, MAX_PATH_LEN, "%s/summary.html", dt.outdir);
	dt.summary_out = fopen(path, "w+");
	if (!dt.summary_out) {
		fprintf(stderr, "%s can't open.(%s)\n", path, strerror(errno));
		err_exit();
	}
	if (out_top_html(&dt) < 0)
		err_exit();
	if (parse_input(&dt) < 0)
		err_exit();
	fclose(dt.summary_out);
	exit(0);
}
*/
