/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bt_split.c - log split by each pid program                               */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2006                         */
/*             Authors: Yumiko Sugita (sugita@sdl.hitachi.co.jp),            */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <sys/mman.h>
#define CONFIG_SMP	/* for NR_CPUS */
#include <linux/threads.h>
#include <getopt.h>
#include <libiberty.h>
#include "bt_utils.h"
#include "avltree.h"

#define BT_SPLIT_VER	BT_VER
#define	COPYRIGHT	"Copyright (c) Hitachi, Ltd. 2005-2006"

struct tv {
	unsigned long		tv_sec;
	unsigned long		tv_usec;
};

struct __pid_info {
	int			cpu;
	struct tv		tv;
	off_t			i_rec;
	off_t			n_rec;
};

struct pid_log_info {
	pid_t			pid;
	char			comm[BT_COMM_LEN + 1];
	struct tv		comm_tv;
	node			*info;	// avl-tree of struct __pid_info
};

struct cpu_file_info {
	int			fd;
	off_t			size;
};

static struct cpu_file_info cpu_file_info[NR_CPUS];
static struct pid_log_info **pid_log_info;
static size_t pid_max = 1024;

static int __dump_pid_pos(void *__dt, void *user_data)
{
	struct __pid_info *__p = (struct __pid_info*)__dt;

	printf("   cpu:%d, sec:%ld, usec:%ld\n",
	       __p->cpu, __p->tv.tv_sec, __p->tv.tv_usec);
	return 0;
}

void dump_pid_pos(void)
{
	size_t i;
	struct pid_log_info *p;

	if (!pid_log_info)
		return;
	for (i = 0; i < pid_max; i++) {
		p = pid_log_info[i];
		if (!p)
			break;
		printf("pid:%d (%s) ------>\n", p->pid, p->comm);
		for_each_node(p->info, __dump_pid_pos, NULL);
	}
}

void print_pid(void)
{
	size_t i;
	struct pid_log_info *p;

	if (!pid_log_info)
		return;
	for (i = 0; i < pid_max; i++) {
		p = pid_log_info[i];
		if (!p)
			break;
		printf("%d %s\n", p->pid, p->comm);
	}
}

static void __free_pid_info(void *__dt)
{
	struct __pid_info *__p = (struct __pid_info*)__dt;

	free(__p);
}

void free_pid_info(void)
{
	size_t i;
	struct pid_log_info *p;

	if (!pid_log_info)
		return;
	for (i = 0; i < pid_max; i++) {
		p = pid_log_info[i];
		if (!p)
			break;
		free_tree(p->info, __free_pid_info);
		free(p);
		pid_log_info[i] = NULL;
	}
	free(pid_log_info);
	pid_log_info = NULL;
}

void cpu_file_close(void)
{
	int i;
	struct cpu_file_info *cpu_info;

	for (i = 0; i < NR_CPUS; i++) {
		cpu_info = &cpu_file_info[i];
		if (cpu_info->fd)
			close(cpu_info->fd);
	}
}

struct pid_log_info* find_pid_info_index(pid_t pid)
{
	size_t i, step = 1024;
	struct pid_log_info *p;

	for (i = 0; i < pid_max; i++) {
		p = pid_log_info[i];
		if (!p)
			goto ALLOC;
		if (p->pid == pid)
			return p;
	}
	/* enlarge pid_log_info area */
	pid_log_info = xrealloc(pid_log_info,
				(pid_max + step) *sizeof(struct pid_log_info*));
	for (; i < pid_max + step; i++)
		pid_log_info[i] = NULL;
	i = pid_max;
	pid_max += step;
ALLOC:
	p = xmalloc(sizeof(struct pid_log_info));
	p->pid = pid;
	p->comm[0] = '\0';
	p->info = NULL;
	pid_log_info[i] = p;
	return p;
}

static inline int compare_timestamp(struct tv *t1, struct tv *t2)
{
	if (t1->tv_sec < t2->tv_sec)
		return 1;
	else if (t1->tv_sec > t2->tv_sec)
		return -1;
	else {
		if (t1->tv_usec < t2->tv_usec)
			return 1;
		else if (t1->tv_usec > t2->tv_usec)
			return -1;
	}
	return 0;
}

static int f_compare_timestamp(void *target, void *each)
{
	struct tv *t1 = &((struct __pid_info*)target)->tv;
	struct tv *t2 = &((struct __pid_info*)each)->tv;

	return compare_timestamp(t2, t1);
}

struct __pid_info* add_cpu_pid_info(int cpu, struct pid_record *rec,
				    off_t i_rec)
{
	struct pid_log_info *p;
	struct __pid_info *__p;

	/* get pid-info */
	p = find_pid_info_index(rec->pid);
	if (!p)
		return NULL;

	/* setup pid-sub-info for add*/
	__p = xmalloc(sizeof(struct __pid_info));
	__p->cpu = cpu;
	__p->tv.tv_sec = rec->tv_sec;
	__p->tv.tv_usec = rec->tv_usec & ~BT_FLAG_PID;
	__p->i_rec = i_rec;
	__p->n_rec = 0;

	/* add pid-sub-info */
	p->info = insert_tree(__p, p->info, f_compare_timestamp);
	if (!p->info)
		return NULL;
	return __p;
}

int set_pid_comm(struct pid_record *pid_rec, char *comm)
{
	struct pid_log_info *p;
	struct tv tv;
	int rc;

	p = find_pid_info_index(pid_rec->pid);
	if (!p)
		return -1;
	tv.tv_sec = pid_rec->tv_sec;
	tv.tv_usec = pid_rec->tv_usec;
	rc = compare_timestamp(&p->comm_tv, &tv);
	if (rc > 0) {
		p->comm_tv.tv_sec = tv.tv_sec;
		p->comm_tv.tv_usec = tv.tv_usec;
		p->comm[BT_COMM_LEN] = '\0';
		memcpy(p->comm, comm, BT_COMM_LEN);
	}
	return 0;
}

#define MODE_SPLIT_ONLY			0
#define MODE_SEARCH_ONLY		1
#define MODE_SEARCH_AND_SPLIT		2
#define MODE_PRINT_PID			3
#define MODE_PRINT_PID_SYSCALL_TIME	4

/* data for chk_pid_pos_per_cpu */
struct chk_pid_args {
	int mode;
	unsigned long addr;
	int cpu;
	struct pid_record last_rec;
	struct __pid_info *last_info;
};

int chk_pid_pos_per_cpu(struct bt_record *p, off_t i_rec, void *dt)
{
	struct chk_pid_args *args = (struct chk_pid_args*)dt;
	struct warn_record *warn;
	struct __pid_info *__p;

	if (is_pid_record(p)) {
		__p = args->last_info;
		if (__p)
			__p->n_rec = i_rec - __p->i_rec;
		args->last_rec = *(struct pid_record*)p;
		args->last_info = add_cpu_pid_info(args->cpu,
						   (struct pid_record*)p,
						   i_rec);
		if (!args->last_info)
			return -1;
		return 0;
	}
	if (is_comm_record(p) && args->last_rec.tv_sec) {
		if (set_pid_comm(&args->last_rec,
				 ((struct comm_record*)p)->comm) < 0)
			return - 1;
		return 0;
	}
	if (is_warn_record(p)) {
		warn = (struct warn_record*)p;
		printf("WARN(cpu:%d, rec:%lld): bts left only: %ld\n",
		       args->cpu, i_rec, warn->left);
		return 0;
	}
	if (args->mode == MODE_SPLIT_ONLY || args->mode == MODE_PRINT_PID)
		return 0;
	/* Don't check p->from cause Pentium-M's p->from is not correct. */
	if (!args->last_rec.tv_sec || p->to != args->addr)
		return 0;
	printf("%12ld %6ld %12lld %3d %6ld 0x%08lx\n",
	       args->last_rec.tv_sec,
	       args->last_rec.tv_usec & ~BT_FLAG_PID,
	       i_rec, args->cpu, args->last_rec.pid, args->addr);
	return 0;
}

int chk_pid_pos(int mode, char *dir_path, unsigned long addr)
{
	DIR *dir;
	struct dirent* d;
	int max = 128;
	char path[max];
	struct stat st;
	int rc, cpu;
	struct cpu_file_info *cpu_info;
	struct chk_pid_args args;
	struct __pid_info *__p;

	if ((dir = opendir(dir_path)) == NULL) {
		fprintf(stderr, "can't open %s\n", dir_path);
		return -1;
	}
	if (mode == MODE_SEARCH_ONLY || mode == MODE_SEARCH_AND_SPLIT) {
		printf("sec          usec   rec-index    cpu pid    address\n");
		printf("------------ ------ ------------ --- ------ ----------\n");
	}
	args.mode = mode;
	args.addr = addr;
	rc = 0;
	while ((d = readdir(dir)) != NULL) {
		if (strcmp(".", d->d_name) == 0 || strcmp("..", d->d_name) == 0)
			continue;
		snprintf(path, max, "%s/%s", dir_path, d->d_name);
		if (lstat(path, &st) < 0) {
			fprintf(stderr, "%s lstat failed.(%s)\n", path,
				strerror(errno));
			rc = -1;
			break;
		}
		if (S_ISDIR(st.st_mode) ||
		    sscanf(d->d_name, "cpu%d", &cpu) != 1)
			continue;
		cpu_info = &cpu_file_info[cpu];
		if ((cpu_info->fd = open(path, O_RDONLY)) < 0) {
			fprintf(stderr, "%s open failed.(%s)\n", path,
				strerror(errno));
			rc = -1;
			break;
		}
		if (!st.st_size) {
			cpu_info->size = 0;
			continue;
		}
		cpu_info->size = st.st_size;
		args.cpu = cpu;
		memset(&args.last_rec, 0, sizeof(struct pid_record));
		args.last_info = NULL;
		rc = for_each_bt_record(cpu_info->fd, cpu_info->size,
					chk_pid_pos_per_cpu, &args);
		if (rc < 0)
			break;
		__p = args.last_info;
		if (__p)
			__p->n_rec = st.st_size / sizeof(struct bt_record)
								- __p->i_rec;
	}
	closedir(dir);
	if (rc < 0)
		return rc;
	return 0;
}

static int cp_records(void *__dt, void *user_data)
{
	struct __pid_info *__p = (struct __pid_info*)__dt;
	struct cpu_file_info *cpu_info = &cpu_file_info[__p->cpu];
	int to_fd = (int)user_data;
	off_t all_len, len, max = sizeof(struct bt_record) * 1024;
	ssize_t rc;
	char buf[sizeof(struct bt_record) * 1024];

	if (u_lseek(cpu_info->fd, __p->i_rec * sizeof(struct bt_record),
		    SEEK_SET) < 0)
		return -1;
	all_len = __p->n_rec * sizeof(struct bt_record);
	while (all_len) {
		len = all_len > max ? max : all_len;
		all_len -= len;
		rc = u_read(cpu_info->fd, buf, len);
		if (rc < 0)
			return -1;
		rc = u_write(to_fd, buf, len);
		if (rc < 0)
			return -1;
	}
	return 0;
}

int create_pid_files(char *dir_path)
{
	size_t i;
	struct pid_log_info *p;
	int fd;
	int path_max = 128;
	char path[path_max];

	for (i = 0; i < pid_max; i++) {
		p = pid_log_info[i];
		if (!p)
			break;
		snprintf(path, path_max, "%s/%d", dir_path, p->pid);
		if ((fd = open(path, O_CREAT|O_EXCL|O_RDWR, S_IRUSR|S_IWUSR))
		    < 0) {
			fprintf(stderr, "can't create %d.(%s)\n", p->pid,
				strerror(errno));
			return -1;
		}
		if (for_each_node(p->info, cp_records, (void*)fd) < 0) {
			close(fd);
			return -1;
		}
		close(fd);
	}
	return 0;
}

struct syscall_time {
	long 			n_syscall; /* -1 means wait BT_FLAG_T_START */
	unsigned long long	timestamp;
};

#define NR_syscalls	285
struct syscall_time_tbl {
	long			n;
	unsigned long long	total_time;
	unsigned long long	min_time;
	unsigned long long	max_time;
} syscall_time_tbl[NR_syscalls];

static int print_syscall_time(void *__dt, void *user_data)
{
	struct __pid_info *__p = (struct __pid_info*)__dt;
	struct cpu_file_info *cpu_info = &cpu_file_info[__p->cpu];
	struct syscall_time *st = (struct syscall_time*)user_data;
	off_t all_len, i, len, max = sizeof(struct bt_record) * 1024;
	ssize_t rc;
	char buf[sizeof(struct bt_record) * 1024];
	struct tmr_record *rec;
	struct syscall_time_tbl *tbl;
	unsigned long long syscall_t;

	if (u_lseek(cpu_info->fd, __p->i_rec * sizeof(struct bt_record),
		    SEEK_SET) < 0)
		return -1;
	all_len = __p->n_rec * sizeof(struct bt_record);
	while (all_len) {
		len = all_len > max ? max : all_len;
		all_len -= len;
		rc = u_read(cpu_info->fd, buf, len);
		if (rc < 0)
			return -1;
		rec = (struct tmr_record*)buf;
		for (i = 0; i < len / sizeof(struct bt_record); i++) {
			if (rec->flags & BT_FLAG_T_START) {
				if (st->n_syscall >= 0) {
					fprintf(stderr,
						"There is no system-call end timestamp (%ld:%20llu -> %ld:%20llu)\n",
						st->n_syscall, st->timestamp,
						rec->flags & ~BT_FLAG_T_START,
						rec->timestamp);
					//return -1;
				}
				st->n_syscall = rec->flags & ~BT_FLAG_T_START;
				st->timestamp = rec->timestamp;
			} else if (rec->flags & BT_FLAG_T_STOP) {
				/* trace start when during system-call */
				if (st->n_syscall < 0)
					continue;
				syscall_t = rec->timestamp - st->timestamp;
				printf("  %3ld %20llu\n",
				       st->n_syscall, syscall_t);
				tbl = &syscall_time_tbl[st->n_syscall];
				tbl->n++;
				tbl->total_time += syscall_t;
				if (tbl->total_time < syscall_t)
					fprintf(stderr,
						"total time too short\n");
				if (syscall_t < tbl->min_time)
					tbl->min_time = syscall_t;
				if (syscall_t > tbl->max_time)
					tbl->max_time = syscall_t;
				st->n_syscall = -1;
			}
			rec++;
		}
	}
	return 0;
}

int print_pid_syscall_time(void)
{
	size_t i;
	struct pid_log_info *p;
	struct syscall_time st;
	struct syscall_time_tbl *tbl;
	long double avrg;

	for (i = 0; i < NR_syscalls; i++) {
		tbl = &syscall_time_tbl[i];
		tbl->min_time = 0xffffffffffffffffULL;
	}
	for (i = 0; i < pid_max; i++) {
		p = pid_log_info[i];
		if (!p)
			break;
		printf("====== System call time of %d(%s) ======\n",
		       p->pid, p->comm);
		printf("  n   cnt\n");
		printf("  --- --------------------\n");
		st.n_syscall = -1;
		if (for_each_node(p->info, print_syscall_time, &st) < 0)
			return -1;
	}
	for (i = 0; i < NR_syscalls; i++) {
		tbl = &syscall_time_tbl[i];
		if (!tbl->n) {
			tbl->min_time = 0;
			avrg = 0.0;
		} else
			avrg = (long double)tbl->total_time / tbl->n;
		printf("%3d,%lu,%llu,%llu,%.2Lf,%llu\n",
		       i, tbl->n, tbl->min_time, tbl->max_time, avrg,
		       tbl->total_time);
	}
	return 0;
}

void err_exit(void)
{
	free_pid_info();
	cpu_file_close();
	exit(1);
}

void usage(void)
{
	fprintf(stderr, "bt_split %s\n", BT_SPLIT_VER);
	fprintf(stderr, "    %s\n\n", COPYRIGHT);
#if 0
	fprintf(stderr, "bt_split [-p] [-S addr] [-s addr] -d dir\n");
	fprintf(stderr, "  -p: print pid list\n");
	fprintf(stderr, "  -S: search address, and do not split logfile\n");
	fprintf(stderr, "  -s: search address\n");
#else
	fprintf(stderr, "bt_split -d dir\n");
#endif
	fprintf(stderr, "  -d: log directory\n");
}

int main(int argc, char *argv[])
{
	int verbose = 0, mode = MODE_SPLIT_ONLY;
	unsigned long addr;
	char c, *dir = NULL, *p_end;

	while ((c = getopt(argc, argv, "pPs:S:d:v")) != -1) {
		switch (c) {
		case 'p':
		case 'P':
			if (mode != MODE_SPLIT_ONLY) {
				usage();
				err_exit();
			}
			mode = c == 'p' ?
				MODE_PRINT_PID : MODE_PRINT_PID_SYSCALL_TIME;
			break;
		case 's':
		case 'S':
			if (mode != MODE_SPLIT_ONLY) {
				usage();
				err_exit();
			}
			mode = c == 'S' ?
				MODE_SEARCH_ONLY : MODE_SEARCH_AND_SPLIT;
			addr = strtoul(optarg, &p_end, 0);
			if (*p_end != '\0') {
				fprintf(stderr, "invalid address(%s).\n",
					optarg);
				err_exit();
			}
			break;
		case 'd':
			dir = optarg;
			break;
		case 'v':
			verbose = 1;
			break;
		default:
			usage();
			err_exit();
		}
	}
	if (optind < argc || !dir) {
		usage();
		err_exit();
	}
	pid_log_info = xcalloc(pid_max, sizeof(struct pid_log_info*));

	if (chk_pid_pos(mode, dir, addr) < 0)
		err_exit();
	if (mode == MODE_PRINT_PID) {
		print_pid();
		goto EXIT;
	} else if (mode == MODE_PRINT_PID_SYSCALL_TIME) {
		if (print_pid_syscall_time() < 0)
			err_exit();
		goto EXIT;
	}
	if (verbose)
		dump_pid_pos();
	if (mode != MODE_SEARCH_ONLY) {
		if (create_pid_files(dir) < 0)
			err_exit();
	}
EXIT:
	free_pid_info();
	cpu_file_close();

	exit(0);
}
