/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bt_utils.c - utilities                                                   */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2006                         */
/*             Authors: Yumiko Sugita (sugita@sdl.hitachi.co.jp),            */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <sys/utsname.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include "bt_utils.h"

/*----------------------------------------------------------------------------*/
/*  utility functions                                                         */
/*----------------------------------------------------------------------------*/
int u_open(const char *path, off_t *size)
{
	int fd, rc;
	struct stat st;

	if ((fd = open(path, O_RDONLY)) < 0) {
		fprintf(stderr, "can't open %s.(%s)\n", path, strerror(errno));
		return fd;
	}
	if ((rc = lstat(path, &st)) < 0) {
		fprintf(stderr, "%s lstat failed.(%s)\n", path,strerror(errno));
		return rc;
	}
	*size = st.st_size;
	return fd;
}

int u_close(int fd)
{
	int rc;

	if (fd < 0)
		return 0;
	if ((rc = close(fd)) < 0)
		fprintf(stderr, "close failed.(%s)\n", strerror(errno));
	return rc;
}

off_t u_lseek(int fd, off_t offset, int whence)
{
	off_t rc;

	if ((rc = lseek(fd, offset, whence)) < 0)
		fprintf(stderr, "lseek failed.(%s)\n", strerror(errno));
	return rc;
}

ssize_t u_read(int fd, void *buf, size_t count)
{
	ssize_t rc = 0;
	size_t tmp;

	for (tmp = count; tmp; tmp -= rc) {
		rc = read(fd, buf, tmp);
		if (rc < 0) {
			if (errno == EINTR)
				continue;
			fprintf(stderr, "read failed.(%s)\n", strerror(errno));
			return rc;
		}
	}
	return count;
}

ssize_t u_write(int fd, const void *buf, size_t count)
{
	ssize_t rc = 0;
	size_t tmp;

	for (tmp = count; tmp; tmp -= rc) {
		rc = write(fd, buf, tmp);
		if (rc < 0) {
			if (errno == EINTR)
				continue;
			fprintf(stderr, "write failed.(%s)\n", strerror(errno));
			return rc;
		}
	}
	return count;
}

static off_t rec_align_size;

static int gcd(int a, int b)
{
	if (a == 0 || b == 0)
		return 0;
	while (a != b) {
		if (a > b)
			a = a - b;
		else
			b = b - a;
	}
	return a;
}

static int lcm(int a, int b)
{
	if (a == 0 || b == 0)
		return 0;
	return (a / gcd(a, b)) * b;	// lcm = a * b / gcd(a, b)
}

static off_t get_rec_align_offset(off_t offset, int get_next_top)
{
	if (!rec_align_size)
		rec_align_size = lcm(getpagesize(), sizeof(struct bt_record));

	offset = offset / rec_align_size * rec_align_size;
	if (get_next_top)
		offset += rec_align_size;
	return offset;
}

#ifdef MMAP_ALIGN_TEST
struct align_test_data {
	off_t	offset;
	off_t	expect1;
	off_t	expect2;
};

static int __test_get_rec_align_offset(struct align_test_data* data)
{
	off_t r;

	r = get_rec_align_offset(data->offset, 0);
	//printf("T:%lld,%d=>%lld\n", data->offset, 0, r);
	if (r != data->expect1)
		return -1;
	r = get_rec_align_offset(data->offset, 1);
	//printf("T:%lld,%d=>%lld\n", data->offset, 1, r);
	if (r != data->expect2)
		return -1;
	return 0;
}

static int test_get_rec_align_offset()
{
	int i;
	struct align_test_data data[] = {
		{0,	0,	12288},
		{1,	0,	12288},
		{12287,	0,	12288},
		{12288,	12288,	24576},
		{12289,	12288,	24576},
	};

	for (i = 0; i < ARRAY_SIZE(data); i++) {
		if (__test_get_rec_align_offset(&data[i]) < 0)
			return -1;
	}
	return 0;
}
#endif

static off_t find_mmappable_size(int fd, off_t size)
{
	size_t size2;
	void *p;

#ifdef MMAP_ALIGN_TEST
	printf("TEST_RESULT:%d\n", test_get_rec_align_offset());
#endif
	for (;;) {
		size2 = size;
#ifdef MMAP_ALIGN_TEST
		if (size == size2 && size <= 12288)
#else
		if (size == size2)
#endif
			break;
		size = get_rec_align_offset((size / 2) - 1, 1);
	}
	for (;;) {
		p = mmap(NULL, size, PROT_READ, MAP_PRIVATE, fd, 0);
		if (p != MAP_FAILED) {
			munmap(p, size);
			break;
		}
		size = get_rec_align_offset((size / 2) - 1, 1);
	}
	return size;
}

/* For the mmap call, offset value is necessary to be page align.
 * Prefix 'poff_' of the local variables mean page-aligned value.
 */
int for_each_block_record(int fd, off_t i_rec_from, off_t i_rec_to,
			  t_func_each_bt_record f, void *dt)
{
	int rc, rec_size = sizeof(struct bt_record);
	off_t poff_from, poff_to, mappable_size, from_diff, size;
	off_t map_size, offset, i_rec;
	char *p_map, *p, *p_max;

	if (i_rec_from == i_rec_to)
		return 0;
	/* First, we find the mmap-able size */
	poff_from = get_rec_align_offset(i_rec_from * rec_size, 0);
	poff_to = get_rec_align_offset(i_rec_to * rec_size, 1);
	mappable_size = find_mmappable_size(fd, poff_to - poff_from);

	from_diff = (i_rec_from * rec_size) - poff_from;
	size = (i_rec_to - i_rec_from) * rec_size;

	/* for DEBUG
	printf("DBG:(i_from=%lld i_to=%lld)\n"
	       "    (poff_from=%lld poff_to=%lld mappable_size=%lld)\n"
	       "    (from_diff=%lld rsize=%lld)\n",
	       i_rec_from, i_rec_to,
	       poff_from, poff_to, mappable_size, from_diff, size);
	       */

	i_rec = i_rec_from;
	for (offset = poff_from; size; offset += map_size) {
		if (from_diff + size > mappable_size)
			map_size = mappable_size;
		else
			map_size = from_diff + size;
		size -= map_size - from_diff;
		p_map = mmap(NULL, map_size, PROT_READ, MAP_PRIVATE, fd,offset);
		/* for DEBUG
		printf("MMAP:from_diff=%lld map_size=%lld offset=%lld\n",
		       from_diff, map_size, offset);
		*/
		if (p_map == MAP_FAILED) {
			fprintf(stderr, "mmap failed.(%s)\n", strerror(errno));
			return -1;
		}
		p = p_map;
		p_max = p + map_size;
		if (from_diff) {
			p += from_diff;
			from_diff = 0;
		}
		for (; p < p_max; p += rec_size, i_rec++) {
			rc = f((struct bt_record*)p, i_rec, dt);
			if (rc) {
				munmap(p_map, map_size);
				return rc;
			}
		}
		munmap(p_map, map_size);
	}
	return rc;
}

int for_each_bt_record(int fd, off_t size, t_func_each_bt_record f, void *dt)
{
	return for_each_block_record(fd, 0, size / sizeof(struct bt_record),
				     f, dt);
}

