/*****************************************************************************/
/* The development of this program is partly supported by IPA                */
/* (Information-Technology Promotion Agency, Japan).                         */
/*****************************************************************************/

/*****************************************************************************/
/*  bt_on_off.c - branch trace module on-off driver                          */
/*  Copyright: Copyright (c) Hitachi, Ltd. 2005-2007                         */
/*             Authors: Yumiko Sugita (sugita@sdl.hitachi.co.jp),            */
/*                      Satoshi Fujiwara (sa-fuji@sdl.hitachi.co.jp)         */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include <linux/module.h>
#ifdef USE_SYS_KPROBES
#  include <linux/kprobes.h>
#else
#  include "djprobe.h"
#endif
#include "bt.h"

static unsigned long on_addr;
static int on_size;
static unsigned long off_addr;
static int off_size;
static int enable_repeat;
static int bt_on_called, bt_off_called;

static void bt_disable_wrapper(void *);
static DECLARE_WORK(bt_disable_work, bt_disable_wrapper, NULL);

//#define DBG_PRINT
#ifdef DBG_PRINT
#  define dprintk(...)	printk(__VA_ARGS__)
#else
#  define dprintk(...)	
#endif

static void bt_on_probe(struct pt_regs *regs)
{
	preempt_disable();
	if (enable_repeat || !bt_on_called) {
		dprintk("bt_on_probe called.\n");
		bt_enable_per_cpu(NULL);
		bt_on_called = 1;
	}
	preempt_enable();
}

static void bt_disable_wrapper(void *__private)
{
	bt_disable();
}

static void bt_off_probe(struct pt_regs *regs)
{
	preempt_disable();
	if (enable_repeat || !bt_off_called) {
		dprintk("bt_off_probe called.\n");
		if (enable_repeat)
			bt_disable_per_cpu(NULL);
		else {
			bt_disable_per_cpu((void*)1);
			/* If any cpu execute the off probe, it disable the
			 * all cpu's trace
			 */
			schedule_delayed_work(&bt_disable_work, 1);
		}
		bt_off_called = 1;
	}
	preempt_enable();
}

#ifdef USE_SYS_KPROBES
static struct kprobe on_kp, off_kp;
static int bt_on_kp_wrapper(struct kprobe *kp, struct pt_regs *regs)
{
	bt_on_probe(regs);
	return 0;
}
static int bt_off_kp_wrapper(struct kprobe *kp, struct pt_regs *regs)
{
	bt_off_probe(regs);
	return 0;
}
#else
static struct djprobe on_djp, off_djp;
static void bt_on_djp_wrapper(struct djprobe *djp, struct pt_regs *regs)
{
	bt_on_probe(regs);
}
static void bt_off_djp_wrapper(struct djprobe *djp, struct pt_regs *regs)
{
	bt_off_probe(regs);
}
#endif

static int bt_on_off_init(void)
{
	int rc;

	if (on_addr) {
		if (!on_size)
			return -EINVAL;
#ifdef USE_SYS_KPROBES
		on_kp.addr = (kprobe_opcode_t*)on_addr;
		on_kp.pre_handler = bt_on_kp_wrapper;
		if ((rc = register_kprobe(&on_kp)) < 0)
			return rc;
#else
		on_djp.addr = (void*)on_addr;
		on_djp.handler = bt_on_djp_wrapper;
		on_djp.size = on_size;
		if ((rc = register_djprobe(&on_djp)) < 0)
			return rc;
#endif
		dprintk("bt_on_probe registered.\n");
	}
	if (off_addr) {
		if (!off_size)
			return -EINVAL;
#ifdef USE_SYS_KPROBES
		off_kp.addr = (kprobe_opcode_t*)off_addr;
		off_kp.pre_handler = bt_off_kp_wrapper;
		if ((rc = register_kprobe(&off_kp)) < 0)
			return rc;
#else
		off_djp.addr = (void*)off_addr;
		off_djp.handler = bt_off_djp_wrapper;
		off_djp.size = off_size;
		if ((rc = register_djprobe(&off_djp)) < 0)
			return rc;
#endif
		dprintk("bt_off_probe registered.\n");
	}
	return 0;
}

static void bt_on_off_cleanup(void)
{
	if (on_addr && on_size) {
#ifdef USE_SYS_KPROBES
		unregister_kprobe(&on_kp);
#else
		unregister_djprobe(&on_djp);
#endif
		dprintk("bt_on_probe unregistered.\n");
	}
	if (off_addr && off_size) {
#ifdef USE_SYS_KPROBES
		unregister_kprobe(&off_kp);
#else
		unregister_djprobe(&off_djp);
#endif
		dprintk("bt_off_probe unregistered.\n");
	}
	return;
}

module_init(bt_on_off_init);
module_exit(bt_on_off_cleanup);

module_param(on_addr, ulong, 0);
MODULE_PARM_DESC(on_addr, "Btrax ON probe address.");
module_param(on_size, int, 0);
MODULE_PARM_DESC(on_size, "Btrax ON probe size.");
module_param(off_addr, ulong, 0);
MODULE_PARM_DESC(off_addr, "Btrax OFF probe address.");
module_param(off_size, int, 0);
MODULE_PARM_DESC(off_size, "Btrax OFF probe size.");
module_param(enable_repeat, int, 0);
MODULE_PARM_DESC(enable_repeat, "enable Btrax ON/OFF repeat.");

MODULE_AUTHOR("Yumiko Sugita <sugita@sdl.hitachi.co.jp>,\n" \
	      "\t\tSatoshi Fujiwara <sa-fuji@sdl.hitachi.co.jp>");
MODULE_DESCRIPTION("Btrax ON/OFF probe controller");
MODULE_LICENSE("GPL");
