/*
 djprobe -- Direct jmp-insertion probe
 Copyright (c) 2005 Hitachi,Ltd.,
 Created by Masami Hiramatsu<hiramatu@sdl.hitachi.co.jp>

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */
#include <linux/version.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/rcupdate.h>
#include <asm/system.h>
#include <asm/pgtable.h>
#include <linux/vmalloc.h>
#include "djprobe.h"

#define _DEBUG_(a) a

#ifndef CONFIG_SMP
cpumask_t cpu_online_map = (cpumask_t){{[0] = 1UL}};
#endif
/*
 * On pentium series, Unsynchronized cross-modifying code
 * operations can cause unexpected instruction execution results.
 * So after code modified, we should synchronize it on each processor.
 */
static void __local_serialize_cpu(void * info) 
{
	__asm__ __volatile__ ("cpuid" : : : "ax", "bx", "cx", "dx");
}

static inline void smp_serialize_cpus(void) 
{
	on_each_cpu(__local_serialize_cpu, NULL, 1,1);
}

/* jmp code manipulators */
struct __arch_jmp_op {
	char op;
	long raddr;
} __attribute__((packed));
/* insert jmp code */
static inline void __set_jmp_op(void *from, void *to)
{
	struct __arch_jmp_op *jop;
	jop = (struct __arch_jmp_op *)from;
	jop->raddr=(long)(to) - ((long)(from) + 5);
	smp_serialize_cpus();
	jop->op = RELATIVEJUMP_INSTRUCTION;
}
/* switch back to the kprobe */
static inline void __set_breakpoint_op(void *dest, void *orig)
{
	struct __arch_jmp_op *jop = (struct __arch_jmp_op *)dest, 
		*jop2 = (struct __arch_jmp_op *)orig;

	jop->op = BREAKPOINT_INSTRUCTION;
	smp_serialize_cpus();
	jop->raddr = jop2->raddr;
}
#include <asm/segment.h>

#if KERNEL_VERSION(2,6,10) > LINUX_VERSION_CODE
#define KPROBE_INSN_HEAD(kp) (kp)->insn[0]
#else
#define KPROBE_INSN_HEAD(kp) (kp)->ainsn.insn[0]
#endif

/* djprobe call back function: called from stub code */
extern void asmlinkage djprobe_callback(struct djprobe_instance * djpi,
					struct pt_regs *regs)
{
	struct djprobe * djp;
	list_for_each_entry_rcu(djp, &djpi->plist, plist) {
		if (djp->handler)
			djp->handler(djp, regs);
	}
}

/*
 * Copy post processing instructions
 * Target instructions MUST be relocatable.
 */
static int  arch_prepare_djprobe_instance(struct djprobe_instance *djpi,
						   unsigned long size)
{
	kprobe_opcode_t *stub;
	stub = djpi->stub.insn;

	/* copy arch-dep-instance from template */
	memcpy((void*)stub, (void*)&arch_tmpl_stub_entry, ARCH_STUB_SIZE);

	/* set probe information */
	*((long*)(stub + ARCH_STUB_VAL_IDX)) = (long)djpi;
	/* set probe function */
	*((long*)(stub + ARCH_STUB_CALL_IDX)) = (long)djprobe_callback;

	/* copy instructions into the middle of axporbe instance */
	memcpy((void*)(stub + ARCH_STUB_INST_IDX),
	       (void*)djpi->kp.addr, size);
	djpi->stub.size = size;

	/* set returning jmp instruction at the tail of axporbe instance*/
	__set_jmp_op(stub + ARCH_STUB_INST_IDX + size, 
		     (void*)((long)djpi->kp.addr + size));

	return 0;
}

/*
 * Insert "jmp" instruction into the probing point.
 */
static int arch_install_djprobe_instance(struct djprobe_instance *djpi)
{
	kprobe_opcode_t *stub;
	stub = djpi->stub.insn;
	__set_jmp_op((void*)djpi->kp.addr, (void*)stub);
	return 0;
}
/* Write back original instructions & kprobe */
void  arch_uninstall_djprobe_instance(struct djprobe_instance *djpi)
{
	kprobe_opcode_t *stub;
	stub = &djpi->stub.insn[ARCH_STUB_INST_IDX];
	__set_breakpoint_op((void*)djpi->kp.addr, (void*)stub);
	/* fixup dummy instruction */
	KPROBE_INSN_HEAD(&(djpi->kp)) = djpi->stub.insn[ARCH_STUB_INST_IDX];
}

/* djprobe handler : switch to a bypass code */
int  djprobe_bypass_handler(struct kprobe * kp, struct pt_regs * regs)
{
	struct djprobe_instance *djpi = 
		container_of(kp,struct djprobe_instance, kp);
	kprobe_opcode_t *stub = djpi->stub.insn;

	if (DJPI_EMPTY(djpi)) {
		/* fixup dummy instruction */
		KPROBE_INSN_HEAD(kp) = djpi->stub.insn[ARCH_STUB_INST_IDX];
		return 0;
	} else {
		regs->eip = (unsigned long)stub;
		regs->eflags |= TF_MASK;
		regs->eflags &= ~IF_MASK;
		/*
		 * dummy return code :
		 * This code is to avoid to be changed eip value by 
		 * resume_execute() of kprobes
		 */
		KPROBE_INSN_HEAD(kp) = RETURN_INSTRUCTION;
		return 1; /* already prepared */
	}
}


// common code

/*get_insn_slot() was copied from kprobes.c (2.6.13) */
/*
 * kprobe->ainsn.insn points to the copy of the instruction to be
 * single-stepped. x86_64, POWER4 and above have no-exec support and
 * stepping on the instruction on a vmalloced/kmalloced/data page
 * is a recipe for disaster
 */
#define INSNS_PER_PAGE(size) (PAGE_SIZE/(size * sizeof(kprobe_opcode_t)))

struct kprobe_insn_page {
	struct hlist_node hlist;
	kprobe_opcode_t *insns;		/* Page of instruction slots */
	int nused;
	char slot_used[1];
};

struct kprobe_insn_page_list {
	struct hlist_head list;
	int insn_size;			/* Size of an instruction slot */
};

/**
 * __get_insn_slot() - Find a slot on an executable page for an instruction.
 * We allocate an executable page if there's no room on existing ones.
 */
static kprobe_opcode_t 
 *__get_insn_slot(struct kprobe_insn_page_list *pages)
{
	struct kprobe_insn_page *kip;
	struct hlist_node *pos;
	int ninsns = INSNS_PER_PAGE(pages->insn_size);

	hlist_for_each(pos, &pages->list) {
		kip = hlist_entry(pos, struct kprobe_insn_page, hlist);
		if (kip->nused < ninsns ) {
			int i;
			for (i = 0; i < ninsns; i++) {
				if (!kip->slot_used[i]) {
					kip->slot_used[i] = 1;
					kip->nused++;
					return kip->insns + (i * pages->insn_size);
				}
			}
			/* Surprise!  No unused slots.  Fix kip->nused. */
			kip->nused = ninsns;
		}
	}

	/* All out of space.  Need to allocate a new page. Use slot 0.*/
	kip = kmalloc(sizeof(struct kprobe_insn_page)+sizeof(char)*(ninsns-1),
		      GFP_KERNEL);
	if (!kip) {
		return NULL;
	}

	/*
	 * Use module_alloc so this page is within +/- 2GB of where the
	 * kernel image and loaded module images reside. This is required
	 * so x86_64 can correctly handle the %rip-relative fixups.
	 */
	/*kip->insns = __vmalloc(PAGE_SIZE, GFP_KERNEL | __GFP_HIGHMEM,
			       PAGE_KERNEL_EXEC);*/
	kip->insns = vmalloc(PAGE_SIZE);
	if (!kip->insns) {
		kfree(kip);
		return NULL;
	}
	INIT_HLIST_NODE(&kip->hlist);
	hlist_add_head(&kip->hlist, &pages->list);
	memset(kip->slot_used, 0, ninsns);
	kip->slot_used[0] = 1;
	kip->nused = 1;
	return kip->insns;
}

static void  __free_insn_slot(struct kprobe_insn_page_list *pages, 
				       kprobe_opcode_t *slot)
{
	struct kprobe_insn_page *kip;
	struct hlist_node *pos;
	int ninsns = INSNS_PER_PAGE(pages->insn_size);

	hlist_for_each(pos, &pages->list) {
		kip = hlist_entry(pos, struct kprobe_insn_page, hlist);
		if (kip->insns <= slot &&
		    slot < kip->insns + (ninsns * pages->insn_size)) {
			int i = (slot - kip->insns) / pages->insn_size;
			kip->slot_used[i] = 0;
			kip->nused--;
			if (kip->nused == 0) {
				/*
				 * Page is no longer in use.  Free it unless
				 * it's the last one.  We keep the last one
				 * so as not to have to set it up again the
				 * next time somebody inserts a probe.
				 */
				hlist_del(&kip->hlist);
				if (hlist_empty(&pages->list)) {
					INIT_HLIST_NODE(&kip->hlist);
					hlist_add_head(&kip->hlist,
						       &pages->list);
				} else {
					vfree(kip->insns);
					kfree(kip);
				}
			}
			return;
		}
	}
}

/*
 * The djprobe do not refer instances list when probe function called.
 * This list is operated on registering and unregistering djprobe.
 */

#ifndef DEFINE_SPINLOCK
#define DEFINE_SPINLOCK(x) spinlock_t x = SPIN_LOCK_UNLOCKED
#endif
static LIST_HEAD(djprobe_list);
static DEFINE_SPINLOCK(djprobe_lock);
/* Instruction pages for djprobe's stub code */
static struct kprobe_insn_page_list djprobe_insn_pages = {
	HLIST_HEAD_INIT, 0};

static inline void __free_djprobe_instance(struct djprobe_instance *djpi)
{
	list_del(&djpi->list);
	if (djpi->kp.addr) unregister_kprobe(&(djpi->kp));
	__free_insn_slot(&djprobe_insn_pages, djpi->stub.insn);
	kfree(djpi);
}

static inline struct djprobe_instance * 
	__get_djprobe_instance(void *addr, int size)
{
	struct djprobe_instance *djpi;
	list_for_each_entry(djpi, &djprobe_list, list) {
		if (((long)addr < (long)djpi->kp.addr + DJPI_ARCH_SIZE(djpi))
		    && ((long)djpi->kp.addr < (long)addr + size)) {
			return djpi;
		}
	}
	return NULL;
}

/* smp code */
static void  work_check_djprobe_instances(void *data);
static DEFINE_PER_CPU(struct work_struct, djprobe_check_works);

static inline void init_djprobe(void)
{
	int cpu;
	struct work_struct *wk;
	djprobe_insn_pages.insn_size = ARCH_STUB_SIZE;
	for_each_cpu(cpu) { /* we should initialize all percpu data */
		wk = &per_cpu(djprobe_check_works, cpu);
		INIT_WORK(wk, work_check_djprobe_instances, NULL);
	}
}

static void  work_check_djprobe_instances(void *data)
{
	struct list_head *pos;
	struct djprobe_instance *djpi;

	spin_lock(&djprobe_lock);
	list_for_each(pos, &djprobe_list) {
		djpi = container_of(pos, struct djprobe_instance, list);
		if (DJPI_CHECKED(djpi)) 
			continue; /* already checked */
		cpu_set(smp_processor_id(), djpi->checked_cpus);
		if (DJPI_CHECKED(djpi)) {
			if (DJPI_EMPTY(djpi)) {
				pos = pos->prev; /* pos is going to be freed */
				__free_djprobe_instance(djpi);
			} else {
				arch_install_djprobe_instance(djpi);
			}
		}
	}
	spin_unlock(&djprobe_lock);
}

static void  
	schedule_check_djprobe_instance(struct djprobe_instance *djpi) 
{
	int cpu;
	struct work_struct *wk;
	cpus_clear(djpi->checked_cpus);
	for_each_online_cpu(cpu) {
		wk = &per_cpu(djprobe_check_works, cpu);
		if (list_empty(&wk->entry)) /* not scheduled yet */
			schedule_delayed_work_on(cpu, wk, 0);
	}
}

#define __install_djprobe_instance(djpi) \
	schedule_check_djprobe_instance(djpi)

#define __uninstall_djprobe_instance(djpi) \
	schedule_check_djprobe_instance(djpi)

/* Use kprobe to check safety and install */
static int  install_djprobe_instance(struct djprobe_instance *djpi)
{
	int ret;
	ret = register_kprobe(&(djpi->kp));
	if (ret == 0) {
		__install_djprobe_instance(djpi);
	}
	return ret;
}

/* Use kprobe to check safety and release */
static void  uninstall_djprobe_instance(struct djprobe_instance *djpi)
{
	arch_uninstall_djprobe_instance(djpi);
	__uninstall_djprobe_instance(djpi);
}

int  register_djprobe(struct djprobe * djp)
{
	struct djprobe_instance *djpi;
	int ret = 0;
	
	if (djp == NULL || djp->addr == NULL || 
	    djp->size > ARCH_STUB_INSN_MAX || 
	    djp->size < ARCH_STUB_INSN_MIN || 
	    djp->inst != NULL)
		return -EINVAL;

	spin_lock(&djprobe_lock);
	INIT_LIST_HEAD(&djp->plist);
	/* check confliction with other djprobes */
	djpi = __get_djprobe_instance(djp->addr, djp->size);
	if (djpi) {
		if (djpi->kp.addr == djp->addr) {
			djp->inst = djpi;
			list_add_rcu(&djp->plist, &djpi->plist);
			goto out;
		} else {
			ret = -EEXIST; /* a djprobe were inserted */
			goto out;
		}
	}
	/* make a new instance */
	djpi = kmalloc(sizeof(struct djprobe_instance),GFP_KERNEL);
	if (djpi == NULL) {
		ret = -ENOMEM; /* memory allocation error */
		goto out;
	}
	memset(djpi, 0, sizeof(struct djprobe_instance)); /* for kprobe */
	/* attach */
	INIT_LIST_HEAD(&djpi->plist);
	djp->inst = djpi;
	list_add_rcu(&djp->plist, &djpi->plist);
	djpi->kp.addr = djp->addr;
	INIT_LIST_HEAD(&djpi->list);
	list_add(&djpi->list, &djprobe_list);

	/* prepare stub */
	djpi->stub.insn = __get_insn_slot(&djprobe_insn_pages);
	if (djpi->stub.insn == NULL) {
		kfree(djpi);
		ret = -ENOMEM; /* memory allocation error */
		goto out;
	}
	djpi->kp.pre_handler = djprobe_bypass_handler;
	arch_prepare_djprobe_instance(djpi, djp->size); /*TODO : remove size*/
	ret = install_djprobe_instance(djpi);
	if (ret < 0) { /* failed to install */
		djp->inst = NULL;
		list_del_rcu(&djp->plist);
		djpi->kp.addr = NULL;
		__free_djprobe_instance(djpi);
	}
out:
	spin_unlock(&djprobe_lock);
	return ret;
}

void  unregister_djprobe(struct djprobe * djp)
{
	struct djprobe_instance *djpi;
	if (djp == NULL || djp->inst == NULL) 
		return ;

	djpi = djp->inst;
	spin_lock(&djprobe_lock);
	djp->inst = NULL;
	list_del_rcu(&djp->plist);
	if (DJPI_EMPTY(djpi)) {
		uninstall_djprobe_instance(djpi);
	}
	spin_unlock(&djprobe_lock);
}

EXPORT_SYMBOL_GPL(register_djprobe);
EXPORT_SYMBOL_GPL(unregister_djprobe);

/*
 * Initiation/Termination Functions.
 */ 

static int mod_init_djprobe(void) 
{
	/* initialize the list for instances */
	init_djprobe();
	return 0;
}

static void mod_exit_djprobe(void)
{
	return ;
}

module_init(mod_init_djprobe);
module_exit(mod_exit_djprobe);
MODULE_AUTHOR("M.Hiramatsu <hiramatu@sdl.hitachi.co.jp>");
MODULE_LICENSE("GPL");


