/*
 * Crackerjack Project
 *
 * Copyright (C) 2007-2008, Hitachi, Ltd.
 * Author(s): Takahiro Yasui <takahiro.yasui.mp@hitachi.com>,
 *            Yumiko Sugita <yumiko.sugita.yf@hitachi.com>,
 *            Satoshi Fujiwara <sa-fuji@sdl.hitachi.co.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * $Id:$
 *
 */

#include <sys/syscall.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <getopt.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <time.h>
#include <signal.h>

#include "../../common.j.h/include_j_h.h"

/*
 * Macros
 */
#define SYSCALL_NAME	"clock_nanosleep"


/*
 * Global variables
 */
static int opt_debug;
static char *progname;

enum test_type {
	NORMAL,
	NULL_POINTER,
	SEND_SIGINT,
};


/*
 * Data Structure
 */
struct test_case {
	int ttype;
	clockid_t clk_id;
	int flags;
	time_t sec;
	long nsec;
	int ret;
	int err;
};


/* Test cases
 *
 *   test status of errors on man page
 *
 *   EINTR		v (function was interrupted by a signal)
 *   EINVAL		v (invalid tv_nsec, etc.)
 *   ENOTSUP		can't check because not supported clk_id generates
 *   			EINVAL
 */
static struct test_case tcase[] = {
	{ // case00
		.ttype		= NORMAL,
		.clk_id		= CLOCK_REALTIME,
		.flags		= 0,
		.sec		= 0,
		.nsec		= 500000000, // 500msec
		.ret		= 0,
		.err		= 0,
	},
	{ // case01
		.ttype		= NORMAL,
		.clk_id		= CLOCK_MONOTONIC,
		.flags		= 0,
		.sec		= 0,
		.nsec		= 500000000, // 500msec
		.ret		= 0,
		.err		= 0,
	},
	{ // case02
		.ttype		= NORMAL,
		.clk_id		= CLOCK_REALTIME,
		.flags		= 0,
		.sec		= 0,
		.nsec		= -1,	// invalid
		.ret		= EINVAL,
		.err		= 0,
	},
	{ // case03
		.ttype		= NORMAL,
		.clk_id		= CLOCK_REALTIME,
		.flags		= 0,
		.sec		= 0,
		.nsec		= 1000000000,	// invalid
		.ret		= EINVAL,
		.err		= 0,
	},
	{ // case04
		.ttype		= NORMAL,
		.clk_id		= CLOCK_THREAD_CPUTIME_ID, // not supported
		.flags		= 0,
		.sec		= 0,
		.nsec		= 500000000, // 500msec
		.ret		= ENOTSUP, // RHEL4U1 + 2.6.18 returns EINVAL
		.err		= 0,
	},
	{ // case05
		.ttype		= SEND_SIGINT,
		.clk_id		= CLOCK_REALTIME,
		.flags		= 0,
		.sec		= 10,
		.nsec		= 0,
		.ret		= EINTR,
		.err		= 0,
	},
#if 0	// glibc generates SEGV error (RHEL4U1 + 2.6.18)
	{ // caseXX
		.ttype		= NULL_POINTER,
		.clk_id		= CLOCK_REALTIME,
		.flags		= 0,
		.sec		= 0,
		.nsec		= 500000000, // 500msec
		.ret		= EFAULT,
		.err		= 0,
	},
#endif
};

/*
 * chk_difftime()
 *   Return : OK(0), NG(-1)
 */
#define MAX_MSEC_DIFF	20
static int chk_difftime(struct timespec *bef, struct timespec *aft,
			time_t sec, long nsec)
{
	struct timespec t;
	time_t expect;
	time_t result;

	t.tv_sec = aft->tv_sec - bef->tv_sec;
	t.tv_nsec = aft->tv_nsec - bef->tv_nsec;
	if (t.tv_nsec < 0) {
		t.tv_sec -= 1;
		t.tv_nsec += 1000000000;
	}
	expect = (sec * 1000) + (nsec / 1000000);
	result = (t.tv_sec * 1000) + (t.tv_nsec / 1000000);
	PRINTF("check sleep time: (min:%ld) < %ld < (max:%ld) (msec)\n",
	       expect - MAX_MSEC_DIFF, result, expect + MAX_MSEC_DIFF);
	if (result < expect - MAX_MSEC_DIFF || result > expect + MAX_MSEC_DIFF)
		return -1;
	return 0;
}

/*
 * do_test()
 *
 *   Input  : TestCase Data
 *   Return : RESULT_OK(0), RESULT_NG(1)
 *
 */
static int do_test(struct test_case *tc)
{
	int sys_ret;
	int sys_errno;
	int result = RESULT_OK;
	struct timespec beftp, afttp, rq, rm;
	int rc, range_ok = 1, remain_ok = 1;
	pid_t pid = 0;

	/*
	 * Check before sleep time
	 */
	if (tc->ttype == SEND_SIGINT) {
		pid = create_sig_proc(500000, SIGINT);
		if (pid < 0)
			return 1;
	}

	/*
	 * Check before sleep time
	 */
	rc = clock_gettime(tc->clk_id, &beftp);
	if (rc < 0) {
		EPRINTF("clock_gettime failed.\n");
		result = 1;
		goto EXIT;
	}

	/*
	 * Execute system call
	 */
	rq.tv_sec = tc->sec;
	rq.tv_nsec = tc->nsec;
	// !!!CAUTION: 'clock_gettime' returns errno itself
	errno = 0;
	if (tc->ttype == NULL_POINTER)
		sys_ret = clock_nanosleep(tc->clk_id, tc->flags, NULL, &rm);
	else
		sys_ret = clock_nanosleep(tc->clk_id, tc->flags, &rq, &rm);
	sys_errno = errno;
	if (sys_ret != 0)
		goto TEST_END;

	/*
	 * Check after sleep time
	 */
	rc = clock_gettime(tc->clk_id, &afttp);
	if (rc < 0) {
		EPRINTF("clock_gettime failed.\n");
		result = 1;
		goto EXIT;
	}
	range_ok = chk_difftime(&beftp, &afttp, tc->sec, tc->nsec) == 0;

	/*
	 * Check remaining time
	 */
TEST_END:
	if (tc->ttype == NORMAL || tc->ttype == SEND_SIGINT) {
		PRINTF("remain time: %ld %ld\n", rm.tv_sec, rm.tv_nsec);
		if (tc->ttype == NORMAL)
			remain_ok = rm.tv_sec == 0 && rm.tv_nsec == 0;
		else
			remain_ok = rm.tv_sec != 0 || rm.tv_nsec != 0;
	}

	/*
	 * Check results
	 */
	result |= (sys_ret != tc->ret) || !range_ok || !remain_ok;
	if (!range_ok)
		PRINT_RESULT_EXTRA(0, tc->ret, tc->err, sys_ret, sys_errno,
				   "time range check", range_ok);
	else
		PRINT_RESULT_EXTRA(0, tc->ret, tc->err, sys_ret, sys_errno,
				   "remain time check", remain_ok);
EXIT:
	if (pid > 0) {
		int st;
		kill(pid, SIGTERM);
		wait(&st);
	}
	return result;
}


/*
 * sighandler()
 */
void sighandler(int sig)
{
	if (sig == SIGINT)
		return;
	// NOTREACHED
	return;
}


/*
 * usage()
 */
static void usage(const char *progname)
{
	EPRINTF("usage: %s [options]\n", progname);
	EPRINTF("This is a regression test program of %s system call.\n",
		SYSCALL_NAME);
	EPRINTF("options:\n");
	EPRINTF("    -d --debug           Show debug messages\n");
	EPRINTF("    -h --help            Show this message\n");

	RPRINTF("NG\n");
	exit(1);
}


/*
 * main()
 */
int main(int argc, char *argv[])
{
	int result = RESULT_OK;
	int c;
	int i;

	struct option long_options[] = {
		{ "debug", no_argument, 0, 'd' },
		{ "help",  no_argument, 0, 'h' },
		{ NULL, 0, NULL, 0 }
	};

	progname = strchr(argv[0], '/');
	progname = progname ? progname + 1 : argv[0];

	while ((c = getopt_long(argc, argv, "dh", long_options, NULL)) != -1) {
		switch (c) {
		case 'd':
			opt_debug = 1;
			break;

		default:
			usage(progname);
			// NOTREACHED
		}
	}

	if (argc != optind) {
		EPRINTF("Options are not match.\n");
		usage(progname);
		// NOTREACHED
	}

	/*
	 * Execute test
	 */
	signal(SIGINT, sighandler);
	for (i = 0; i < (int)(sizeof(tcase) / sizeof(tcase[0])); i++) {
		int ret;

		PRINTF("(case%02d) START\n", i);
		ret = do_test(&tcase[i]);
		PRINTF("(case%02d) END => %s\n", i, (ret == 0) ? "OK" : "NG");

		result |= ret;
	}

	/*
	 * Check results
	 */
	switch(result) {
	case RESULT_OK:
		RPRINTF("OK\n");
		break;

	default:
		RPRINTF("NG\n");
		break;
	}

	return 0;
}

