/*
 * Crackerjack Project
 *
 * Copyright (C) 2007-2008, Hitachi, Ltd.
 * Author(s): Takahiro Yasui <takahiro.yasui.mp@hitachi.com>,
 *            Yumiko Sugita <yumiko.sugita.yf@hitachi.com>,
 *            Satoshi Fujiwara <sa-fuji@sdl.hitachi.co.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * $Id:$
 *
 */

#include <sys/syscall.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <dirent.h>
#include <getopt.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include <libgen.h>
#include <limits.h>
#include <linux/inotify.h>

#include "../../common.j.h/include_j_h.h"

/*
 * Macros
 */
#define SYSCALL_NAME	"inotify_rm_watch"

#ifndef __NR_inotify_init
#  define __NR_inotify_init		291
#  define __NR_inotify_add_watch	292
#  define __NR_inotify_rm_watch		293
#endif


/*
 * Global variables
 */
static int opt_debug;
static char *progname;
static char *progdir;


enum test_type {
	NORMAL,
	FD_NONE,
	FD_NOT_EXIST,
	FD_FILE,
	INVALID_WD,
};


/*
 * Data Structure
 */
struct test_case {
	int ttype;
	int ret;
	int err;
};


/* Test cases
 *
 *   test status of errors on man page
 *
 *   EBADF		v (not a valid descriptor)
 *   EINVAL		v (invalid event-mask or fd)
 */
static struct test_case tcase[] = {
	{ // case00
		.ttype		= NORMAL,
		.ret		= 0,
		.err		= 0,
	},
	{ // case01
		.ttype		= FD_NONE,
		.ret		= -1,
		.err		= EBADF,
	},
	{ // case02
		.ttype		= FD_NOT_EXIST,
		.ret		= -1,
		.err		= EBADF,
	},
	{ // case03
		.ttype		= FD_FILE,
		.ret		= -1,
		.err		= EINVAL,
	},
	{ // case04
		.ttype		= INVALID_WD,
		.ret		= -1,
		.err		= EINVAL,
	},
};


static void print_inotify_event(char *p, int len)
{
	struct inotify_event *ev;
	int rec_len;

	if (len <= 0 || !opt_debug)
		return;
	PRINTF("================\n");
	while (len >= sizeof(struct inotify_event)) {
		ev = (struct inotify_event*)p;
		PRINTF("- - - - - - - - \n");
		PRINTF("wd:\t%d\n", ev->wd);
		PRINTF("mask:\t0x%08x\n", ev->mask);
		PRINTF("cookie:\t%d\n", ev->cookie);
		PRINTF("name:\t>%.*s<\n", ev->len, ev->name);
		rec_len = sizeof(struct inotify_event) + ev->len;
		p += rec_len;
		len -= rec_len;
	}
	PRINTF("\n");
}


#define ev_ok(ev, _wd, _mask, _cookie, _len, _name)		\
	(ev->wd == (_wd) && ev->mask == (_mask) &&		\
	 ev->cookie == (_cookie) && ev->len >= (_len) &&	\
	 strcmp(ev->name, (_name)) == 0)

#define TEST_FILE1	"test.file"
#define TEST_FILE2	"test.file2"

/*
 * do_test()
 *
 *   Input  : TestCase Data
 *   Return : RESULT_OK(0), RESULT_NG(1)
 *
 */
static int do_test(struct test_case *tc)
{
	int sys_ret;
	int sys_errno;
	int result = RESULT_OK;
	char fp1[PATH_MAX], buf[4096];
	int fd = -1, wd = -1, fd1 = -1;
	int rc, cmp_ok = 1;
	struct inotify_event *ev;
	int rec_len, n;
	char *p;
	char *path;
	unsigned long mask;

	switch (tc->ttype) {
	case FD_NOT_EXIST:
		fd = INT_MAX - 1;
		/* fallthrough */
	case FD_NONE:
		break;
	case FD_FILE:
		fd = open("/", O_RDONLY);
		if (fd < 0) {
			EPRINTF("can't open \"/\".\n");
			return 1;
		}
		break;
	default:
		fd = syscall(__NR_inotify_init);
		if (fd < 0) {
			EPRINTF("inotify_init failed.\n");
			return 1;
		}
		break;
	}

	path = progdir;
	mask = IN_CREATE | IN_OPEN | IN_MODIFY | IN_CLOSE_WRITE
		| IN_MOVED_FROM | IN_MOVED_TO | IN_DELETE;

	wd = syscall(__NR_inotify_add_watch, fd, path, mask);
	if (wd < 0 && !(tc->ttype == FD_NOT_EXIST || tc->ttype == FD_NONE ||
			tc->ttype == FD_FILE)) {
		EPRINTF("inotify_add_watch failed.\n");
		result = 1;
		goto EXIT;
	}

	/*
	 * Execute system call
	 */
	errno = 0;
	if (tc->ttype == INVALID_WD)
		sys_ret = syscall(__NR_inotify_rm_watch, fd, -1);
	else
		sys_ret = syscall(__NR_inotify_rm_watch, fd, wd);
	sys_errno = errno;
	if (sys_ret < 0)
		goto TEST_END;

	/*
	 * Check inotify
	 */
	// create file
	sprintf(fp1, "%s/%s", progdir, TEST_FILE1);
	fd1 = open(fp1, O_CREAT|O_EXCL|O_RDWR, S_IRUSR|S_IWUSR);
	if (fd1 < 0) {
		EPRINTF("open failed.\n");
		result = 1;
		goto EXIT;
	}
	close(fd1);

	// delete file
	rc = unlink(fp1);
	if (rc < 0) {
		EPRINTF("unlink failed.\n");
		result = 1;
		goto EXIT;
	}

	// get inotify events
	do {
		rc = read(fd, buf, 4096);
	} while (rc < 0 && errno == EAGAIN);
	if (rc < 0) {
		EPRINTF("read failed.\n");
		result = 1;
		goto EXIT;
	}

	// check inotify events
	print_inotify_event(buf, rc);
	p = buf;
	for (n = 0; rc >= sizeof(struct inotify_event); n++) {
		ev = (struct inotify_event*)p;
		switch (n) {
		case 0:
			if (!ev_ok(ev, wd, IN_IGNORED, 0, 0, ""))
				cmp_ok = 0;
			break;
		default:
			cmp_ok = 0;
			break;
		}
		if (opt_debug)
			PRINTF("%d cmk_ok: %d\n", n, cmp_ok);
		rec_len = sizeof(struct inotify_event) + ev->len;
		p += rec_len;
		rc -= rec_len;
	}
	if (rc != 0)
		cmp_ok = 0;

	/*
	 * Check results
	 */
TEST_END:
	result |= (sys_errno != tc->err) || !cmp_ok;
	PRINT_RESULT_CMP(sys_ret >= 0, tc->ret, tc->err, sys_ret, sys_errno,
			 cmp_ok);
EXIT:
	if (fd >= 0)
		close(fd);
	if (wd >= 0)
		close(wd);

	return result;
}


/*
 * usage()
 */
static void usage(const char *progname)
{
	EPRINTF("usage: %s [options]\n", progname);
	EPRINTF("This is a regression test program of %s system call.\n",
		SYSCALL_NAME);
	EPRINTF("options:\n");
	EPRINTF("    -d --debug           Show debug messages\n");
	EPRINTF("    -h --help            Show this message\n");

	RPRINTF("NG\n");
	exit(1);
}


/*
 * main()
 */
int main(int argc, char *argv[])
{
	int result = RESULT_OK;
	int c;
	int i;

	struct option long_options[] = {
		{ "debug", no_argument, 0, 'd' },
		{ "help",  no_argument, 0, 'h' },
		{ NULL, 0, NULL, 0 }
	};

	progname = strchr(argv[0], '/');
	progname = progname ? progname + 1 : argv[0];

	progdir = strdup(argv[0]);
	progdir = dirname(progdir);

	while ((c = getopt_long(argc, argv, "dh", long_options, NULL)) != -1) {
		switch (c) {
		case 'd':
			opt_debug = 1;
			break;

		default:
			usage(progname);
			// NOTREACHED
		}
	}

	if (argc != optind) {
		EPRINTF("Options are not match.\n");
		usage(progname);
		// NOTREACHED
	}

	/*
	 * Execute test
	 */
	for (i = 0; i < (int)(sizeof(tcase) / sizeof(tcase[0])); i++) {
		int ret;

		PRINTF("(case%02d) START\n", i);
		ret = do_test(&tcase[i]);
		PRINTF("(case%02d) END => %s\n", i, (ret == 0) ? "OK" : "NG");

		result |= ret;
	}

	/*
	 * Check results
	 */
	switch(result) {
	case RESULT_OK:
		RPRINTF("OK\n");
		break;

	default:
		RPRINTF("NG\n");
		break;
	}

	return 0;
}

