/*
 * Crackerjack Project
 *
 * Copyright (C) 2007-2008, Hitachi, Ltd.
 * Author(s): Takahiro Yasui <takahiro.yasui.mp@hitachi.com>,
 *            Yumiko Sugita <yumiko.sugita.yf@hitachi.com>,
 *            Satoshi Fujiwara <sa-fuji@sdl.hitachi.co.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * $Id:$
 *
 */

#include <sys/syscall.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <unistd.h>
#include <getopt.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <stdio.h>

#include "../../common.j.h/include_j_h.h"

/*
 * Macros
 */
#define SYSCALL_NAME	"munmap"
#define INVALP		((void *)-1)


/*
 * Global variables
 */
static int opt_debug;
static char *progname;


/*
 * Data Structure
 */
struct test_case {
	int ttype;
	size_t length;
	int ret;
	int err;
};

enum test_type {
	NORMAL,
	INVALID_START,
	NOT_ALIGNED_START,
	INVALID_LENGTH,
	UNMAP_PARTIAL,
};


/* Test cases
 *
 *   test status of errors on man page
 *
 *   EINVAL		v (start, length or offset are too large or not aligned)
 */
static struct test_case tcase[] = {
	{ // case00
		.ttype		= NORMAL,
		.length		= 4096, // 1 page
		.ret		= 0,
		.err		= 0,
	},
	{ // case01
		.ttype		= NORMAL,
		.length		= (4096 * 3), // 3 pages
		.ret		= 0,
		.err		= 0,
	},
	{ // case02
		.ttype		= NORMAL,
		.length		= 1024,	// small size
		.ret		= 0,
		.err		= 0,
	},
	{ // case03
		.ttype		= UNMAP_PARTIAL,
		.length		= (4096 * 3), // 3 pages
		.ret		= 0,
		.err		= 0,
	},
	{ // case03
		.ttype		= INVALID_START,
		.length		= 4096,
		.ret		= -1,
		.err		= EINVAL,
	},
	{ // case04
		.ttype		= NOT_ALIGNED_START,
		.length		= 4096,
		.ret		= -1,
		.err		= EINVAL,
	},
	{ // case05
		.ttype		= INVALID_LENGTH,
		.length		= 4096,
		.ret		= -1,
		.err		= EINVAL,
	},
};

/*
 * do_test()
 *
 *   Input  : TestCase Data
 *   Return : RESULT_OK(0), RESULT_NG(1)
 *
 */
static int do_test(struct test_case *tc)
{
	void *p;
	char *fname = "/etc/passwd";
	int sys_ret;
	int sys_errno;
	int result = RESULT_OK;
	int fd = -1;
	size_t length = tc->length;

	/*
	 * Prepare the file descriptor.
	 */
	fd = open(fname, O_RDONLY);
	if (fd < 0) {
		EPRINTF("open(\"%s\") failed. (ret)=%d errno=%d (%s)\n",
			fname, fd, errno, strerror(errno));
		result |= RESULT_NG;
		goto TEST_EXIT;
	}

	/*
	 * mapping memory
	 */
	p = mmap(NULL, length, PROT_READ, MAP_PRIVATE, fd, 0);
	if (p == MAP_FAILED) {
		EPRINTF("mmap failed. (ret)=%p errno=%d (%s)\n",
			p, errno, strerror(errno));
		result |= RESULT_NG;
		goto TEST_EXIT;
	}

	switch (tc->ttype) {
	case INVALID_START:
		p = INVALP;
		break;
	case NOT_ALIGNED_START:
		p = (char*)p + 1024;
		break;
	case INVALID_LENGTH:
		length = -1;
		break;
	case UNMAP_PARTIAL:
		if (tc->length > 4096) {
			length -= 4096;
			p = (char*)p + 4096;
		}
		break;
	}

	/*
	 * Execute system call
	 */
	errno = 0;
	sys_ret = munmap(p, length);
	sys_errno = errno;

	/*
	 * Check results
	 */
	result |= (sys_errno != tc->err);
	PRINT_RESULT(sys_ret >= 0, tc->ret, tc->err, sys_ret, sys_errno);

TEST_EXIT:
	if (fd >= 0)
		close(fd);
	return result;
}


/*
 * usage()
 */
static void usage(const char *progname)
{
	EPRINTF("usage: %s [options]\n", progname);
	EPRINTF("This is a regression test program of %s system call.\n",
		SYSCALL_NAME);
	EPRINTF("options:\n");
	EPRINTF("    -d --debug           Show debug messages\n");
	EPRINTF("    -h --help            Show this message\n");

	RPRINTF("NG\n");
	exit(1);
}


/*
 * main()
 */
int main(int argc, char *argv[])
{
	int result = RESULT_OK;
	int c;
	int i;

	struct option long_options[] = {
		{ "debug", no_argument, 0, 'd' },
		{ "help",  no_argument, 0, 'h' },
		{ NULL, 0, NULL, 0 }
	};

	progname = strchr(argv[0], '/');
	progname = progname ? progname + 1 : argv[0];

	while ((c = getopt_long(argc, argv, "dh", long_options, NULL)) != -1) {
		switch (c) {
		case 'd':
			opt_debug = 1;
			break;

		default:
			usage(progname);
			// NOTREACHED
		}
	}

	if (argc != optind) {
		EPRINTF("Options are not match.\n");
		usage(progname);
		// NOTREACHED
	}

	/*
	 * Execute test
	 */
	for (i = 0; i < (int)(sizeof(tcase) / sizeof(tcase[0])); i++) {
		int ret;

		PRINTF("(case%02d) START\n", i);
		ret = do_test(&tcase[i]);
		PRINTF("(case%02d) END => %s\n", i, (ret == 0) ? "OK" : "NG");

		result |= ret;
	}

	/*
	 * Check results
	 */
	switch(result) {
	case RESULT_OK:
		RPRINTF("OK\n");
		break;

	default:
		RPRINTF("NG\n");
		break;
	}

	return 0;
}

