/*
 * Crackerjack Project
 *
 * Copyright (C) 2007-2008, Hitachi, Ltd.
 * Author(s): Takahiro Yasui <takahiro.yasui.mp@hitachi.com>,
 *            Yumiko Sugita <yumiko.sugita.yf@hitachi.com>,
 *            Satoshi Fujiwara <sa-fuji@sdl.hitachi.co.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * $Id:$
 *
 */

#include <sys/syscall.h>
#include <sys/types.h>
#include <sys/select.h>
#include <sys/wait.h>
#include <getopt.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <libgen.h>
#include <limits.h>
#include <signal.h>
#include "asm/poll.h"

#include "../../common.j.h/include_j_h.h"

/*
 * Macros
 */
#define SYSCALL_NAME	"ppoll"

#ifndef __NR_ppoll
#  define __NR_ppoll	309
#endif

#ifndef POLLRDHUP
#  define POLLRDHUP	0x2000
#endif


/*
 * Global variables
 */
static int opt_debug;
static char *progname;
static char *progdir;

enum test_type {
	NORMAL,
	MASK_SIGNAL,
	TIMEOUT,
	FD_ALREADY_CLOSED,
	SEND_SIGINT,
	INVALID_NFDS,
	INVALID_FDS,
	MINUS_NSEC,
	TOO_LARGE_NSEC,
};


/*
 * Data Structure
 */
struct test_case {
	int ttype;
	short expect_revents;
	int ret;
	int err;
};


/* Test cases
 *
 *   test status of errors on man page
 *
 *   EBADF		can't check because EBADF never happen even though
 *   			fd was invalid.	In this case, information of invalid
 *   			fd is set in revents
 *   EFAULT		v ('fds' array in the invalid address space)
 *   EINTR		v (a non blocked signal was caught)
 *   EINVAL		v ('nfds' is over the 'RLIMIT_NOFILE' value)
 *   ENOMEM		can't check because it's difficult to create no-memory
 */
static struct test_case tcase[] = {
	{ // case00
		.ttype		= NORMAL,
		.expect_revents	= POLLOUT,
		.ret		= 0,
		.err		= 0,
	},
	{ // case01
		.ttype		= MASK_SIGNAL,
		.expect_revents	= 0, // don't care
		.ret		= 0,
		.err		= 0,
	},
	{ // case02
		.ttype		= TIMEOUT,
		.expect_revents	= 0, // don't care
		.ret		= 0,
		.err		= 0,
	},
	{ // case03
		.ttype		= FD_ALREADY_CLOSED,
		.expect_revents	= POLLNVAL,
		.ret		= 0,
		.err		= 0,
	},
	{ // case04
		.ttype		= SEND_SIGINT,
		.ret		= -1,
		.err		= EINTR,
	},
	{ // case05
		.ttype		= INVALID_NFDS,
		.ret		= -1,
		.err		= EINVAL,
	},
	{ // case06
		.ttype		= INVALID_FDS,
		.ret		= -1,
		.err		= EFAULT,
	},
#if 0
	{ // case07
		.ttype		= MINUS_NSEC,
		.ret		= -1,
		.err		= EINVAL, // RHEL4U1 + 2.6.18 returns SUCCESS
	},
	{ // case08
		.ttype		= TOO_LARGE_NSEC,
		.ret		= -1,
		.err		= EINVAL, // RHEL4U1 + 2.6.18 returns SUCCESS
	},
#endif
};

#define NUM_TEST_FDS	1
/*
 * do_test()
 *
 *   Input  : TestCase Data
 *   Return : RESULT_OK(0), RESULT_NG(1)
 *
 */
static int do_test(struct test_case *tc)
{
	int sys_ret;
	int sys_errno;
	int result = RESULT_OK;
	int fd = -1, cmp_ok = 1;
	char fpath[PATH_MAX];
	struct pollfd *p_fds, fds[NUM_TEST_FDS];
	unsigned int nfds = NUM_TEST_FDS;
	struct timespec *p_ts, ts;
	sigset_t *p_sigmask, sigmask;
	size_t sigsetsize = 0;
	pid_t pid = 0;

	fd = setup_file(progdir, "test.file", fpath);
	if (fd < 0)
		return 1;
	fds[0].fd = fd;
	fds[0].events = POLLIN | POLLPRI | POLLOUT | POLLRDHUP;
	fds[0].revents = 0;
	p_fds = fds;
	p_ts = NULL;
	p_sigmask = NULL;

	switch (tc->ttype) {
	case TIMEOUT:
		nfds = 0;
		ts.tv_sec = 0;
		ts.tv_nsec = 50000000;	// 50msec
		p_ts = &ts;
		break;
	case FD_ALREADY_CLOSED:
		close(fd);
		fd = -1;
		cleanup_file(fpath);
		break;
	case MASK_SIGNAL:
		sigemptyset(&sigmask);
		sigaddset(&sigmask, SIGINT);
		p_sigmask = &sigmask;
		//sigsetsize = sizeof(sigmask);
		sigsetsize = 8;
		nfds = 0;
		ts.tv_sec = 0;
		ts.tv_nsec = 300000000;	// 300msec => need to be enough for
					//   waiting the signal
		p_ts = &ts;
		// fallthrough
	case SEND_SIGINT:
		nfds = 0;
		pid = create_sig_proc(100000, SIGINT); // 100msec
		if (pid < 0)
			return 1;
		break;
	case INVALID_NFDS:
		//nfds = RLIMIT_NOFILE + 1; ==> RHEL4U1 + 2.6.18 returns SUCCESS
		nfds = -1;
		break;
	case INVALID_FDS:
		p_fds = (void*)0xc0000000;
		break;
	case MINUS_NSEC:
		ts.tv_sec = 0;
		ts.tv_nsec = -1;
		p_ts = &ts;
		break;
	case TOO_LARGE_NSEC:
		ts.tv_sec = 0;
		ts.tv_nsec = 1000000000;
		p_ts = &ts;
		break;
	}

	/*
	 * Execute system call
	 */
	errno = 0;
	sys_ret = syscall(__NR_ppoll, p_fds, nfds, p_ts, p_sigmask, sigsetsize);
	sys_errno = errno;
	if (sys_ret <= 0 || tc->ret < 0)
		goto TEST_END;

	cmp_ok = fds[0].revents == tc->expect_revents;
	if (opt_debug) {
		EPRINTF("EXPECT: revents=0x%04x\n", tc->expect_revents);
		EPRINTF("RESULT: revents=0x%04x\n", fds[0].revents);
	}

TEST_END:
	/*
	 * Check results
	 */
	result |= (sys_errno != tc->err) || !cmp_ok;
	PRINT_RESULT_CMP(sys_ret >= 0, tc->ret, tc->err, sys_ret, sys_errno,
			 cmp_ok);

	if (fd >= 0)
		cleanup_file(fpath);
	if (pid > 0) {
		int st;
		kill(pid, SIGTERM);
		wait(&st);
	}
	return result;
}


/*
 * sighandler()
 */
void sighandler(int sig)
{
	if (sig == SIGINT)
		return;
	// NOTREACHED
	return;
}


/*
 * usage()
 */
static void usage(const char *progname)
{
	EPRINTF("usage: %s [options]\n", progname);
	EPRINTF("This is a regression test program of %s system call.\n",
		SYSCALL_NAME);
	EPRINTF("options:\n");
	EPRINTF("    -d --debug           Show debug messages\n");
	EPRINTF("    -h --help            Show this message\n");

	RPRINTF("NG\n");
	exit(1);
}


/*
 * main()
 */
int main(int argc, char *argv[])
{
	int result = RESULT_OK;
	int c;
	int i;

	struct option long_options[] = {
		{ "debug", no_argument, 0, 'd' },
		{ "help",  no_argument, 0, 'h' },
		{ NULL, 0, NULL, 0 }
	};

	progname = strchr(argv[0], '/');
	progname = progname ? progname + 1 : argv[0];

	progdir = strdup(argv[0]);
	progdir = dirname(progdir);

	while ((c = getopt_long(argc, argv, "dh", long_options, NULL)) != -1) {
		switch (c) {
		case 'd':
			opt_debug = 1;
			break;

		default:
			usage(progname);
			// NOTREACHED
		}
	}

	if (argc != optind) {
		EPRINTF("Options are not match.\n");
		usage(progname);
		// NOTREACHED
	}

	/*
	 * Execute test
	 */
	signal(SIGINT, sighandler);
	for (i = 0; i < (int)(sizeof(tcase) / sizeof(tcase[0])); i++) {
		int ret;

		PRINTF("(case%02d) START\n", i);
		ret = do_test(&tcase[i]);
		PRINTF("(case%02d) END => %s\n", i, (ret == 0) ? "OK" : "NG");

		result |= ret;
	}

	/*
	 * Check results
	 */
	switch(result) {
	case RESULT_OK:
		RPRINTF("OK\n");
		break;

	default:
		RPRINTF("NG\n");
		break;
	}

	return 0;
}

