/*
 * Crackerjack Project
 *
 * Copyright (C) 2007-2008, Hitachi, Ltd.
 * Author(s): Takahiro Yasui <takahiro.yasui.mp@hitachi.com>,
 *            Yumiko Sugita <yumiko.sugita.yf@hitachi.com>,
 *            Satoshi Fujiwara <sa-fuji@sdl.hitachi.co.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * $Id:$
 *
 */

#include <sys/syscall.h>
#include <sys/types.h>
#include <sys/select.h>
#include <sys/wait.h>
#include <getopt.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <libgen.h>
#include <limits.h>
#include <signal.h>

#include "../../common.j.h/include_j_h.h"

/*
 * Macros
 */
#define SYSCALL_NAME	"select"


/*
 * Global variables
 */
static int opt_debug;
static char *progname;
static char *progdir;

enum test_type {
	NORMAL,
	TIMEOUT,
	FD_ALREADY_CLOSED,
	SEND_SIGINT,
	INVALID_NFDS,
	MINUS_NSEC,
	TOO_LARGE_NSEC,
};


/*
 * Data Structure
 */
struct test_case {
	int ttype;
	int expect_read, expect_write, expect_except;
	int ret;
	int err;
};

struct sig {
	sigset_t *sigset;
	size_t sigsetsize;
};


/* Test cases
 *
 *   test status of errors on man page
 *
 *   EBADF		v (not a valid descriptor)
 *   EINTR		v (a non blocked signal was caught)
 *   EINVAL		v ('n' is negative or the value contained within
 *   			   'timeout' is invalid)
 *   ENOMEM		can't check because it's difficult to create no-memory
 */
static struct test_case tcase[] = {
	{ // case00
		.ttype		= NORMAL,
		.expect_read	= 1,
		.expect_write	= 1,
		.expect_except	= 0,
		.ret		= 0,
		.err		= 0,
	},
	{ // case02
		.ttype		= TIMEOUT,
		.expect_read	= 0, // don't care
		.expect_write	= 0, // don't care
		.expect_except	= 0, // don't care
		.ret		= 0,
		.err		= 0,
	},
	{ // case03
		.ttype		= FD_ALREADY_CLOSED,
		.ret		= -1,
		.err		= EBADF,
	},
	{ // case04
		.ttype		= SEND_SIGINT,
		.ret		= -1,
		.err		= EINTR,
	},
	{ // case05
		.ttype		= INVALID_NFDS,
		.ret		= -1,
		.err		= EINVAL,
	},
	{ // case06
		.ttype		= MINUS_NSEC,
		.ret		= -1,
		.err		= EINVAL,
	},
	{ // case07
		.ttype		= TOO_LARGE_NSEC,
		.ret		= -1,
		.err		= EINVAL, // RHEL4U1 + 2.6.18 returns SUCCESS
	},
};

/*
 * do_test()
 *
 *   Input  : TestCase Data
 *   Return : RESULT_OK(0), RESULT_NG(1)
 *
 */
static int do_test(struct test_case *tc)
{
	int sys_ret;
	int sys_errno;
	int result = RESULT_OK;
	int n = -1, fd = -1, cmp_ok = 1;
	char fpath[PATH_MAX];
	fd_set readfds, writefds, exceptfds;
	struct timeval *p_tv, tv;
	pid_t pid = 0;

	fd = setup_file(progdir, "test.file", fpath);
	if (fd < 0)
		return 1;
	if (n < fd + 1)
		n = fd + 1;
	FD_ZERO(&readfds);
	FD_ZERO(&writefds);
	FD_ZERO(&exceptfds);
	FD_SET(fd, &readfds);
	FD_SET(fd, &writefds);
	FD_SET(fd, &exceptfds);
	p_tv = NULL;

	switch (tc->ttype) {
	case TIMEOUT:
		n = 0;
		tv.tv_sec = 0;
		tv.tv_usec = 50000;	// 50msec
		p_tv = &tv;
		break;
	case FD_ALREADY_CLOSED:
		close(fd);
		fd = -1;
		cleanup_file(fpath);
		break;
	case SEND_SIGINT:
		n = 0;
		pid = create_sig_proc(100000, SIGINT); // 100msec
		if (pid < 0)
			return 1;
		break;
	case INVALID_NFDS:
		n = -1;
		break;
	case MINUS_NSEC:
		tv.tv_sec = 0;
		tv.tv_usec = -1;
		p_tv = &tv;
		break;
	case TOO_LARGE_NSEC:
		tv.tv_sec = 0;
		tv.tv_usec = 1000000;
		p_tv = &tv;
		break;
	}

	/*
	 * Execute system call
	 */
	errno = 0;
	sys_ret = select(n, &readfds, &writefds, &exceptfds, p_tv);
	sys_errno = errno;
	if (sys_ret <= 0 || tc->ret < 0)
		goto TEST_END;

	cmp_ok = FD_ISSET(fd, &readfds) == tc->expect_read &&
		 FD_ISSET(fd, &writefds) == tc->expect_write &&
		 FD_ISSET(fd, &exceptfds) == tc->expect_except;
	if (opt_debug) {
		EPRINTF("Each fd(expect) => read:%d write:%d except:%d\n",
			tc->expect_read,
			tc->expect_write,
			tc->expect_except);
		EPRINTF("Each fd(result) => read:%d write:%d except:%d\n",
			FD_ISSET(fd, &readfds),
			FD_ISSET(fd, &writefds),
			FD_ISSET(fd, &exceptfds));
	}

TEST_END:
	/*
	 * Check results
	 */
	result |= (sys_errno != tc->err) || !cmp_ok;
	PRINT_RESULT_CMP(sys_ret >= 0, tc->ret, tc->err, sys_ret, sys_errno,
			 cmp_ok);

	if (fd >= 0)
		cleanup_file(fpath);
	if (pid > 0) {
		int st;
		kill(pid, SIGTERM);
		wait(&st);
	}
	return result;
}


/*
 * sighandler()
 */
void sighandler(int sig)
{
	if (sig == SIGINT)
		return;
	// NOTREACHED
	return;
}


/*
 * usage()
 */
static void usage(const char *progname)
{
	EPRINTF("usage: %s [options]\n", progname);
	EPRINTF("This is a regression test program of %s system call.\n",
		SYSCALL_NAME);
	EPRINTF("options:\n");
	EPRINTF("    -d --debug           Show debug messages\n");
	EPRINTF("    -h --help            Show this message\n");

	RPRINTF("NG\n");
	exit(1);
}


/*
 * main()
 */
int main(int argc, char *argv[])
{
	int result = RESULT_OK;
	int c;
	int i;

	struct option long_options[] = {
		{ "debug", no_argument, 0, 'd' },
		{ "help",  no_argument, 0, 'h' },
		{ NULL, 0, NULL, 0 }
	};

	progname = strchr(argv[0], '/');
	progname = progname ? progname + 1 : argv[0];

	progdir = strdup(argv[0]);
	progdir = dirname(progdir);

	while ((c = getopt_long(argc, argv, "dh", long_options, NULL)) != -1) {
		switch (c) {
		case 'd':
			opt_debug = 1;
			break;

		default:
			usage(progname);
			// NOTREACHED
		}
	}

	if (argc != optind) {
		EPRINTF("Options are not match.\n");
		usage(progname);
		// NOTREACHED
	}

	/*
	 * Execute test
	 */
	signal(SIGINT, sighandler);
	for (i = 0; i < (int)(sizeof(tcase) / sizeof(tcase[0])); i++) {
		int ret;

		PRINTF("(case%02d) START\n", i);
		ret = do_test(&tcase[i]);
		PRINTF("(case%02d) END => %s\n", i, (ret == 0) ? "OK" : "NG");

		result |= ret;
	}

	/*
	 * Check results
	 */
	switch(result) {
	case RESULT_OK:
		RPRINTF("OK\n");
		break;

	default:
		RPRINTF("NG\n");
		break;
	}

	return 0;
}

