/*
 * Crackerjack Project
 *
 * Copyright (C) 2007-2008, Hitachi, Ltd.
 * Author(s): Takahiro Yasui <takahiro.yasui.mp@hitachi.com>,
 *            Yumiko Sugita <yumiko.sugita.yf@hitachi.com>,
 *            Satoshi Fujiwara <sa-fuji@sdl.hitachi.co.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * $Id:$
 *
 */
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <fcntl.h>
#include <unistd.h>
#include <getopt.h>
#include <string.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>

#include <asm/errno.h>
#include "../../include/socketcall.h"


/*
 * Macros
 */
#define SYSCALL_NAME	"sendto"

#define MBUF_LEN	64

/*
 * Global variables
 */
static int opt_debug;
static char *progname;
static int port;

// default is 'connected socket'
enum test_type {
	FD_NONE,
	FD_NOT_EXIST,
	FD_FILE,
	NULL_MSG,
	INVALID_ADDR_LEN,
	TOO_LARGE_MSG,
	NOT_CONNECTED,
	ALREADY_CONNECTED,
	NO_PEER_ADDR,
	MINUS_ADDR,
	PEER_ADDR_EXISTS,
	SERVER_DISCONNECT,
	SERVER_SEND_SIGINT,
	NORMAL,
};


/*
 * Data Structure
 */
struct test_case {
	int domain;
	int ttype;
	char *path;	// for PF_UNIX
	u_int32_t addr;	// for PF_INET
	int flag;
	int ret;
	int err;
};


/* Test cases
 *
 *   test status of errors on man page
 *
 *   EAGAIN/EWOULDBLOCK	can't check because send would not block
 *   EBADF		v (not a valid descriptor)
 *   ECONNRESET		can't check becasue shutdown by peer generates ENOTCONN
 *   EDESTADDRREQ	v (not connect-mode and no peer address is set)
 *   EFAULT		v (points to not process address space)
 *   EINTR		can't check because EINTR never happen even though
 *   			SIGINT was received
 *   EINVAL		v (invalid argument)
 *   EISCON		v (connect-mode socket was connected already)
 *   ENOBUFS		can't check because it's difficult to create no-memory
 *   ENOMEM		can't check because it's difficult to create no-memory
 *   ENOTCONN		v (not connected)
 *   ENOTSOCK		v (file descriptor)
 *   EOPNOTSUPP		v (invalid 'flags')
 *   EPIPE		v (local end has been shut down)
 */
static struct test_case tcase[] = {
	// -------- PF_UNIX --------
	{ // case00
		.domain	= PF_UNIX,
		.ttype	= FD_NONE,
		.path	= "",
		.ret	= -1,
		.err	= EBADF,
	},
	{ // case01
		.domain	= PF_UNIX,
		.ttype	= FD_NOT_EXIST,
		.path	= "",
		.ret	= -1,
		.err	= EBADF,
	},
	{ // case02
		.domain	= PF_UNIX,
		.ttype	= FD_FILE,
		.path	= "",
		.ret	= -1,
		.err	= ENOTSOCK,
	},
	{ // case03
		.domain	= PF_UNIX,
		.ttype	= NULL_MSG,
		.path	= "",
		.ret	= -1,
		.err	= EFAULT,
	},
	{ // case04
		.domain	= PF_UNIX,
		.ttype	= INVALID_ADDR_LEN,
		.path	= "",
		.ret	= -1,
		.err	= EINVAL,
	},
	{ // case05
		.domain	= PF_UNIX,
		.ttype	= NORMAL,
		.path	= "",
		.flag	= MSG_OOB,	// PF_UNIX domain is not support this.
		.ret	= -1,
		.err	= EOPNOTSUPP,
	},
#if 0
	{ // caseXX
		.domain	= PF_UNIX,
		.ttype	= TOO_LARGE_MSG,
		.path	= "",
		.ret	= -1,
		.err	= EMSGSIZE,	// RHEL4U1 + 2.6.18 returns EFAULT
	},
#endif
	{ // case06
		.domain	= PF_UNIX,
		.ttype	= NOT_CONNECTED,
		.path	= "",
		.ret	= -1,
		.err	= ENOTCONN,
	},
	{ // case07
		.domain	= PF_UNIX,
		.ttype	= ALREADY_CONNECTED,
		.path	= "",
		.ret	= -1,
		.err	= EISCONN,
	},
	{ // case08
		.domain	= PF_UNIX,
		.ttype	= NO_PEER_ADDR,
		.path	= "",
		.ret	= -1,
		.err	= EDESTADDRREQ,	// RHEL4U1 + 2.6.18 returns ENOTCONN
	},
	{ // case09
		.domain	= PF_UNIX,
		.ttype	= MINUS_ADDR,
		.path	= "",
		.ret	= -1,
		.err	= EFAULT,
	},
	{ // case10
		.domain	= PF_UNIX,
		.ttype	= PEER_ADDR_EXISTS,
		.path	= "",
		.ret	= -1,
		.err	= EOPNOTSUPP,
	},
	{ // case11
		.domain	= PF_UNIX,
		.ttype	= SERVER_DISCONNECT,
		.path	= "",
		.flag	= MSG_NOSIGNAL,
		.ret	= -1,
		.err	= EPIPE,
	},
#if 0
	{ // caseXX
		.domain	= PF_UNIX,
		.ttype	= SERVER_SEND_SIGINT,
		.path	= "",
		.ret	= -1,
		.err	= EINTR,
	},
#endif
	{ // case12
		.domain	= PF_UNIX,
		.ttype	= NORMAL,
		.path	= "",
		.ret	= 0,
		.err	= 0,
	},
	// -------- PF_INET --------
	{ // case13
		.domain	= PF_INET,
		.ttype	= FD_NONE,
		.ret	= -1,
		.err	= EBADF,
	},
	{ // case14
		.domain	= PF_INET,
		.ttype	= FD_NOT_EXIST,
		.ret	= -1,
		.err	= EBADF,
	},
	{ // case15
		.domain	= PF_INET,
		.ttype	= FD_FILE,
		.ret	= -1,
		.err	= ENOTSOCK,
	},
	{ // case16
		.domain	= PF_INET,
		.ttype	= NULL_MSG,
		.addr	= INADDR_ANY,
		.ret	= -1,
		.err	= EFAULT,
	},
	{ // case17
		.domain	= PF_INET,
		.ttype	= INVALID_ADDR_LEN,
		.addr	= INADDR_ANY,
		.ret	= -1,
		.err	= EINVAL,	// RHEL4U1 + 2.6.18 returns SUCCESS
	},
#if 0
	{ // caseXX
		.domain	= PF_INET,
		.ttype	= NORMAL,
		.addr	= INADDR_ANY,
		.flag	= MSG_OOB,	// PF_INET domain is support this.
		.ret	= 0,
		.err	= 0,
	},
#endif
#if 0
	{ // caseXX
		.domain	= PF_INET,
		.ttype	= TOO_LARGE_MSG,
		.addr	= INADDR_ANY,
		.ret	= -1,
		.err	= EMSGSIZE,
	},
#endif
	{ // case18
		.domain	= PF_INET,
		.ttype	= NOT_CONNECTED,
		.addr	= INADDR_ANY,
		.ret	= -1,
		.err	= EPIPE,
	},
	{ // case19
		.domain	= PF_INET,
		.ttype	= ALREADY_CONNECTED,
		.addr	= INADDR_ANY,
		.ret	= -1,
		.err	= EISCONN,	// RHEL4U1 + 2.6.18 returns SUCCESS
	},
	{ // case20
		.domain	= PF_INET,
		.ttype	= NO_PEER_ADDR,
		.addr	= INADDR_ANY,
		.ret	= -1,
		.err	= EDESTADDRREQ,
	},
	{ // case21
		.domain	= PF_INET,
		.ttype	= MINUS_ADDR,
		.addr	= INADDR_ANY,
		.ret	= -1,
		.err	= EFAULT,
	},
#if 0
	{ // caseXX
		.domain	= PF_INET,
		.ttype	= PEER_ADDR_EXISTS,
		.addr	= INADDR_ANY,
		.ret	= -1,
		.err	= EDESTADDRREQ,
	},
#endif
#if 0
	// In spite of the server discennected the socket, sendto was success,
	// and following recvmsg returns 0 (it means 'disconnected').
	{ // caseXX
		.domain	= PF_INET,
		.ttype	= SERVER_DISCONNECT,
		.addr	= INADDR_ANY,
		.flag	= MSG_NOSIGNAL,
		.ret	= -1,
		.err	= EPIPE,
	},
	{ // caseXX
		.domain	= PF_INET,
		.ttype	= SERVER_SEND_SIGINT,
		.addr	= INADDR_ANY,
		.ret	= -1,
		.err	= EINTR,
	},
#endif
	{ // case22
		.domain	= PF_INET,
		.ttype	= NORMAL,
		.addr	= INADDR_ANY,
		.ret	= 0,
		.err	= 0,
	},
};

/*
 * do_test()
 *
 *   Input  : TestCase Data
 *   Return : RESULT_OK(0), RESULT_NG(1)
 *
 */
static int do_test(struct test_case *tc)
{
	int fd = -1, rc, type;
	int sys_ret;
	int sys_errno;
	int result = RESULT_OK;
	struct sockaddr_in in_addr;
	struct sockaddr_un un_addr;
	struct sockaddr *addr;
	int i, len, st, cmp_ok;
	pid_t sv_pid = 0;
	char mbuf[MBUF_LEN], rmbuf[MBUF_LEN];
	int mlen, svtype;
	////int flval;

	type = (tc->ttype == NO_PEER_ADDR || tc->ttype == MINUS_ADDR) ?
		SOCK_DGRAM : SOCK_STREAM;
	//type = SOCK_STREAM;
	if (tc->domain == PF_UNIX) {
		un_addr.sun_family = AF_UNIX;
		strcpy(un_addr.sun_path, tc->path);
		len = sizeof(un_addr);
		addr = (struct sockaddr*)&un_addr;
	} else {
		if (port == 0)
			port = (getpid() % 1024) * MAX_SOCK_TESTCASES + 11000;
		else
			port++;
		in_addr.sin_family = AF_INET;
		in_addr.sin_port = htons(port);
		in_addr.sin_addr.s_addr = tc->addr;
		len = sizeof(in_addr);
		addr = (struct sockaddr*)&in_addr;
	}
	switch (tc->ttype) {
	case SERVER_DISCONNECT:
		svtype = SV_DISCONN_BY_ACCEPT;
		break;
	case SERVER_SEND_SIGINT:
		svtype = SV_SEND_SIGINT_BY_ACCEPT;
		break;
	default:
		svtype = SV_NORMAL;
		break;
	}
	sv_pid = create_server_proc(type, 0, addr, len, svtype);
	if (sv_pid < 0) {
		result = 1;
		goto EXIT;
	}

	switch (tc->ttype) {
	case FD_NOT_EXIST:
		fd = INT_MAX - 1;
		/* fallthrough */
	case FD_NONE:
		break;
	case FD_FILE:
		fd = open("/", O_RDONLY);
		if (fd < 0) {
			EPRINTF("can't open \"/\".\n");
			return 1;
		}
		break;
	default:
		// do socket
		fd = socket(tc->domain, type, 0);
		if (fd < 0) {
			EPRINTF("can't create socket.\n");
			result = 1;
			goto EXIT;
		}

		// do connect
		if (tc->ttype != NOT_CONNECTED &&
		    tc->ttype != NO_PEER_ADDR &&
		    tc->ttype != MINUS_ADDR &&
		    tc->ttype != PEER_ADDR_EXISTS) {
			rc = connect(fd, addr, len);
			if (rc < 0) {
				EPRINTF("connect failed.\n");
				result = 1;
				goto EXIT;
			}
		}
		////flval = fcntl(fd, F_GETFL, 0);
		////fcntl(fd, F_SETFL, flval | O_NONBLOCK);
		break;
	}

	// do sendto
	mlen = 16;
	for (i = 0; i < 16; i++)
		mbuf[0] = i;

	switch (tc->ttype) {
	case TOO_LARGE_MSG:
		mlen = 0xffffffff;
		break;
	case SERVER_DISCONNECT:
	case SERVER_SEND_SIGINT:
		usleep(100000);	// timing for server action
		break;
	}

	errno = 0;
	switch (tc->ttype) {
	case INVALID_ADDR_LEN:
		len += 50;
		/* fallthrough */
	case PEER_ADDR_EXISTS:
	case ALREADY_CONNECTED:
		sys_ret = sendto(fd, mbuf, mlen, tc->flag, addr, len);
		break;
	case MINUS_ADDR:
		sys_ret = sendto(fd, mbuf, mlen, tc->flag,
				 (struct sockaddr*)-1, len);
		break;
	case NULL_MSG:
		sys_ret = sendto(fd, NULL, mlen, tc->flag, NULL, 0);
		break;
	default:
		sys_ret = sendto(fd, mbuf, mlen, tc->flag, NULL, 0);
		break;
	}
	////printf("CHKIT %d %d\n", sys_ret, errno);
	sys_errno = errno;

	// check whether sendto data is really sended one
	cmp_ok = 1;
	if (sys_ret > 0) {
		do {
			rc = recv(fd, rmbuf, mlen, tc->flag);
		} while (rc < 0 && errno == EAGAIN);
		if (rc <= 0) {
			EPRINTF("recv failed.\n");
			result = 1;
			goto EXIT;
		}
		if (rc != 16)
			cmp_ok = 0;
		for (i = 0; i < 16; i++)
			if (mbuf[i] != rmbuf[i]) {
				cmp_ok = 0;
				break;
			}
	}

	/*
	 * Check results
	 */
	result |= (sys_errno != tc->err) || !cmp_ok;
	PRINT_RESULT_CMP(sys_ret >= 0, tc->ret, tc->err, sys_ret, sys_errno,
			 cmp_ok);
EXIT:
	if (fd >= 0)
		close(fd);
	if (sv_pid > 0) {
		kill(sv_pid, SIGTERM);
		wait(&st);
	}
	return result;
}


/*
 * sighandler()
 */
void sighandler(int sig)
{
	if (sig == SIGINT || sig == SIGPIPE)
		return;
	// NOTREACHED
	return;
}


/*
 * usage()
 */
static void usage(const char *progname)
{
	EPRINTF("usage: %s [options]\n", progname);
	EPRINTF("This is a regression test program of %s system call.\n",
		SYSCALL_NAME);
	EPRINTF("options:\n");
	EPRINTF("    -d --debug           Show debug messages\n");
	EPRINTF("    -h --help            Show this message\n");

	RPRINTF("NG\n");
	exit(1);
}


/*
 * main()
 */
int main(int argc, char *argv[])
{
	int result = RESULT_OK;
	int c;
	int i;

	struct option long_options[] = {
		{ "debug", no_argument, 0, 'd' },
		{ "help",  no_argument, 0, 'h' },
		{ NULL, 0, NULL, 0 }
	};

	progname = strchr(argv[0], '/');
	progname = progname ? progname + 1 : argv[0];

	while ((c = getopt_long(argc, argv, "dh", long_options, NULL)) != -1) {
		switch (c) {
		case 'd':
			opt_debug = 1;
			break;

		default:
			usage(progname);
			// NOTREACHED
		}
	}

	if (argc != optind) {
		EPRINTF("Options are not match.\n");
		usage(progname);
		// NOTREACHED
	}

	/*
	 * Execute test
	 */
	signal(SIGINT, sighandler);
	signal(SIGPIPE, sighandler);
	for (i = 0; i < (int)(sizeof(tcase) / sizeof(tcase[0])); i++) {
		int ret;

		PRINTF("(case%02d) START\n", i);
		ret = do_test(&tcase[i]);
		PRINTF("(case%02d) END => %s\n", i, (ret == 0) ? "OK" : "NG");

		result |= ret;
	}

	/*
	 * Check results
	 */
	switch(result) {
	case RESULT_OK:
		RPRINTF("OK\n");
		break;

	default:
		RPRINTF("NG\n");
		break;
	}

	return 0;
}

