/*
 * Crackerjack Project
 *
 * Copyright (C) 2007-2008, Hitachi, Ltd.
 * Author(s): Takahiro Yasui <takahiro.yasui.mp@hitachi.com>,
 *            Yumiko Sugita <yumiko.sugita.yf@hitachi.com>,
 *            Satoshi Fujiwara <sa-fuji@sdl.hitachi.co.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * $Id:$
 *
 */

#include <sys/syscall.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <getopt.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <limits.h>

#include "../../common.j.h/include_j_h.h"

/*
 * Macros
 */
#define SYSCALL_NAME	"vmsplice"

#ifndef __NR_vmsplice
#  define __NR_vmsplice		316
#endif

#ifndef SPLICE_F_MOVE
#  define SPLICE_F_MOVE		(0x01)
#endif
#ifndef SPLICE_F_NONBLOCK
#  define SPLICE_F_NONBLOCK	(0x02)
#endif
#ifndef SPLICE_F_MORE
#  define SPLICE_F_MORE		(0x04)
#endif
#ifndef SPLICE_F_GIFT
#  define SPLICE_F_GIFT		(0x08)
#endif
#ifndef IOV_MAX
#  define IOV_MAX		1024
#endif

/*
 * Global variables
 */
static int opt_debug;
static char *progname;


/*
 * Data Structure
 */
struct test_case {
	int ttype;
	unsigned long nr_segs;
	unsigned int flags;
	int ret;
	int err;
};


enum test_type {
	NORMAL,
	FD_NONE,
	FD_NOT_EXIST,
	FD_FILE,
};


/* Test cases
 *
 *   test status of errors on man page
 *
 *   EBADF		v (not a valid descriptor)
 *   EINVAL		v (invalid nr_segs or call with SPLICE_F_GIFT but
 *                         memory not exist on the page alignment)
 *   ENOMEM		can't check because it's difficult to create no-memory
 */
static struct test_case tcase[] = {
	{ // case00
		.ttype		= NORMAL,
		.nr_segs	= 1,
		.flags		= 0,
		.ret		= 0,
		.err		= 0,
	},
	{ // case01
		.ttype		= NORMAL,
		.nr_segs	= IOV_MAX,
		.flags		= 0,
		.ret		= 0,
		.err		= 0,
	},
	{ // case02
		.ttype		= NORMAL,
		.nr_segs	= 0,
		.flags		= 0,
		.ret		= -1,
		.err		= EINVAL, // RHEL4U1 + 2.6.18 returns SUCCESS
	},
	{ // case03
		.ttype		= NORMAL,
		.nr_segs	= IOV_MAX + 1,
		.flags		= 0,
		.ret		= -1,
		.err		= EINVAL,
	},
	{ // case04
		.ttype		= NORMAL,
		.nr_segs	= 1,
		.flags		= SPLICE_F_GIFT,
		.ret		= -1,
		.err		= EINVAL,
	},
	{ // case05
		.ttype		= FD_NONE,
		.nr_segs	= 1,
		.flags		= 0,
		.ret		= -1,
		.err		= EBADF,
	},
	{ // case06
		.ttype		= FD_NOT_EXIST,
		.nr_segs	= 1,
		.flags		= 0,
		.ret		= -1,
		.err		= EBADF,
	},
	{ // case07
		.ttype		= FD_FILE,
		.nr_segs	= 1,
		.flags		= 0,
		.ret		= -1,
		.err		= EBADF,
	},
};


/*
 * do_test()
 *
 *   Input  : TestCase Data
 *   Return : RESULT_OK(0), RESULT_NG(1)
 *
 */
static int do_test(struct test_case *tc)
{
	int sys_ret;
	int sys_errno;
	int result = RESULT_OK;
	int fd[2] = { -1, -1 };
	int i, rc;
	struct iovec iov[IOV_MAX + 1];
	char *p, buf[8192];

	switch (tc->ttype) {
	case FD_NOT_EXIST:
		fd[1] = INT_MAX - 1;
		/* fallthrough */
	case FD_NONE:
		break;
	case FD_FILE:
		fd[1] = open("/", O_RDONLY);
		if (fd[1] < 0) {
			EPRINTF("can't open \"/\".\n");
			return 1;
		}
		break;
	default:
		rc = pipe(fd);
		if (rc < 0) {
			EPRINTF("pipe failed.\n");
			return -1;
		}
		break;
	}

	/*
	 * Set up iov
	 */
	p = buf;
	for (i = 0; i < IOV_MAX + 1; i++) {
		iov[i].iov_base = p;
		iov[i].iov_len = 2;
		p += 2;
	}

	/*
	 * Execute system call
	 */
	errno = 0;
	sys_ret = syscall(__NR_vmsplice, fd[1], iov, tc->nr_segs, tc->flags);
	sys_errno = errno;

	/*
	 * Check results
	 */
	result |= (sys_errno != tc->err);
	PRINT_RESULT(sys_ret >= 0, tc->ret, tc->err, sys_ret, sys_errno);

	if (fd[0] >= 0)
		close(fd[0]);
	if (fd[1] >= 0)
		close(fd[1]);

	return result;
}


/*
 * usage()
 */
static void usage(const char *progname)
{
	EPRINTF("usage: %s [options]\n", progname);
	EPRINTF("This is a regression test program of %s system call.\n",
		SYSCALL_NAME);
	EPRINTF("options:\n");
	EPRINTF("    -d --debug           Show debug messages\n");
	EPRINTF("    -h --help            Show this message\n");

	RPRINTF("NG\n");
	exit(1);
}


/*
 * main()
 */
int main(int argc, char *argv[])
{
	int result = RESULT_OK;
	int c;
	int i;

	struct option long_options[] = {
		{ "debug", no_argument, 0, 'd' },
		{ "help",  no_argument, 0, 'h' },
		{ NULL, 0, NULL, 0 }
	};

	progname = strchr(argv[0], '/');
	progname = progname ? progname + 1 : argv[0];

	while ((c = getopt_long(argc, argv, "dh", long_options, NULL)) != -1) {
		switch (c) {
		case 'd':
			opt_debug = 1;
			break;

		default:
			usage(progname);
			// NOTREACHED
		}
	}

	if (argc != optind) {
		EPRINTF("Options are not match.\n");
		usage(progname);
		// NOTREACHED
	}

	/*
	 * Execute test
	 */
	for (i = 0; i < (int)(sizeof(tcase) / sizeof(tcase[0])); i++) {
		int ret;

		PRINTF("(case%02d) START\n", i);
		ret = do_test(&tcase[i]);
		PRINTF("(case%02d) END => %s\n", i, (ret == 0) ? "OK" : "NG");

		result |= ret;
	}

	/*
	 * Check results
	 */
	switch(result) {
	case RESULT_OK:
		RPRINTF("OK\n");
		break;

	default:
		RPRINTF("NG\n");
		break;
	}

	return 0;
}

