// -*- mode:c++; indent-tabs-mode:nil; tab-width:2;buffer-file-coding-system:euc-jp-unix; -*-
/**
   @file
   @brief 

   @author seagull
   @version "$Id: BugItem.cpp,v 1.1.1.1 2004/01/12 12:19:12 seagull Exp $"
 */
#include "common.h"

#include <fstream>
#include <ctime>
#if TIME_WITH_SYS_TIME
#include <sys/time.h>
#endif
#include <cstring>
#include <cstdlib>

#include <libxml/xmlIO.h>
#include <libxml/encoding.h>

#include "../lib/getdate.h"
#include "log.h"
#include "Exception.h"
#include "Preferences.h"
#include "BugItem.h"

#include "XMLHelper.h"

namespace bugxee {
  

  /**
     Destoractor
   */
  BugItem::~BugItem()
  {
  }



  /**
     Change subject
     @param subject New subject
   */
  BugItem::Jornal_t const&
  BugItem::changeSubject(char const* subject)
  {
    Jornal_t tmp;
    tmp.stamp();
    tmp.subject_ = subject;
    jornals_.push_back(tmp);
    return jornals_.back();
  }

  /**
     Append message to bug.
     @param subject Message subject.
     @param message Message body.
   */
  BugItem::Jornal_t const&
  BugItem::appendMessage(char const* message)
  {
    Jornal_t tmp;
    tmp.stamp();
    tmp.message_ = message;
    jornals_.push_back(tmp);
    return jornals_.back();
  }

  BugItem::Jornal_t const&
  BugItem::appendJornal(BugItem::Jornal_t const& src)
  {
    jornals_.push_back(src);
    return jornals_.back();
  }


  /**
     Clean bug jornal
   */
  void
  BugItem::Jornal_t::clear()
  {
    timestamp_ = 0;
    userName_.clear();
    userMailAddress_.clear();
    status_.clear();
    maintener_.clear();
    priority_ = 0;
    addedCategory_.clear();
    removedCategory_.clear();
    subject_.clear();
    message_.clear();
  }

  void
  BugItem::Jornal_t::stamp()
  {
    Preferences const& pref = Preferences::instance();

    struct timeval tv;
    struct timezone tz;
    OS_VERIFY(::gettimeofday(&tv, &tz));

    timestamp_ = tv.tv_sec; // + tz.tz_minuteswest * 60;
    userName_ = pref.userName();
    userMailAddress_ = pref.userMailAddress();
  }


  /**
     Load bug details from file.
     @param path Path to bug file.
   */
  void
  BugItem::loadFromFile(char const* path)
  {
    struct handler_t {
      int* bugno;
      BugItem* item;
      BugItem::Jornal_t jornal_;

      enum { OUTSIDE, INROOT, INMESSAGE } state;

      static void startElement(void* userdata, xmlChar const* name, xmlChar const** attrs)
      {
        handler_t* ctxt = reinterpret_cast<handler_t*>(userdata);

        char const* elmname = reinterpret_cast<char const*>(name);
        if (! ::strcasecmp(elmname, "bugxee-bug") )
          {
            if (ctxt->state != OUTSIDE)
              return ;
            ctxt->state = INROOT;
            while (*attrs)
              {
                char const* attrname = reinterpret_cast<char const*>(attrs[0]);
                char const* attrvalue = reinterpret_cast<char const*>(attrs[1]);
                attrs += 2;
                if (! ::strcasecmp(attrname, "bugno"))
                  *ctxt->bugno = ::atoi(attrvalue);
              }
          }
        else if (! ::strcasecmp(elmname, "jornal") )
          {
            if (ctxt->state != INROOT)
              return ;
            ctxt->state = INMESSAGE;
            ctxt->jornal_.clear();
            while (*attrs)
              {
                char const* attrname = reinterpret_cast<char const*>(attrs[0]);
                char const* attrvalue = reinterpret_cast<char const*>(attrs[1]);
                attrs += 2;
                if (! ::strcasecmp(attrname, "timestamp"))
                  ctxt->jornal_.timestamp_ = date::get_date(attrvalue);
                else if (! ::strcasecmp(attrname, "user"))
                  ctxt->jornal_.userName_ = attrvalue;
                else if (! ::strcasecmp(attrname, "mail-address"))
                  ctxt->jornal_.userMailAddress_ = attrvalue;
                else if (! ::strcasecmp(attrname, "status"))
                  ctxt->jornal_.status_ = attrvalue;
                else if (! ::strcasecmp(attrname, "maintener"))
                  ctxt->jornal_.maintener_ = attrvalue;
                else if (! ::strcasecmp(attrname, "priority"))
                  ctxt->jornal_.priority_ = atoi(attrvalue);
                else if (! ::strcasecmp(attrname, "added-category"))
                  ctxt->jornal_.addedCategory_ = attrvalue;
                else if (! ::strcasecmp(attrname, "removed-category"))
                  ctxt->jornal_.removedCategory_ = attrvalue;
                else if (! ::strcasecmp(attrname, "subject"))
                  ctxt->jornal_.subject_ = attrvalue;
              }
          }
      }

      static void
      endElement(void* userdata, xmlChar const* name)
      {
        handler_t* ctxt = reinterpret_cast<handler_t*>(userdata);
        char const* elmname = reinterpret_cast<char const*>(name);
        if (! ::strcasecmp(elmname, "jornal") )
          {
            ctxt->state = INROOT;
            ctxt->item->appendJornal(ctxt->jornal_);
          }
        else if (! ::strcasecmp(elmname, "bugxee-bug") )
          ctxt->state = OUTSIDE;
      }

      static void
      characters(void* userdata, xmlChar const* ch, int len)
      {
        handler_t* ctxt = reinterpret_cast<handler_t*>(userdata);
        if (ctxt->state == INMESSAGE)
          ctxt->jornal_.message_.append(reinterpret_cast<char const*>(ch), len);
      }
    };

    ::xmlSAXHandler sax_handler;
    handler_t context;
    context.item = this;
    context.bugno = &this->bugno_;
    context.state = handler_t::OUTSIDE;
    ::xmlInitializePredefinedEntities();
    ::xmlParserCtxtPtr ctxt = XML::initSAXHandler(&sax_handler, path, &context);
    sax_handler.startElement = handler_t::startElement;
    sax_handler.endElement = handler_t::endElement;
    sax_handler.characters = handler_t::characters;
    int result = ::xmlParseDocument(ctxt);
    ctxt->sax = NULL;
    ::xmlFreeParserCtxt(ctxt);
    ::xmlCleanupPredefinedEntities();
    if (result < 0)
      throw std::runtime_error(_("Failed to load bug item."));
  }



  /**
     Save bug details to file
     @param path Path to file
   */
  void
  BugItem::saveToFile(char const* path) const
  {
    std::ofstream ost(path);
    if (! ost)
      throw OSError(path);
    saveToStream(ost);
  }


  /**
     Save bug details to stream.
     @param ost Target stream.
   */
  void
  BugItem::saveToStream(std::ostream& ost) const
  {
    // Create a document and root node.
    xmlDocPtr doc = xmlNewDoc((xmlChar const*)"1.0");
    xmlDocSetRootElement(doc, toDOMTree());

    ost << doc;

    xmlFreeDoc(doc);
  }

  /**
     Convert bug detail to DOM node
     @returns DOM element node
   */
  xmlNodePtr
  BugItem::toDOMTree(xmlNodePtr parent) const
  {
    log.debug(_("Building DOM tree of BugItem for BUG#%04d"), bugno());

    xmlNodePtr root = parent? XML::NewChild(parent, NULL, "bugxee-bug", NULL) :
                              XML::NewNode(NULL, "bugxee-bug");
    char tmp[128]; sprintf(tmp, "%04d", bugno());
    xmlSetProp(root, (xmlChar const*)"bugno", (xmlChar const*)tmp);

    for (JornalList::const_iterator i = jornals_.begin();
         i != jornals_.end(); i++)
      {
        xmlNodePtr node = XML::NewTextChild(root, NULL, "jornal",
                                            i->message_.empty()? NULL : i->message_.c_str());

        XML::SetProp(node, "timestamp", date::date2str(i->timestamp_));
        XML::SetProp(node, "user", i->userName_);
        XML::SetProp(node, "mail-address", i->userMailAddress_);
        if (! i->status_.empty())
          XML::SetProp(node, "status", i->status_);
        if (! i->maintener_.empty())
          XML::SetProp(node, "maintener", i->maintener_);
        if (i->priority_)
          {
            sprintf(tmp, "%d", i->priority_);
            XML::SetProp(node, "priority", tmp);
          }
        if (! i->addedCategory_.empty())
          XML::SetProp(node, "added-category", i->addedCategory_);
        if (! i->removedCategory_.empty())
          XML::SetProp(node, "removed-category", i->removedCategory_);
        if (! i->subject_.empty())
          XML::SetProp(node, "subject", i->subject_);
      }

    return root;
  }



  /**
     Get a subject of this bug.
     @returns Subject string.
     @remarks
     The subject of bug equals subject of first jornal.
     So first jornal entry must hs a subject and mesage.
   */
  std::string const&
  BugItem::subject() const
  {
    static std::string result;
    for (JornalList::const_iterator i = jornal_begin();
         i != jornal_end(); i++)
      if (! i->subject_.empty())
        result = i->subject_;
    return result;
  }


  /**
     Get maintenner name for currently assigned on this bug.
     @returns Maintenner name.
   */
  std::string const&
  BugItem::maintener() const
  {
    static std::string dft("");
    std::string const* result = NULL;

    for (JornalList::const_iterator i = jornal_begin();
         i != jornal_end(); i++)
      if (! i->maintener_.empty())
        result = &i->maintener_;
    return result? *result : dft;
  }

  /**
   */
  std::string const&
  BugItem::status() const
  {
    static std::string dft("incomming");
    std::string const* result = NULL;
    for (JornalList::const_iterator i = jornal_begin();
         i != jornal_end(); i++)
      if (! i->status_.empty())
        result = &i->status_;
    return result? *result : dft;
  }


  /**
   */
  unsigned int
  BugItem::priority() const
  {
    int result = 5;
    for (JornalList::const_iterator i = jornal_begin();
         i != jornal_end(); i++)
      if (i->priority_)
        result = i->priority_;
    return result;
  }

  /**
   */
  time_t
  BugItem::lastModified() const
  {
    time_t result = 0;
    for (JornalList::const_iterator i = jornal_begin();
         i != jornal_end(); i++)
      if (i->timestamp_)
        result = i->timestamp_;
    return result;
  }


  /**
   */
  void
  BugItem::enumCategories(std::set<std::string>* dst) const
  {
    for (JornalList::const_iterator i = jornal_begin();
         i != jornal_end(); i++)
      {
        if (! i->addedCategory_.empty())
          dst->insert(i->addedCategory_);
        if (! i->removedCategory_.empty())
          dst->erase(i->removedCategory_);
      }
  }

}

