// -*- mode:c++; indent-tabs-mode:nil; tab-width:2;buffer-file-coding-system:euc-jp-unix; -*-
/**
   @file
   @brief Module for abstract repository class.

   @author seagull
   @version "$Id: LocalRepository.cpp,v 1.1.1.1 2004/01/12 12:19:12 seagull Exp $"
 */
#include "common.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>

#include <ctime>
#include <iostream>
#include <fstream>
#include <stdexcept>

#include <boost/lambda/lambda.hpp>
//#include <boost/lambda/bind.hpp>
//#include <boost/lambda/loops.hpp>
//#include <boost/lambda/construct.hpp>
#include <boost/bind.hpp>

#include "log.h"
#include "Exception.h"
#include "LocalRepository.h"
#include "XMLHelper.h"
#include "Summary.h"

namespace bugxee {

  void
  LocalRepository::loadTitle() const
  {
    throw std::runtime_error(_("No title present by repository."));
  }



  void
  LocalRepository::loadConfig() throw (std::runtime_error)
  {
    struct handler_t {
      std::string* title_;
      std::list<std::string>* statuses_;
      std::set<std::string>* mainteners_;
      std::set<std::string>* categories_;

      static void onRepositoryAttribute(handler_t* ctxt, char const* name, char const* value)
      {
        if (! ::strcasecmp(name, "title"))
          *(ctxt->title_) = value;
      }
      static void onStatusAttribute(handler_t* ctxt, char const* name, char const* value)
      {
        if (! ::strcasecmp(name, "name"))
          ctxt->statuses_->push_back(value);
      }
      static void onMaintenerAttribute(handler_t* ctxt, char const* name, char const* value)
      {
        if (! ::strcasecmp(name, "name"))
          ctxt->mainteners_->insert(value);
      }
      static void onCategoryAttribute(handler_t* ctxt, char const* name, char const* value)
      {
        if (! ::strcasecmp(name, "name"))
          ctxt->categories_->insert(value);
      }

      static void startElement(handler_t* ctxt, char const* name, char const** attrs)
      {
        struct {
          char const* name;
          void (*proc)(handler_t*, char const*, char const*);
          bool equals(char const* p) const { return ! ::strcasecmp(p, name); }
        } *p, tbl[] = {
          { "bugxee-repository",  onRepositoryAttribute },
          { "status", onStatusAttribute },
          { "maintener", onMaintenerAttribute },
          { "category", onCategoryAttribute },
          { NULL, NULL }
        };

        for (p = tbl; p->name; p++)
          if (! ::strcasecmp(name, p->name))
            {
              for ( ; *attrs; attrs++)
                p->proc(ctxt, attrs[0], attrs[1]);
              break;
            }
      }
    };

    std::string path(dir_ + "/config.xml");
    ::xmlSAXHandler sax_handler;
    handler_t context;
    context.title_ = &title_;
    context.statuses_ = &statuses_;
    context.mainteners_ = &mainteners_;
    context.categories_ = &categories_;

    ::xmlInitializePredefinedEntities();
    ::xmlParserCtxtPtr ctxt = XML::initSAXHandler(&sax_handler, path.c_str(), &context);
    sax_handler.startElement = ::startElementSAXFunc(handler_t::startElement);
    int result = ::xmlParseDocument(ctxt);
    ctxt->sax = NULL;
    ::xmlFreeParserCtxt(ctxt);
    ::xmlCleanupPredefinedEntities();

    if (result < 0)
      throw std::runtime_error(_("Failed to load config file."));
  }


  void
  LocalRepository::saveConfig() throw (std::runtime_error)
  {
    xmlDocPtr doc = XML::newDoc();
    xmlNodePtr root = xmlNewDocNode(doc, NULL, (xmlChar const*)"bugxee-repository", NULL);
    xmlDocSetRootElement(doc, root);

    XML::SetProp(root, "title", title_);

    struct helper {
      static void perform(xmlNodePtr parent, char const* elm, char const* attr, std::string const& val)
      {
        XML::SetProp(XML::NewChild(parent, NULL, elm, NULL), attr, val);
      }
    };

    std::for_each(statuses_.begin(), statuses_.end(),
                  boost::bind(&helper::perform, root, "status", "name", _1) );
    std::for_each(mainteners_.begin(), mainteners_.end(),
                  boost::bind(&helper::perform, root, "maintener", "name", _1) );
    std::for_each(categories_.begin(), categories_.end(),
                  boost::bind(&helper::perform, root, "category", "name", _1) );

    std::string path(dir_ + "/config.xml");
    std::ofstream ost(path.c_str());
    if (! ost)
      {
        ::xmlFreeDoc(doc);
        throw OSError(path + " open");
      }

    ost << doc;

    xmlFreeDoc(doc);
  }


  /**
     Commit a transaction
   */
  void
  LocalRepository::commit()
  {
    if (maintenersModified_ || categoriesModified_)
      saveConfig();
    if (stateModified_)
      saveState();
  }


  /**
     
   */
  void
  LocalRepository::loadState() throw(std::runtime_error)
  {
    struct handler_t {
      int* nextBugno_;

      static void startElement(handler_t* ctxt, char const* name, char const** attrs)
      {
        if (! ::strcasecmp(name, "state"))
          {
            while (*attrs)
              {
                if (! ::strcasecmp(attrs[0], "next-bugno"))
                  *ctxt->nextBugno_ = atoi(attrs[1]);
                attrs += 2;
              }
          }
      }
    };

    std::string path(dir_ + "/state.xml");
    ::xmlSAXHandler sax_handler;
    handler_t context;
    context.nextBugno_ = &nextBugno_;
    ::xmlParserCtxtPtr ctxt = XML::initSAXHandler(&sax_handler, path.c_str(), &context);
    sax_handler.startElement = ::startElementSAXFunc(handler_t::startElement);
    int result = ::xmlParseDocument(ctxt);
    ctxt->sax = NULL;
    ::xmlFreeParserCtxt(ctxt);

    if (result < 0)
      throw std::runtime_error(_("Failed to load state file."));
  }


  void
  LocalRepository::saveState() throw (OSError)
  {
    log.debug(_("Trying to save a state file."));
    std::ofstream ost((dir_ + "/state.xml").c_str());
    if (! ost)
          throw OSError("Open state file");
    ost <<
      "<?xml version='1.0'?>\n"
      "<state next-bugno='" << nextBugno_ << "'/>";
    ost.close();
  }


  void
  LocalRepository::open() throw(std::runtime_error)
  {
    loadConfig();
    loadState();
  }




  /**
     Load bug index cache file.
     @param domain  Target domain name of index
     @param name Target index name
     @remarks
     domain and name must not be NULL or empty string, unless exception will throws.
     If the index already loaded, will returns previous one.
   */
  Repository::Index_t&
  LocalRepository::loadIndex(char const* domain, char const* name)
  {
    if (! domain || ! *domain)
      throw std::invalid_argument(_("empty index domain name"));
    if (! name || ! *name)
      throw std::invalid_argument(_("empty index name"));

    std::string idxname(makeIndexInternalName(domain, name));

    Indexes_t::iterator iIdx = indexes_.find(idxname);
    if (iIdx != indexes_.end())
      return iIdx->second;

    std::string path(dir_ + "/index." + idxname);
    log.debug(_("Trying to load a index file for %s"), path.c_str());

    struct handler_t {
      Index_t* idx_;

      static void startElement(void* userdata, xmlChar const* name,
                               xmlChar const** attrs)
      {
        handler_t* ctxt = reinterpret_cast<handler_t*>(userdata);
        if (! std::strcmp(reinterpret_cast<char const*>(name), "li"))
          while (*attrs)
            {
              if (! ::strcasecmp(reinterpret_cast<char const*>(attrs[0]), "bugno"))
                ctxt->idx_->insert(atoi(reinterpret_cast<char const*>(attrs[1]) )  );
              attrs += 2;
            }
      }
    };
    ::xmlSAXHandler sax_handler;
    handler_t context;
    context.idx_ = &(indexes_[idxname]);
    ::xmlParserCtxtPtr ctxt = XML::initSAXHandler(&sax_handler, path.c_str(), &context);
    sax_handler.startElement = handler_t::startElement;
    int result = ::xmlParseDocument(ctxt);
    ctxt->sax = NULL;
    ::xmlFreeParserCtxt(ctxt);

    if (result < 0)
      throw std::runtime_error(_("Failed to load index file."));
    else
      log.debug(_("Success to load %d bugs in index %s"),
                context.idx_->size(), idxname.c_str());

    return *context.idx_;
  }


  /**
     Save bug index cache file.
     @param domain Target domain name of index.
     @param name Target index name.
   */
  void
  LocalRepository::saveIndex(char const* domain, char const* name)
  {
    if (! domain || ! *domain)
      throw std::invalid_argument(_("empty index domain name"));
    if (! name || ! *name)
      throw std::invalid_argument(_("empty index name"));

    std::string idxname(makeIndexInternalName(domain, name));

    Indexes_t::iterator i = indexes_.find(idxname);
    if (i == indexes_.end())
      return ;
    Index_t& idx = i->second;

    log.debug(_("Trying to save a index file for %s"), idxname.c_str());
    std::string path(dir_ + "/index." + idxname);
    std::ofstream ost(path.c_str());
    if (ost)
      {
        using namespace boost::lambda;
        ost << "<?xml version='1.0'?>\n"
            << "<bugxee-index>";
        std::for_each(idx.begin(), idx.end(),
                      ost << constant("<li bugno='") << boost::lambda::_1 << "'/>");
        ost << "</bugxee-index>";
        ost.close();
      }
  }


  /**
     Lock repository really.
   */
  void
  LocalRepository::lock_impl() const
  {
    std::string lockpath;
    lockpath = dir_ + "/.lock";

    log.debug(_("To make a lock file as %s"), lockpath.c_str());

    struct stat st;
    while (::stat(lockpath.c_str(), &st) >= 0)
      {
        log.error(_("Repository is now locking by other user. wait %dsec...."), 5);
        if (::sleep(5) != 0)
          throw std::runtime_error(_("Repository locks to canceled."));
      }
    if (::symlink("/dev/null", lockpath.c_str()) < 0)
      throw std::runtime_error(_("Failed to lock repository."));
  }


  /**
     Unlock repository really.
   */
  void
  LocalRepository::unlock_impl() const
  {
    std::string lockpath;
    lockpath = dir_ + "/.lock";

    log.debug(_("To remove a lock file as %s"), lockpath.c_str());
    if (::unlink(lockpath.c_str()) < 0)
      throw std::logic_error(_("Failed to unlock repository. please contact to administrator...sorry."));
  }




  /**
     Convert bug# to filename
     @param bugno target bug#
     @return filename string.
   */
  std::string
  LocalRepository::bugno2filename(int bugno) const
  {
    char tmp[5];
    std::sprintf(tmp, "/%02d/%04d", bugno / 100, bugno);
    return dir_ + tmp;
  }


  /**
     Make pathname of summary for specified timestamp.
     @param timestamp
     @returns Path name of summary
   */
  std::string
  LocalRepository::summaryPath(time_t timestamp) const
  {
    char buf[30];
    struct tm tm;
    ::gmtime_r(&timestamp, &tm);
    std::strftime(buf, 30, "/summary/%Y%m.xml", &tm);
    return dir_ + buf;
  }


  /**
   */
  void
  LocalRepository::recordSummary(time_t timestamp, int bugno,
                                 Summary::Entry_t::Action_t action,
                                 std::string const& text) const
  {
    std::string path = summaryPath(timestamp);
    Summary summary;
    try {
      summary.loadFromFile(path.c_str());
    }
    catch (...)
      {
      }
    summary.append(timestamp, bugno, action, text);
    std::ofstream ofst(path.c_str());
    summary.saveToStream(ofst);
  }



  /**
   */
  int
  LocalRepository::assignBugno()
  {
    stateModified_ = true;
    return nextBugno_++;
  }




  /**
     Initialize repository directory.
   */
  void
  LocalRepository::initialize()
  {
    log.debug(_("Checking for target directory is exists and empty?"));
    DIR* dir = ::opendir(dir_.c_str());
    if (dir == NULL)
      throw OSError(_("opendir to prepare init repository"));
    struct dirent* ent;
    while ((ent = readdir(dir)) != NULL)
      {
        if (::strcmp(ent->d_name, ".") && ::strcmp(ent->d_name, ".."))
          {
            ::closedir(dir);
            throw std::runtime_error(_("Repository directory is not empty."));
          }
      }
    ::closedir(dir);



    log.info(_("Trying to initialize repository %s "), dir_.c_str());
    //    static char errormessage[] = "Failed to initialize repository.";
    static char const* defaultStatuses[] = {
      "incomming", "opening", "fixed", "pending", "closed", "rejected", "deleted", NULL
    };


    Lock locker(*this);

    std::ofstream ost;

    // Create separate directory.
    log.debug(_("Trying to create separate directories."));
    for (int i = 0; i < 100; i++)
      {
        char subdir[5];
        std::sprintf(subdir, "/%02d", i);
        if (mkdir((dir_ + subdir).c_str(), 0775) < 0)
          throw OSError(std::string("mkdir ") + subdir);
      }

    // Create configration file
    log.debug(_("Trying to create bugxee.conf"));
    title_ = "";
    std::copy(defaultStatuses, defaultStatuses + 7, std::back_inserter(statuses_));
    saveConfig();

    // Create index files by statuses.
    log.debug(_("Trying to create index files."));
    for (char const** p =defaultStatuses; *p; p++)
      {
        indexes_[makeIndexInternalName("status", *p)]; // Make empty index
        saveIndex("status", *p); // and save it.
      }

    // Create index files by priority
    for (int i = 1; i <= 9; i++)
      {
        char tmp[10];
        sprintf(tmp, "%d", i);
        indexes_[makeIndexInternalName("priority", tmp)]; // Make empty index
        saveIndex("priority", tmp);
      }

    // Create summary directory
    if (mkdir((dir_ + "/summary").c_str(), 0775) < 0)
      throw OSError(_("make summary directory"));

    // Create repository status file.
    nextBugno_ = 0;
    saveState();
  }





  /**
     Add new bug item.
     @param msg Message object
     @returns new BUG#
   */
  BugItem*
  LocalRepository::addBug(char const* subject, char const* message)
  {
    if (! subject || ! *subject || ! message || ! *message)
      throw std::invalid_argument(_("Invalid message"));

    Lock locker(*this);

    int bugno = assignBugno();
    stateModified_ = true;
    log.info(_("Bug# %d is assigned to new bug"), bugno);

    loadIndex("status", "incomming");
    std::auto_ptr<LocalBugItem> newitem(new LocalBugItem(this, bugno));
    BugItem::Jornal_t jornal;
    jornal.clear();
    jornal.stamp();
    jornal.subject_ = subject;
    jornal.message_ = message;
    newitem->appendJornal(jornal);
    newitem->save();

    recordSummary(newitem->jornal_begin()->timestamp_, bugno,
                  Summary::Entry_t::CREATE, subject);

    bugs_[bugno] = newitem.release();
    index("status", "incomming").insert(bugno);
    index("priority", "5").insert(bugno);
    saveIndex("status", "incomming");
    saveIndex("priority", "5");

    log.info(_("Success to append message to bug#%04d"), bugno);
    return bugs_[bugno];
  }


  void
  LocalRepository::changeSubject(int bugno, char const* subject)
  {
    if (! subject || !* subject)
      throw std::invalid_argument(_("Invalid message"));

    Lock locker(*this);
    BugItem* bug = findByBugno(bugno);
    if (!bug)
      throw UnknownBugNo(bugno);

    BugItem::Jornal_t const& jornal =  bug->changeSubject(subject);
    recordSummary(jornal.timestamp_, bugno, Summary::Entry_t::SUBJECT, "");
    bug->save();

  }


  void
  LocalRepository::addMessage(int bugno, char const* message)
  {
    if (! message || ! *message)
      throw std::invalid_argument(_("Invalid message"));

    Lock locker(*this);

    BugItem* bug = findByBugno(bugno);
    if (! bug)
      throw UnknownBugNo(bugno);

    BugItem::Jornal_t const& jornal =
      bug->appendMessage(message);
    recordSummary(jornal.timestamp_, bugno, Summary::Entry_t::MESSAGE, "");
    bug->save();
  }



  void
  LocalRepository::addJornal(int bugno, BugItem::Jornal_t const& jornal)
  {
    Lock locker(*this);

    BugItem* bug = findByBugno(bugno);
    if (!bug)
      throw UnknownBugNo(bugno);

    std::string oldStatus = bug->status();
    unsigned int oldPriority = bug->priority();
    std::string oldMaintener = bug->maintener();

    bug->appendJornal(jornal);

    if (! jornal.status_.empty())
      {
        recordSummary(jornal.timestamp_, bugno, Summary::Entry_t::STATUS_CHANGE, jornal.status_);
        updateIndex("status", *bug, oldStatus.c_str(), jornal.status_.c_str());
      }

    if (jornal.priority_ && jornal.priority_ != oldPriority)
      {
        char tmp[2][10];
        sprintf(tmp[0], "%d", oldPriority);
        sprintf(tmp[1], "%d", jornal.priority_);
        recordSummary(jornal.timestamp_, bugno, Summary::Entry_t::PRIORITY_CHANGE, tmp[1]);

        updateIndex("priority", *bug,
                    (oldPriority? tmp[0] : NULL), (jornal.priority_? tmp[1] : NULL));
      }
    if (! jornal.maintener_.empty() && jornal.maintener_ != oldMaintener)
      {
        recordSummary(jornal.timestamp_, bugno, Summary::Entry_t::ASSIGN, jornal.maintener_);
        updateIndex("maintener", *bug,
                    oldMaintener.c_str(), jornal.maintener_.c_str());
      }
    if (! jornal.addedCategory_.empty())
      {
        recordSummary(jornal.timestamp_, bugno, Summary::Entry_t::ADD_CATEGORY, jornal.addedCategory_);
        updateIndex("category", *bug, NULL, jornal.addedCategory_.c_str());
      }
    if (! jornal.removedCategory_.empty())
      {
        recordSummary(jornal.timestamp_, bugno, Summary::Entry_t::DEL_CATEGORY, jornal.removedCategory_);
        updateIndex("category", *bug, jornal.removedCategory_.c_str(), NULL);
      }

    bug->save();
  }


  /**
     Load existing bug.
     @param bugno Target BUG#
   */
  BugItem*
  LocalRepository::loadBug(int bugno)
  {
    std::auto_ptr<LocalBugItem> bug(new LocalBugItem(this, bugno));
    bug->load();
    return bug.release();
  }


  /**
     Join maintener to repository.
     @param name Name of the Mainter.
   */
  void
  LocalRepository::joinMaintener(char const* name)
  {
    if (mainteners_.find(name) == mainteners_.end())
      {
        mainteners_.insert(name);
        indexes_[makeIndexInternalName("maintener", name)];
        saveIndex("maintener", name);
      }
    maintenersModified_ = true;
  }


  /**
   */
  void
  LocalRepository::addCategory(std::string const& category)
  {
    if (categories_.find(category) == categories_.end())
      {
        categories_.insert(category);
        indexes_[makeIndexInternalName("category", category.c_str())];
        saveIndex("category", category.c_str());
      }
    categoriesModified_ = true;
  }



  /**
   */
  void
  LocalRepository::LocalBugItem::load()
  {
    LocalRepository* rep = reinterpret_cast<LocalRepository*>(repository());
    std::string path(rep->bugno2filename(bugno()));
    loadFromFile(path.c_str());
  }

  void
  LocalRepository::LocalBugItem::save()
  {
    LocalRepository* rep = reinterpret_cast<LocalRepository*>(repository());
    std::string path(rep->bugno2filename(bugno()));
    saveToFile(path.c_str());
  }


}
