// -*- mode:c++; indent-tabs-mode:nil; tab-width:2;buffer-file-coding-system:euc-jp-unix; -*-
/**
   @file
   @brief Preferences

   @author seagull
   @version "$Id: Preferences.cpp,v 1.1.1.1 2004/01/12 12:19:12 seagull Exp $"
*/
#include "common.h"

#include <string.h>
#include <stdlib.h>
#include <iostream>
#include <stdexcept>
#if HAVE_LANGINFO_H
#include <langinfo.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "log.h"
#include "Exception.h"
#include "charsets.h"
#include "XMLHelper.h"
#include "Preferences.h"


namespace bugxee {
  
  Preferences&
  Preferences::instance()
  {
    static Preferences inst;
    if (! inst.isLoaded())
      {
        // Get environment value for repository path.
        char* env = std::getenv("BUGXEE_REPOSITORY");
        if (env && *env)
          inst.repository_ = env;
        env = std::getenv("NAME");
        if (env && *env) inst.userName_ = env;
        env = std::getenv("USERNAME");
        if (env && *env) inst.userName_ = env;
        env = std::getenv("LOGNAME");
        if (env && *env) inst.userName_ = env;

        // Load user specific configration.
        char* home = std::getenv("HOME");
        if (home && *home != '\0')
          {
            try
              {
                inst.load(std::string(home).c_str());
              }
            catch (std::exception& e)
              {
                log.info(_("Failed to load user specific preferences: %s"), e.what());
              }
          }

        // Load module specific configration.
        try
          {
            inst.load(".");
          }
        catch (std::exception& e)
          {
            log.info(("Failed to load module specific preferences: %s"), e.what());
          }

        // Generate mail address from user and host name, if no specified.
        if (inst.userMailAddress_.empty())
          {
            inst.userMailAddress_ = inst.userName_;
            char const* env = std::getenv("HOSTNAME");
            if (env && *env)
              {
                inst.userMailAddress_.append("@");
                inst.userMailAddress_.append(env);
              }
          }

      }

    return inst;
  }


  /**
     Constractor.
   */
  Preferences::Preferences()
  {
    loaded_ = false;
    outputFormat_ = "text";
  }

  /**
     Destoracter.
   */
  Preferences::~Preferences()
  {
  }



  /**
     Parse global command line options.
     @param argc Pointer to ARGC
     @param argv ARGV
     @remarks
       Parsed option removes from argc and argv.
   */
  void
  Preferences::parseOptions(int* argc, char* argv[])
  {
    for(int i = 1; i < *argc && argv[i][0] == '-'; i++)
      {
        char* opt = argv[i];
        argv[i] = NULL;
        switch (opt[1])
          {
          case 'd':
            i++;
            if (i >= *argc)
              throw std::runtime_error(_("No repository present with -d option."));

            repository_ = argv[i];
            argv[i] = NULL;
            break;
          case 'u':
            userName_ = opt + 2;
            break;
          case 'm':
            userMailAddress_ = opt + 2;
            break;
          case 'o':
            outputFormat_ = opt + 2;
            break;
          case 'c':
            localCharset_ = opt + 2;
            break;
          default:
            throw UnknownOptionError(opt);
          }
      }

    char** p = argv + 1;
    for (int i = 1; i < *argc; i++)
      if (argv[i] != NULL)
        *(p++) = argv[i];
    *argc = p - argv;
  }



  /**
     Load preference(.bugxeerc) file.
     @param dir Path to directory contains target file.
   */
  void
  Preferences::load(char const* dir)
  {
    struct handler_t {
      std::string* repdir_;
      std::string* username_;
      std::string* mailaddress_;
      std::string* localCharset_;
      std::string* outputFormat_;
      std::string* editor_;
      Preferences* pref;
      std::string dir;
      static void startElement(void* userData, const xmlChar* name,
                                 const xmlChar** attrs)
      {
        handler_t* ctxt = reinterpret_cast<handler_t*>(userData);
        if (! std::strcmp(reinterpret_cast<char const*>(name), "bugxee-config"))
          {
            while (*attrs != NULL)
              {
                char const* const name = reinterpret_cast<char const*>(attrs[0]);
                char const* const value = reinterpret_cast<char const*>(attrs[1]);

                if (! ::strcasecmp(name, "repository"))
                  {
                    *ctxt->repdir_ = value;
                    if (value[0] != '/')
                      *ctxt->repdir_ = ctxt->dir + "/" + value;
                  }
                else if (! ::strcasecmp(name, "name"))
                  *ctxt->username_ = value;
                else if (! ::strcasecmp(name, "mail-address"))
                  *ctxt->mailaddress_ = value;
                else if (! ::strcasecmp(name, "charset"))
                  *ctxt->localCharset_ = value;
                else if (! ::strcasecmp(name, "output-format"))
                  *ctxt->outputFormat_ = value;
                else if (! ::strcasecmp(name, "editor"))
                  *ctxt->editor_ = value;
                attrs += 2;
              }
          }
      }
    };
    std::string path(dir);
    path += "/.bugxeerc";

    struct stat st;
    if (::stat(path.c_str(), &st) < 0)
      throw OSError(_("Checking to load .bugxeerc."));

    ::xmlSAXHandler sax_handler;
    handler_t context;
    context.pref = this;
    context.repdir_ = &this->repository_;
    context.username_ = &this->userName_;
    context.mailaddress_ = &this->userMailAddress_;
    context.localCharset_ = &this->localCharset_;
    context.outputFormat_ = &this->outputFormat_;
    context.editor_ = &this->editor_;
    context.dir = dir;
    ::xmlParserCtxtPtr ctxt = XML::initSAXHandler(&sax_handler, path.c_str(), &context);
    sax_handler.startElement = handler_t::startElement;
    try
      {
        if (::xmlParseDocument(ctxt) >= 0)
          loaded_ = true;
        else if (! ctxt->wellFormed)
          throw std::runtime_error(path + _(" is not wellformed."));
      }
    catch (std::exception& e)
      {
        ctxt->sax = NULL;
        ::xmlFreeParserCtxt(ctxt);
        throw ;
      }
    ctxt->sax = NULL;
    ::xmlFreeParserCtxt(ctxt);

    if (! editor_.empty())
      setenv("EDITOR", editor_.c_str(), 1);
  }


  /**
    Get a local character codeset with current lcale.
  */
  std::string
  Preferences::localCharset() const
  {
    if (! localCharset_.empty())
      return localCharset_;
#if HAVE_LANGINFO_CODESET
    return std::string(::nl_langinfo(CODESET));
#else
    char const* env = std::getenv("OUTPUTCHARSET");
    if (env) return env;

    throw std::runtime_error(_("No charset specified with local."));
#endif
  }


}

std::ostream&
operator <<(std::ostream& ost, bugxee::Preferences const& pref)
{
  ost << "Laded: " << (pref.isLoaded()?"Yes":"No") << std::endl
      << "Repository: " << bugxee::fromUTF8(pref.repository()) << std::endl
      << "User name: " << bugxee::fromUTF8(pref.userName()) << std::endl
      << "Mail address: " << bugxee::fromUTF8(pref.userMailAddress()) << std::endl
    ;
  return ost;
}
