// -*- mode:c++; indent-tabs-mode:nil; tab-width:2;buffer-file-coding-system:euc-jp-unix; -*-
/**
   @file
   @brief Module for abstract repository class.

   @author seagull
   @version "$Id: Repository.cpp,v 1.1.1.1 2004/01/12 12:19:12 seagull Exp $"
 */
#include "common.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>

#include <algorithm>
#include <iostream>
#include <fstream>
#include <stdexcept>


#include "log.h"
#include "Exception.h"
#include "Repository.h"
#include "XMLHelper.h"


namespace bugxee {

  /**
     Constractor of Repository
   */
  Repository::Repository()
  {
    lockcount_ = 0;
    titleLoaded_ = false;
    maintenersModified_ = false;
    categoriesModified_ = false;
  }

  /**
     Desctoractor of Repository
   */
  Repository::~Repository()
  {
  }


  /**
     Increase lock counter of repository.
     Lock reopsitory if current counter is zero.
   */
  void
  Repository::lock()
  {
    if (lockcount_ == 0)
      lock_impl();
    lockcount_++;
  }

  /**
     Decrease lock counter of repository.
     Unlock repository if counter to be zero.
   */
  void
  Repository::unlock()
  {
    if (lockcount_ == 0)
      throw std::logic_error(_("lock count underflow."));

    lockcount_--;
    if (lockcount_ == 0)
      unlock_impl();
  }




  /**
     Get repository title for human readable.
     @returns Repository title
   */
  std::string const&
  Repository::title() const
  {
    if (! titleLoaded_)
      loadTitle();

    return title_;
  }


  /**
     Make mangled internal name of index under domain.
     @param domain Index domain name
     @param name Index entity name
     @return Internal name
   */
  std::string
  Repository::makeIndexInternalName(char const* domain, char const* name)
  {
    return std::string(domain) + "." + name;
  }


  /**
   */
  void
  Repository::updateIndex(char const* domain, int bugno, char const* oldname, char const* newname)
  {
    Index_t* oldidx = NULL;
    Index_t* newidx = NULL;

    if (!oldname || !*oldname) oldname = NULL;
    if (!newname || !*newname) newname = NULL;

    if ((! oldname && ! newname) ||
        (oldname && newname && !strcmp(oldname, newname)) )
      return ;  // no key changing

    if (oldname)
      {
        oldidx = &index(domain, oldname);
        oldidx->erase(bugno);
      }
    if (newname)
      {
        newidx = &index(domain, newname);
        newidx->insert(bugno);
      }
    if (newidx) saveIndex(domain, newname);
    if (oldidx) saveIndex(domain, oldname);
  }


  void
  Repository::initialize()
  {
    throw std::runtime_error(_("Could not initialize this repository."));
  }


  /**
     Find bug by BUG#
     @param bugno BUG#
   */
  BugItem*
  Repository::findByBugno(int bugno)
  {
    std::map<int, BugItem*>::iterator i = bugs_.find(bugno);
    if (i == bugs_.end())
      return loadBug(bugno);
    return i->second;
  }


  /**
     Test for specified status name isvalue(registered by repository).
     @param name Target status name.
     @returns True if valid, unless false.
   */
  bool
  Repository::isValidStatusName(std::string const& name) const
  {
    return (std::find(statuses_.begin(), statuses_.end(), name) != statuses_.end());
  }


  /**
     Test for specified priority is valid.
     @param priority Target priority
     @returns True if valid, unless false.
  */
  bool
  Repository::isValidPriority(unsigned int priority) const
  {
    return priority >= 1 && priority <=9;
  }

  /**
     Test for specified maintener is joined.
     @param name Name of the maintener.
     @returns True if joined, unless false.
   */
  bool
  Repository::isValidMaintener(std::string const& name) const
  {
    return mainteners_.find(name) != mainteners_.end();
  }



  /**
   */
  bool
  Repository::isValidCategory(std::string const& name) const
  {
    return categories_.find(name) != categories_.end();
  }

}

