// -*- mode:c++; indent-tabs-mode:nil; tab-width:2;buffer-file-coding-system:euc-jp-unix; -*-
/**
   @file
   @brief 

   @author seagull
   @version "$Id: Repository.h,v 1.1.1.1 2004/01/12 12:19:12 seagull Exp $"
 */
#if !defined(__BUGXEE__REPOSITORY_H__)
#define __BUGXEE__REPOSITORY_H__

#include <string>
#include <map>
#include <set>
#include <list>
#include <iostream>
#include <stdexcept>
#include <boost/utility.hpp>


#include "Exception.h"
#include "BugItem.h"
#include "Message.h"

namespace bugxee {

  class Repository :
    public boost::noncopyable
  {
  public:
    typedef std::set<int> Index_t;

    class Lock
      : public boost::noncopyable
    {
    protected:
      Repository& repository_;
    public:
      inline Lock(Repository& rep) : repository_(rep) { repository_.lock(); }
      inline ~Lock() { repository_.unlock(); }
    };


  protected:
    mutable std::string title_;
    mutable bool titleLoaded_;
    unsigned int lockcount_;

    std::list<std::string> statuses_;
    std::set<std::string> mainteners_;
    std::set<std::string> categories_;
    bool maintenersModified_;
    bool categoriesModified_;

    typedef std::map<std::string, Index_t> Indexes_t;
    Indexes_t indexes_;

    std::map<int, BugItem*> bugs_;

  protected:
    virtual void loadTitle() const = 0;

    void lock();
    void unlock();

    virtual void lock_impl() const = 0;
    virtual void unlock_impl() const = 0;
    
    static std::string makeIndexInternalName(char const* domain, char const* name);
    virtual Index_t& loadIndex(char const* domain, char const* name) = 0;

    virtual BugItem* loadBug(int bugno) = 0;

  public:
    Repository();
    virtual ~Repository();
    std::string const& title() const;

    std::list<std::string> const& statuses() const { return statuses_; }

    inline Index_t& index(char const* domain, char const* name)
    {
      return loadIndex(domain, name);
    }
    inline Index_t& index(std::string const& domain, std::string const& name)
    { return index(domain.c_str(), name.c_str()); }
    inline Index_t& index(char const* domain, std::string const& name)
    { return index(domain, name.c_str()); }

    virtual void saveIndex(char const* domain, char const* name) = 0;
    void updateIndex(char const* domain, int bugno, char const* oldname, char const* newname);
    inline void updateIndex(char const* domain, BugItem& bug, char const* oldname, char const* newname)
    {
      updateIndex(domain, bug.bugno(), oldname, newname);
    }
    


    bool isValidStatusName(std::string const& name) const;
    bool isValidPriority(unsigned int priority) const;
    bool isValidMaintener(std::string const& name) const;
    bool isValidCategory(std::string const& category) const;

    BugItem* findByBugno(int bugno);


    virtual void initialize();
    virtual BugItem* addBug(char const* subject, char const* message) = 0;
    virtual void changeSubject(int bugno, char const* subject) = 0;
    virtual void addMessage(int bugno, char const* msg) = 0;
    virtual void addJornal(int bugno, BugItem::Jornal_t const& jornal) = 0;
    virtual void joinMaintener(char const* maintener) = 0;
    virtual void addCategory(std::string const& category) = 0;

    virtual void commit() = 0;
  };


}



#endif

