// -*- mode:c++; indent-tabs-mode:nil; tab-width:2;buffer-file-coding-system:euc-jp-unix; -*-
/**
   @file
   @brief 

   @author seagull
   @version "$Id: Summary.cpp,v 1.1.1.1 2004/01/12 12:19:12 seagull Exp $"
 */
#include "common.h"

#include <strings.h>

#include "../lib/getdate.h"
#include "XMLHelper.h"
#include "Summary.h"
#include "Preferences.h"

namespace bugxee {


  /**
    Default constractor for Summary.
  */
  Summary::Summary()
  {
  }

  char const* Summary::entryNames[Summary::Entry_t::NUM_ACTIONS] = {
    "create", "message", "assign", "status", "subject", "priority", "add-category", "del-category"
  };


  /**
    Load summary from file.
    @param path Path to file.
  */
  void
  Summary::loadFromFile(char const* path)
  {
    struct handler_t {
      Summary::Entries_t* entries_;

      static void startElement(handler_t* ctxt, char const* name, char const** attrs)
      {
        if (! ::strcasecmp("summary", name))
          {
          }
        else
          {
            for (int i = 0; i < Entry_t::NUM_ACTIONS; i++)
              if (! ::strcasecmp(entryNames[i], name))
                {
                  ctxt->onEntry(Entry_t::Action_t(i), attrs);
                  break;
                }
          }
      }

      void 
      onEntry(Summary::Entry_t::Action_t action, char const** attrs)
      {
        Summary::Entry_t entry;
        entry.action_ = action;

        for (; *attrs; attrs += 2)
          {
            if (! ::strcasecmp("timestamp", attrs[0]))
              entry.timestamp_ = date::get_date(attrs[1]);
            else if (! ::strcasecmp("bugno", attrs[0]))
              entry.bugno_ = atoi(attrs[1]);
            else if (! ::strcasecmp("text", attrs[0]))
              entry.text_ = attrs[1];
            else if (! ::strcasecmp("user", attrs[0]))
              entry.userName_ = attrs[1];
            else if (! ::strcasecmp("mail-address", attrs[0]))
              entry.userMailAddress_ = attrs[1];
          }
        entries_->push_back(entry);
      }
    } context;
    ::xmlSAXHandler sax_handler;
    context.entries_ = &this->entries_;
    ::xmlInitializePredefinedEntities();
    ::xmlParserCtxtPtr ctxt = XML::initSAXHandler(&sax_handler, path, &context);
    sax_handler.startElement = (startElementSAXFunc)handler_t::startElement;
    //int result = ::xmlParseDocument(ctxt);
    ::xmlParseDocument(ctxt);
    ctxt->sax = NULL;
    ::xmlFreeParserCtxt(ctxt);
  }


  /**
    Save summary to stream.
    @param ost Target stream.
  */
  void
  Summary::saveToStream(std::ostream& ost)
  {
    ::xmlDocPtr doc = XML::newDoc();
    ::xmlNodePtr root = xmlNewDocNode(doc, NULL, (xmlChar*)"summary", NULL);
    ::xmlDocSetRootElement(doc, root);
    try
      {
        for (Entries_t::iterator i = entries_.begin(); i != entries_.end(); i++)
          {
            if (i->action_ < 0 || i->action_ >= Entry_t::NUM_ACTIONS)
              throw std::logic_error(_("Invaid action value."));

            char strbugno[10];
            std::sprintf(strbugno, "%04d", i->bugno_);

            ::xmlNodePtr node = XML::NewChild(root, NULL, entryNames[i->action_], NULL);
            XML::SetProp(node, "bugno", strbugno);
            XML::SetProp(node, "timestamp", date::date2str(i->timestamp_));
            XML::SetProp(node, "user", i->userName_);
            XML::SetProp(node, "mail-address", i->userMailAddress_);
            XML::SetProp(node, "text", i->text_);
          }
        ost << doc;
      }
    catch(...)
      {
        ::xmlFreeDoc(doc);
        throw;
      }
    ::xmlFreeDoc(doc);
  }


  /**
    Append entry into summary.
    @param timestamp
    @param bugno
    @param action
    @param text
  */
  void
  Summary::append(time_t timestamp, int bugno, Entry_t::Action_t action, std::string const& text)
  {
    if (action < 0 || action >= Entry_t::NUM_ACTIONS)
      throw std::logic_error(_("Invaid action value"));

    Preferences const& pref = Preferences::instance();

    Entry_t entry;
    entry.timestamp_ = timestamp;
    entry.bugno_ = bugno;
    entry.action_ = action;
    entry.text_ = text;
    entry.userName_ = pref.userName();
    entry.userMailAddress_ = pref.userMailAddress();

    entries_.push_back(entry);
  }

}


