// -*- mode:c++; indent-tabs-mode:nil; tab-width:2;buffer-file-coding-system:euc-jp-unix; -*-
/**
   @file
   @brief Module of "add" command
   @author seagull
   @version "$Id: add.cpp,v 1.1.1.1 2004/01/12 12:19:12 seagull Exp $"
 */
#include "../common.h"

#include <boost/bind.hpp>
#include <boost/mem_fn.hpp>

#include "../Exception.h"
#include "../log.h"
#include "../Preferences.h"
#include "../editor.h"
#include "../Outputter.h"
#include "../command.h"
#include "add.h"

namespace bugxee {
  namespace command {

    AbstractCommandFactory const* getAddCommandFactory()
    {
      static GenericCommandFactory<AddCmd>
        factory("add",
                _("Add new bug and assign BUG#."),
                _("Usage: bugxee [global options...] add -m MESSAGE\n"
                  "\n"
                  ""
                  "")
                );
      return &factory;
    }


    /**
       Constractr
       @param repository Target repository
     */
    AddCmd::AddCmd(Repository& repository)
      : AbstractCommand(repository),
        messageSpecified_(false)
    {
    }


    /**
       Parse command specific options.
     */
    void
    AddCmd::parseOptions(int argc, char* argv[])
    {
      while (argc > 0)
        {
          if (argv[0][0] == '-')
            {
              switch (argv[0][1])
                {
                case 'm':
                  if (argc <= 1)
                    throw std::runtime_error(_("Empty message"));
                  message_.parseWithLocalEncoding(argv[1]);
                  messageSpecified_ = true;

                  argc--;
                  argv++;
                  break;
                default:
                  throw UnknownOptionError(argv[0]);
                }
            }
          else
            throw UnknownOptionError(argv[0]);

          argc--;
          argv++;
        }

      if (! messageSpecified_)
        for (;;)
          {
            ExternalEditor editor;
            std::auto_ptr<char> msg = editor.launch();
            message_.parseWithLocalEncoding(msg.get());
            if (message_.isValid())
              break;
            throw std::invalid_argument(_("No message present.")); // TODO: implements to retry
          }

      if (! message_.isValid())
        throw std::runtime_error(_("No message present."));
    }


    /**
       Parform add command.
     */
    void
    AddCmd::perform()
    {
      Preferences& pref = Preferences::instance();
      std::auto_ptr<AbstractOutputter> outputter =
        AbstractOutputterFactory::create(pref.outputFormat(), std::cout);
      
      outputter->writeNewBug(repository().addBug(message_.subject().c_str(),
                                                 message_.body().c_str() ));
    }
  }
}
