// -*- mode:c++; indent-tabs-mode:nil; tab-width:2;buffer-file-coding-system:euc-jp-unix; -*-
/**
   @file
   @brief assign command

   @author seagull
   @version "$Id: assign.cpp,v 1.1.1.1 2004/01/12 12:19:12 seagull Exp $"
 */
#include "../common.h"

#include <cstdlib>
#include <iostream>
#include <iomanip>

#include <boost/bind.hpp>
#include <boost/mem_fn.hpp>

#include "../charsets.h"
#include "../log.h"
#include "../command.h"
#include "assign.h"

namespace bugxee {
  namespace command {

    AbstractCommandFactory const*
    getAssignCommandFactory()
    {
      static GenericCommandFactory<AssignCmd>
        factory("assign",
                _("Assign to mentainner."),
                _("Usage: bugxee [global options...] assign [-aMAINTENER] [-sSTATUS] [-pPRIORITY] [-cCATEGORY] [-CCATEGORY] BUG#...\n"
                  "\n"
                  "  Assign bug properties.\n"
                  "\n"
                  "Options:\n"
                  "  -aMAINTENER     Assign maintener.\n"
                  "  -sSTATUS        Change bug status.\n"
                  "  -pPRIORITY      Change bug priority.\n"
                  "  -cCATEGORY      Add category to bug.\n"
                  "  -CCATEGORY      Remove category from bug.\n"
                  "  Any of -m, -s, -c, -C and -p or at least one must specify.\n"
                  "e.g\n"
                  "  BUG#1 and 4 asignes to alice and switch to open.\n"
                  "      $ bugxee assign -malice -sopening 1 4\n"
                ));
      return &factory;
    }


    /**
       Constractor of assign command.
       @param repository Target repository
     */
    AssignCmd::AssignCmd(Repository& repository)
      : AbstractCommand(repository)
    {
      maintener_.clear();
      status_.clear();
      priority_ = 0;
      addedCategory_.clear();
      removedCategory_.clear();
    }


    /**
       Parse command secific options.
       @param argc ARGC
       @param argv ARGV
     */
    void
    AssignCmd::parseOptions(int argc, char* argv[])
    {
      while (argc > 0)
        {
          if (argv[0][0] == '-')
            {
              switch (argv[0][1])
                {
                case 's':
                  status_ = argv[0] + 2;
                  break;
                case 'a':
                  maintener_ = argv[0] + 2;
                  break;
                case 'p':
                  priority_ = atoi(argv[0] + 2);
                  break;
                case 'c':
                  addedCategory_ = argv[0] + 2;
                  break;
                case 'C':
                  removedCategory_ = argv[0] + 2;
                  break;
                default:
                  throw UnknownOptionError(argv[0]);
                }
            }
          else
            targets_.insert(std::atoi(argv[0]));
          
          argc--;
          argv++;
        }

      if (targets_.empty())
        throw InvalidOptionValue(_("Must specify one or more BUG#"));

      //
      if (maintener_.empty() && status_.empty() &&
          addedCategory_.empty() && removedCategory_.empty() && 
          priority_ == 0)
        throw std::invalid_argument(_("Any of -a, -s, -c and -p or at least one must specify."));

      if (! status_.empty() && ! repository().isValidStatusName(status_))
        throw InvalidOptionValue(_("Unknown status"));
      if (priority_ && ! repository_.isValidPriority(priority_))
        throw InvalidOptionValue(_("Invalid priority"));
      if (! maintener_.empty() && ! repository().isValidMaintener(maintener_))
        throw InvalidOptionValue(_("Unknown maintener"));
      if (! addedCategory_.empty() && ! repository().isValidCategory(addedCategory_))
        throw InvalidOptionValue(_("Unknown category"));
      if (! removedCategory_.empty() && ! repository().isValidCategory(removedCategory_))
        throw InvalidOptionValue(_("Unknown category"));

      if ((! removedCategory_.empty() || ! addedCategory_.empty()) &&
          removedCategory_ == addedCategory_)
        throw InvalidOptionValue(_("What? could'nt add and remove with same category."));
    }



    /**
       Perform assign command.
    */
    void
    AssignCmd::perform()
    {
      BugItem::Jornal_t jornal;
      jornal.stamp();
      jornal.maintener_ = maintener_;
      jornal.status_ = status_;
      jornal.priority_ = priority_;
      jornal.addedCategory_ = addedCategory_;
      jornal.removedCategory_ = removedCategory_;

      std::for_each(targets_.begin(), targets_.end(),
                    boost::bind(&Repository::addJornal,
                                boost::ref(repository()), _1, boost::ref(jornal)) );
    }
  }
}
