// -*- mode:c++; indent-tabs-mode:nil; tab-width:2;buffer-file-coding-system:euc-jp-unix; -*-
/**
   @file
   @brief list command

   @author seagull
   @version "$Id: list.cpp,v 1.2 2004/01/14 04:23:55 seagull Exp $"
 */
#include "common.h"

#include <iostream>
//#include <iomanip>

#include <boost/bind.hpp>

#include "Exception.h"
#include "charsets.h"
#include "log.h"
#include "BugItem.h"
#include "command.h"
#include "Outputter.h"
#include "Preferences.h"
#include "commands/list.h"

namespace bugxee {
  namespace command {


    AbstractCommandFactory const* getListCommandFactory()
    {
      static GenericCommandFactory<ListCmd>
        factory("list",
                _("List bug summaries."),
                _("Usage: bugxee [global options...] list [-jSTART] [-nCOUNT] [-sMAINTENER] [-cCATEGORY] STATUSES ... \n"
                  "\n"
                  "  List all categories if no category specified.\n"
                  "\n"
                  "Options:\n"
                  "  -jSKIP         Skip bugs to output.\n"
                  "  -nCOUNT        Output COUNT bugs.\n"
                  "  -mMAINTENER    Filter by maintener.  (No supported yet)\n"
                  "  -pPRIORITY     Filter by priority. (*1)\n"
                  "  -cCATEGORY     Filter by category. (*1)\n"
                  " (*1): Can specify multiple.\n"
                  "\n"
                  "  e.g.\n"
                  "   To print incomming bugs.\n"
                  "        $ bugxee list incomming\n"
                  "   To print incomming and opening bugs. \n"
                  "        $ bugxee list incomming opening  ... or\n"
                  "        $ bugxee list"
                  "   To print 20 bugs in page 3(20 bugs par page)\n"
                  "        $ bugxee list -s40 -c20\n"
                  "   To print bugs assigned to alice and bob.\n"
                  "        $ bugxee list -malice -mbob"
                  ));
      return &factory;
    }

    /**
       Constractor
       @param repository Target repository
     */
    ListCmd::ListCmd(Repository& repository)
      : AbstractCommand(repository)
    {
      statuses_.push_back("incomming");
      statuses_.push_back("opening");
    }


    /**
       Parse command line arguments
       @param argc ARGC
       @param argv ARGV
     */
    void
    ListCmd::parseOptions(int argc, char* argv[])
    {
      bool statusFilterSpecified = false;

      skip_ = 0;
      count_ = 0;

      while (argc > 0)
        {
          if (argv[0][0] == '-')
            {
              switch (argv[0][1])
                {
                case 'j':
                  skip_ = atoi(argv[0] + 2);
                  break;
                case 'n':
                  count_ = atoi(argv[0] + 2);
                  break;
                case 'm':
                  if (! repository().isValidMaintener(argv[0] + 2))
                    throw InvalidOptionValue(_("Unknown maintener"));
                  maintener_ = argv[0] + 2;
                  break;
                case 'p':
                  if (! argv[0][2])
                    throw InvalidOptionValue("-p");
                  {
                    unsigned int n = std::atoi(argv[0] + 2);
                    if (! repository().isValidPriority(n))
                      throw InvalidOptionValue(_("Invalid priority"));
                    priorities_.insert(n);
                  }
                  break;
                case 'c':
                  if (! repository().isValidCategory(argv[0] + 2))
                    throw InvalidOptionValue(_("Unknown category"));
                  categories_.insert(argv[0] + 2);
                  break;
                default:
                  throw UnknownOptionError(argv[0]);
                }
            }
          else
            {
              if (! statusFilterSpecified)
                statuses_.clear();
              statusFilterSpecified = true;
              statuses_.push_back(argv[0]);
            }

          argc--;
          argv++;
        }
    }



    /**
       Perform list command.
     */
    void
    ListCmd::perform()
    {
      //
      // Merge index for each statuses.
      //
      Repository::Index_t idx;
      for (std::list<std::string>::iterator j = statuses_.begin();
           j != statuses_.end(); j++)
      {
        Repository::Index_t& subset = repository().index("status", j->c_str());
        idx.insert(subset.begin(), subset.end());
      }

      // Filter by maintener
      if (! maintener_.empty())
        {
          Repository::Index_t& subset = repository().index("maintener", maintener_.c_str());
          Repository::Index_t tmp(idx);
          idx.clear();
          std::set_intersection(tmp.begin(), tmp.end(), subset.begin(), subset.end(),
                                std::inserter(idx, idx.end()));
        }

      // Filter by priority
      if (! priorities_.empty())
        {
          Repository::Index_t filteridx;
          for (std::set<int>::iterator j = priorities_.begin();
               j != priorities_.end(); j++)
            {
              char tmp[10];
              sprintf(tmp, "%d", *j);
              Repository::Index_t& subset = repository().index("priority", tmp);
              filteridx.insert(subset.begin(), subset.end());
            }
          Repository::Index_t tmp(idx);
          idx.clear();
          std::set_intersection(tmp.begin(), tmp.end(), filteridx.begin(), filteridx.end(),
                                std::inserter(idx, idx.end()));
        }

      // Filter by category
      if (! categories_.empty())
        {
          Repository::Index_t filteridx;
          for (std::set<std::string>::iterator j = categories_.begin();
               j != categories_.end(); j++)
            {
              Repository::Index_t& subset = repository().index("category", j->c_str());
              filteridx.insert(subset.begin(), subset.end());
            }
          Repository::Index_t tmp(idx);
          idx.clear();
          std::set_intersection(tmp.begin(), tmp.end(), filteridx.begin(), filteridx.end(),
                                std::inserter(idx, idx.end()));
        }

      //
      // Create outputter and write out.
      //
      Preferences& pref = Preferences::instance();
      std::auto_ptr<AbstractOutputter> outputter =
        AbstractOutputterFactory::create(pref.outputFormat(), std::cout);

      outputter->setTotalCount(idx.size());
      outputter->start();

      // output search conditions
#if 0
      std::for_each(statuses_.begin(), statuses_.end(),
                    boost::bind(&AbstractOutputter::writeConditionOfStatus,
                                outputter.get(), _1) );
#endif
      if (!maintener_.empty())
        outputter->writeConditionOfMaintener(maintener_);
      std::for_each(priorities_.begin(), priorities_.end(),
                    boost::bind(&AbstractOutputter::writeConditionOfPriority,
                                outputter.get(), _1));
      std::for_each(categories_.begin(), categories_.end(),
                    boost::bind(&AbstractOutputter::writeConditionOfCategory,
                                outputter.get(), _1));

      // output search result
      unsigned int skip = skip_;
      for (std::list<std::string>::iterator i = statuses_.begin();
           i != statuses_.end();
           i++)
        {
          outputter->beginGroup(*i);

          Repository::Index_t const& statusIdx = repository().index("status", *i);
          Repository::Index_t tmpidx;
          std::set_intersection(statusIdx.begin(), statusIdx.end(),
                                idx.begin(), idx.end(),
                                std::inserter(tmpidx, tmpidx.end()) );
          for (Repository::Index_t::const_iterator k = tmpidx.begin();
               k != tmpidx.end(); k++)
            {
              if (skip > 0)
                skip--;
              else if (count_ == 0 || outputter->outputCount() < count_)
                {
                  BugItem* bug = repository().findByBugno(*k);
                  if (bug)
                    outputter->writeList(bug);
                }
            }
          outputter->finishGroup();
        }
      outputter->finish();
    }

  }
}
