// -*- mode:c++; indent-tabs-mode:nil; tab-width:2;buffer-file-coding-system:euc-jp-unix; -*-
/**
   @file
   @brief Module of "post" command
   @author seagull
   @version "$Id: post.cpp,v 1.1.1.1 2004/01/12 12:19:12 seagull Exp $"
 */
#include "../common.h"

#include <boost/bind.hpp>
#include <boost/mem_fn.hpp>
#include <cstring>

#include "../Exception.h"
#include "../log.h"
#include "../Preferences.h"
#include "../editor.h"
#include "../Outputter.h"
#include "../command.h"
#include "post.h"

namespace bugxee {
  namespace command {

    AbstractCommandFactory const* getPostCommandFactory()
    {
      static GenericCommandFactory<PostCmd>
        factory("post",
                _("Post a message to BUG."),
                _("Usage: bugxee [global options...] post [-m MESSAGE] BUG#s...\n"
                  "  If no message specified, editor will be launch.\n"
                  ""
                  "")
                );
      return &factory;
    }


    /**
       Constractr
       @param repository Target repository
     */
    PostCmd::PostCmd(Repository& repository)
      : AbstractCommand(repository)
    {
    }


    /**
       Parse command specific options.
     */
    void
    PostCmd::parseOptions(int argc, char* argv[])
    {
      while (argc > 0)
        {
          if (argv[0][0] == '-')
            {
              switch (argv[0][1])
                {
                case 'm':
                  if (argc <= 1)
                    throw std::runtime_error(_("Empty message"));
                  message_ = std::auto_ptr<char>(strdup(argv[1]));

                  argc--;
                  argv++;
                  break;
                default:
                  throw UnknownOptionError(argv[0]);
                }
            }
          else
            bugnos_.insert(atoi(argv[0]));

          argc--;
          argv++;
        }

      if (message_.get() == NULL)
        for (;;)
          {
            ExternalEditor editor;
            message_ = editor.launch();
              break;
            throw std::invalid_argument(_("No message present.")); // TODO: implements to retry
          }

      if (bugnos_.empty())
        throw InvalidOptionValue(_("Must specify one or more BUG#"));

      if (message_.get() == NULL)
        throw std::runtime_error(_("No message present."));
    }


    /**
       Parform add command.
     */
    void
    PostCmd::perform()
    {
      Preferences& pref = Preferences::instance();
      std::auto_ptr<AbstractOutputter> outputter =
        AbstractOutputterFactory::create(pref.outputFormat(), std::cout);

      std::for_each(bugnos_.begin(), bugnos_.end(),
                    boost::bind(&Repository::addMessage,
                                boost::ref(repository()),
                                _1,
                                message_.get()) );
    }
  }
}
