// -*- mode:c++; indent-tabs-mode:nil; tab-width:2;buffer-file-coding-system:euc-jp-unix; -*-
/**
   @file
   @brief Module of "summary" command
   @author seagull 
   @version "$Id: stat.cpp,v 1.1.1.1 2004/01/12 12:19:12 seagull Exp $"
 */
#include "../common.h"

#include <limits>
#include <numeric>

#include <boost/iterator_adaptors.hpp>
#include <boost/array.hpp>
#include <boost/bind.hpp>
#include <boost/mem_fn.hpp>

#include "../../lib/getdate.h"
#include "../Exception.h"
#include "../log.h"
#include "../editor.h"
#include "../command.h"
#include "../Outputter.h"
#include "../Preferences.h"
#include "stat.h"

namespace bugxee {
  namespace command {

    AbstractCommandFactory const* getStatCommandFactory()
    {
      static GenericCommandFactory<StatCmd>
        factory("stat",
                _("Show statistics."),
                _("Usage: bugxee [global options...] stat\n"
                  "Has no options in this command.")
                );
      return &factory;
    }


    /**
       Constractr
       @param repository Target repository
     */
    StatCmd::StatCmd(Repository& repository)
      : AbstractCommand(repository)
    {
    }


    /**
       Parse command specific options.
     */
    void
    StatCmd::parseOptions(int argc, char* argv[])
    {
      if (argc > 0)
        throw UnknownOptionError(argv[0]);
    }


    /**
       Parform add command.
     */
    void
    StatCmd::perform()
    {
      Repository::Lock locker(repository());

      // Make outputter
      Preferences const& pref = Preferences::instance();
      std::auto_ptr<AbstractOutputter> outputter = 
        AbstractOutputterFactory::create(pref.outputFormat(), std::cout);


      outputter->start();

      struct helper {
        static void bystat(Repository& rep, AbstractOutputter* op, std::string const& status)
        { op->writeStatByStatus(status, count(rep, "status", status.c_str()) ); }
        static void byprio(Repository& rep, AbstractOutputter* op, char const* name)
        { op->writeStatByPriority(name[0] - '0', count(rep, "priority", name) ); }
        static int count(Repository& rep, char const* domain, char const* name)
        { return rep.index(domain, name).size(); }
      };

      // stat by status
      std::for_each(repository().statuses().begin(), repository().statuses().end(),
                    boost::bind(&helper::bystat,
                                boost::ref(repository()), outputter.get(), _1) );

      // stat by priority
      char const names[9][2] = { "1", "2", "3", "4", "5", "6", "7", "8", "9" };
      std::for_each(names, names + 9,
                    boost::bind(&helper::byprio,
                                boost::ref(repository()), outputter.get(), _1) );
      // total
      int total = 
        std::accumulate(boost::make_transform_iterator(names, 
                                                       boost::bind(&helper::count,
                                                                   boost::ref(repository()), "priority", _1)),
                        boost::make_transform_iterator(names + 9,
                                                       boost::bind(&helper::count,
                                                                   boost::ref(repository()), "priority", _1)),
                        0 );
      outputter->writeStatTotal(total);

      outputter->finish();
    }
  }
}
