// -*- mode:c++; indent-tabs-mode:nil; tab-width:2;buffer-file-coding-system:euc-jp-unix; -*-
/**
   @file
   @brief Main module for bugxee.

   @author seagull
   @version "$Id: main.cpp,v 1.1.1.1 2004/01/12 12:19:12 seagull Exp $"
*/
#include "common.h"

#include <stdlib.h>
#include <iostream>
#include <iomanip>
#include <exception>
#include <clocale>

#include <log4cpp/Appender.hh>
#include <log4cpp/OstreamAppender.hh>
#include <log4cpp/BasicLayout.hh>
#include <log4cpp/SimpleLayout.hh>
#include <log4cpp/Priority.hh>


#include "log.h"
#include "Preferences.h"
#include "Repository.h"
#include "LocalRepository.h"
#include "command.h"
#include "Outputter.h"

char const copyright[] =
PACKAGE_NAME " version " PACKAGE_VERSION "\n"
"Copyright 2002, seagull. \n"
"\n"
"Please bug report to " PACKAGE_BUGREPORT "\n";




log4cpp::Category& bugxee::log = log4cpp::Category::getRoot();

/**
   Print usage message to stderr.
 */
void
printUsage()
{
  std::cerr << 
    _("Usage: bugxee [global options...] command [options ...]\n"
      "Global options:\n"
      "  -d DIR       Set repository location to DIR.\n"
      "  -cCHARSET    Set local charset for iconv\n"
      "  -uNAME       Set your name to NAME.\n"
      "  -mADRESS     Set your mail address to ADDRESS.\n"
      "  -oFORMAT     Choose output format.\n"
      "                 -oxml and -otext is available.\n"
      "  -vN          Verbose. N = 0 to 3\n"
      "Commads:\n"
      "  version      Show bugxee version and copyright info.\n"
      "  init         Initialize repository.\n")
    ;
  bugxee::AbstractCommandFactory::FactoryPool const& factories
    = bugxee::AbstractCommandFactory::factories();
  for (bugxee::AbstractCommandFactory::FactoryPool::const_iterator i = factories.begin();
       i != factories.end(); i++)
    std::cerr << "  "
              << i->second->name() << std::setw(13 - ::strlen(i->second->name())) << " "
              << i->second->description() << std::endl;

  std::cerr << "\n" << copyright << std::endl;
}



/**
 */
extern "C"
int
main(int argc, char* argv[])
{
  std::setlocale(LC_ALL, "");
#if defined(NDEBUG)
  bindtextdomain(PACKAGE_NAME, PACKAGE_LOCALE_DIR);
#else
  bindtextdomain(PACKAGE_NAME, ".");
#endif
  textdomain(PACKAGE_NAME);

  //
  // Initialize log outputter
  //
  log4cpp::Appender* appender = new log4cpp::OstreamAppender("stderr", &std::cerr);
  appender->setLayout(new log4cpp::SimpleLayout());
  bugxee::log.setAppender(appender);
#if defined(NDEBUG)
  bugxee::log.setPriority(log4cpp::Priority::ERROR);
#else
  bugxee::log.setPriority(log4cpp::Priority::DEBUG);
#endif


  //
  // Register factories.
  //
  bugxee::registerAllCommands();  // command
  bugxee::AbstractOutputterFactory::registerAllOutputter(); // outputter


  //
  // Load preferences
  //
  bugxee::Preferences* pref;
  try
    {
      pref = &bugxee::Preferences::instance();
      pref->parseOptions(&argc, argv);
      if (pref->userName().empty())
        throw std::runtime_error(_("No user name given."));

    }
  catch (std::exception& e)
    {
      bugxee::log.crit(e.what());
      return EXIT_SUCCESS;
    }

  if (argc < 2)
    {
      bugxee::log.error(_("No command specified."));
      printUsage();
      return EXIT_SUCCESS;
    }
  std::string cmd(argv[1]);

  /*
    Process commands
   */
  if (cmd == "version")
    {
      std::cout << copyright;
      return EXIT_SUCCESS;
    }
  bugxee::log.debugStream() << _("Dump current preferences:\n")
                            << *pref << log4cpp::CategoryStream::ENDLINE;


  try
    {
      bugxee::LocalRepository rep(pref->repository().c_str());
      if (cmd == "init")
        rep.initialize();
      else
        {
          rep.open();
          bugxee::AbstractCommandFactory const* factory =
            bugxee::AbstractCommandFactory::findByName(cmd.c_str());
          if (factory == NULL)
            {
              printUsage();
              bugxee::log.error(_("Unknown command: %s"), cmd.c_str());
            }
          else
            {
              bugxee::AbstractCommand* cmd = factory->create(rep);
              cmd->parseOptions(argc - 2, argv + 2);
              cmd->perform();
              rep.commit();
            }
        }
    }
  catch (std::exception& e)
    {
      bugxee::log.crit(_("exception raised: %s"), e.what());
      return EXIT_SUCCESS;
    }

  return EXIT_SUCCESS;
}

