<?php
// $Id: class.jcode.php,v 1.1 2003/01/21 14:17:56 haruki Exp $
define('JCODE_AUTO', 0);
define('JCODE_EUC', 1);
define('JCODE_SJIS', 2);
define('JCODE_JIS', 3);
define('JCODE_UTF8', 4);
define('JCODE_UNKNOWN', 5);
define('JCODE_ASCII', 6);

define('JCODE_UNKNOWN_CHAR','?');
/**
 * class.jcode - Encoding converter class for Japanese.
 *
 * This class converts the encodings of Japanese character,
 * and is able to handle SJIS, EUC-JP, JIS(ISO-2022-JP), UTF-8.
 * Based on jcode.phps v1.34 by TOMO
 *   http://www.spencernetwork.org/
 * Modified to OO style by Haruki Setoyama
 * class.jcode is free but without any warranty.
 * use this script at your own risk.
 *
 * @author TOMO <groove@spencernetwork.org>
 * @author Haruki Setoyama <haruki@planewave.org>
 */
class jcode {

    /**
     * Converts the encoding of japanese string $str from $from to $to.
	 *
	 * @access 	public
     * @param 	&string	$str
     * @param 	int		$from
     * @param 	int		$to
     * @return 	string
     */
    function convert(&$str, $from, $to)
    {
        static $fname = array(  JCODE_EUC => 'EUC',
                                'EUC-JP' => 'EUC',
                                JCODE_SJIS => 'SJIS',
                                'SJIS' => 'SJIS',
                                'SHIFT-JIS' => 'SJIS',
                                JCODE_JIS => 'JIS',
                                'ISO-2022-JP' => 'JIS',
                                'JIS' => 'JIS',
                                JCODE_UTF8 => 'UTF',
                                'UTF-8' => 'UTF');

        if ($from == JCODE_AUTO || strtolower($from) == 'auto')
            $from = jcode::AutoDetect($str);

        if(isset($fname[$from]) && isset($fname[$to])){
            $function_name = $fname[$from].'to'.$fname[$to];
            return call_user_func( array('jcode', $function_name), $str);
        }
    	return false;
    }

    /**
     * Detects the encoding of Japanese strings.
     *
     * @access public
     * @param &string $str
     * @return int	0:US-ASCII 1:EUC-JP 2:Shift_JIS 3:ISO-2022-JP(JIS) 4:UTF-8 5:Unknown
     */
    function AutoDetect(&$str)
    {
    	if (!ereg("[\x80-\xFF]", $str)) {
    		// --- Check ISO-2022-JP ---
    		if (ereg("\x1B", $str)) return JCODE_JIS; // ISO-2022-JP(JIS)
    		return JCODE_ASCII; //US-ASCII
    	}

    	$b = unpack('C*', ereg_replace("^[^\x80-\xFF]+", "", $str));
    	$n = count($b);
    
    	// --- Check EUC-JP ---
    	$euc = TRUE;
    	for ($i = 1; $i <= $n; ++$i){
    		if ($b[$i] < 0x80) {
    			continue;
    		}
    		if ($b[$i] < 0x8E) {
    			$euc = FALSE; break;
    		}
    		if ($b[$i] == 0x8E) {
    			if (!isset($b[++$i])) {
    				$euc = FALSE; break;
    			}
    			if (($b[$i] < 0xA1) || (0xDF < $b[$i])) {
    				$euc = FALSE; break;
    			}
    		} elseif ((0xA1 <= $b[$i]) && ($b[$i] <= 0xFE)) {
    			if (!isset($b[++$i])) {
    				$euc = FALSE; break;
    			}
    			if (($b[$i] < 0xA1) || (0xFE < $b[$i])) {
    				$euc = FALSE; break;
    			}
    		} else {
    			$euc = FALSE; break;
    		}
    	}
    	if ($euc) return JCODE_EUC; // EUC-JP
    
    	// --- Check UTF-8 ---
    	$utf8 = TRUE;
    	for ($i = 1; $i <= $n; ++$i) {
    		if (($b[$i] < 0x80)) {
    			continue;
    		}
    		if ((0xC0 <= $b[$i]) && ($b[$i] <=0xDF)) {
    			if (!isset($b[++$i])) {
    				$utf8 = FALSE; break;
    			}
    			if (($b[$i] < 0x80) || (0xEF < $b[$i])) {
    				$utf8 = FALSE; break;
    			}
    		} elseif ((0xE0 <= $b[$i]) && ($b[$i] <= 0xEF)) {
    			if (!isset($b[++$i])) {
    				$utf8 = FALSE; break;
    			}
    			if (($b[$i] < 0x80) || (0xBF < $b[$i])) {
    				$utf8 = FALSE; break;
    			}
    			if (!isset($b[++$i])) {
    				$utf8 = FALSE; break;
    			}
    			if (($b[$i] < 0x80) || (0xBF < $b[$i])) {
    				$utf8 = FALSE; break;
    			}
    		} else {
    			$utf8 = FALSE; break;
    		}
    	}
    	if ($utf8) return JCODE_UTF8; // UTF-8
    
    	// --- Check Shift_JIS ---
    	$sjis = TRUE;
    	for ($i = 1; $i <= $n; ++$i) {
    		if (($b[$i] <= 0x80) || (0xA1 <= $b[$i] && $b[$i] <= 0xDF)) {
    			continue;
    		}
    		if (($b[$i] == 0xA0) || ($b[$i] > 0xEF)) {
    			$sjis = FALSE; break;
    		}
    		if (!isset($b[++$i])) {
    			$sjis = FALSE; break;
    		}
    		if (($b[$i] < 0x40) || ($b[$i] == 0x7F) || ($b[$i] > 0xFC)){
    			$sjis = FALSE; break;
    		}
    	}
    	if ($sjis) return JCODE_SJIS; // Shift_JIS
    
    	return JCODE_UNKNOWN; // Unknown
    }

    /********************************************************************
     * to EUC-JP
     ********************************************************************/

    /**
     * jcodeEUC::JIStoEUC()
     *
     * @param &string $str_JIS
     * @return string
     */
    function JIStoEUC(&$str_JIS)
    {
   		$str_EUC = '';
    	$mode = 0;
    	$b = unpack('C*', $str_JIS);
    	$n = count($b);
    
    	for ($i = 1; $i <= $n; ++$i) {
    
    		//Check escape sequence
    		while ($b[$i] == 0x1B) {
    			if (($b[$i+1] == 0x24 && $b[$i+2] == 0x42)
    				|| ($b[$i+1] == 0x24 && $b[$i+2] == 0x40)) {
    				$mode = 1;
    			} elseif (($b[$i+1] == 0x28 && $b[$i+2] == 0x49)) {
    				$mode = 2;
    			} else {
    				$mode = 0;
    			}
    			$i += 3;
    			if (!isset($b[$i])) break 2;
    		}
    
    		//Do convert
    		if ($mode == 1) {
    			$str_EUC .= chr($b[$i] + 0x80).chr($b[++$i] + 0x80);
    		} elseif ($mode == 2) {
    			$str_EUC .= chr(0x8E).chr($b[$i] + 0x80);
    		} else {
    			$str_EUC .= chr($b[$i]);
    		}
    	}
    
    	return $str_EUC;
    }

    /**
     * jcodeEUC::SJIStoEUC()
     *
     * @param &string $str_SJIS
     * @return string
     */
	function SJIStoEUC(&$str_SJIS)
    {
		$b = unpack('C*', $str_SJIS);
    	$n = count($b);
    	$str_EUC = '';
    
    	for ($i = 1; $i <= $n; ++$i) {
    		$b1 = $b[$i];
    		if (0xA1 <= $b1 && $b1 <= 0xDF) {
    			$str_EUC .= chr(0x8E).chr($b1);
    		} elseif ($b1 >= 0x81) {
    			$b2 = $b[++$i];
    			$b1 <<= 1;
    			if ($b2 < 0x9F) {
    				if ($b1 < 0x13F) $b1 -= 0x61; else $b1 -= 0xE1;
    				if ($b2 > 0x7E)  $b2 += 0x60; else $b2 += 0x61;
    			} else {
    				if ($b1 < 0x13F) $b1 -= 0x60; else $b1 -= 0xE0;
    				$b2 += 0x02;
    			}
    			$str_EUC .= chr($b1).chr($b2);
    		} else {
    			$str_EUC .= chr($b1);
    		}
    	}
    
    	return $str_EUC;
    }

    /**
     * jcodeEUC::UTF8toEUC()
     *
     * @param &string $str_UTF8
     * @return string
     */
	function UTF8toEUC(&$str_UTF8)
    {
        static $table_utf8_jis;
		if (!isset($table_utf8_jis)) {
			include('code_table.ucs2jis.php');
		}

    	$str_EUC = '';
    	$b = unpack('C*', $str_UTF8);
    	$n = count($b);
    
    	for ($i = 1; $i <= $n; $i++) {
    		if ($b[$i] >= 0x80) { //Not ASCII
    			if ($b[$i] <= 0xDF) { //2 Bytes
    				$utf8 = ($b[$i++] << 8) + $b[$i];
    			} else { //3 Bytes
    				$utf8 = ($b[$i++] << 16) + ($b[$i++] << 8) + $b[$i];
    			}
    			if (isset($table_utf8_jis[$utf8])) {
    				$jis = $table_utf8_jis[$utf8];
    				if ($jis < 0xFF) { //Hankaku
    					$str_EUC .= chr(0x8E).chr($jis - 0x80);
    				} else { //Zenkaku
    					$str_EUC .= chr(($jis >> 8) - 0x80).chr(($jis & 0xFF) - 0x80);
    				}
    			} else { //Unknown
    				$str_EUC .= JCODE_UNKNOWN_CHAR;
    			}
    		} else { //ASCII
    			$str_EUC .= chr($b[$i]);
    		}
    	}

    	return $str_EUC;
    }

    /********************************************************************
     * to Shift-JIS
     ********************************************************************/

    /**
     * jcode::EUCtoSJIS()
     *
     * @param &string $str_EUC
     * @return string
     */
    function EUCtoSJIS(&$str_EUC)
    {

        $str_SJIS = '';
        $b = unpack('C*', $str_EUC);
        $n = count($b);

        for ($i = 1; $i <= $n; ++$i) {
            $b1 = $b[$i];
            if ($b1 > 0x8E) {
                $b2 = $b[++$i];
                if ($b1 & 0x01) {
                    $b1 >>= 1;
                    if ($b1 < 0x6F) $b1 += 0x31; else $b1 += 0x71;
                    if ($b2 > 0xDF) $b2 -= 0x60; else $b2 -= 0x61;
                } else {
                    $b1 >>= 1;
                    if ($b1 <= 0x6F) $b1 += 0x30; else $b1 += 0x70;
                    $b2 -= 0x02;
                }
                $str_SJIS .= chr($b1).chr($b2);
            } elseif ($b1 == 0x8E) {
                $str_SJIS .= chr($b[++$i]);
            } else {
                $str_SJIS .= chr($b1);
            }
        }
    
        return $str_SJIS;
    }

    /**
     * jcode::JIStoSJIS()
     *
     * @param &string $str_JIS
     * @return string
     */
    function JIStoSJIS(&$str_JIS)
    {
        $str_SJIS = '';
    	$mode = 0;
    	$b = unpack('C*', $str_JIS);
    	$n = count($b);
    
    	for ($i = 1; $i <= $n; ++$i) {
    
    		//Check escape sequence
    		while ($b[$i] == 0x1B) {
    			if (($b[$i+1] == 0x24 && $b[$i+2] == 0x42)
    				|| ($b[$i+1] == 0x24 && $b[$i+2] == 0x40)) {
    				$mode = 1;
    			} elseif (($b[$i+1] == 0x28 && $b[$i+2] == 0x49)) {
    				$mode = 2;
    			} else {
    				$mode = 0;
    			}
    			$i += 3;
    			if (!isset($b[$i])) break 2;
    		}
    
    		//Do convert
    		if ($mode == 1) {
    			$b1 = $b[$i];
    			$b2 = $b[++$i];
    			if ($b1 & 0x01) {
    				$b1 >>= 1;
    				if ($b1 < 0x2F) $b1 += 0x71; else $b1 -= 0x4F;
    				if ($b2 > 0x5F) $b2 += 0x20; else $b2 += 0x1F;
    			} else {
    				$b1 >>= 1;
    				if ($b1 <= 0x2F) $b1 += 0x70; else $b1 -= 0x50;
    				$b2 += 0x7E;
    			}
    			$str_SJIS .= chr($b1).chr($b2);
    		} elseif ($mode == 2) {
    			$str_SJIS .= chr($b[$i] + 0x80);
    		} else {
    			$str_SJIS .= chr($b[$i]);
    		}
    	}
    
    	return $str_SJIS;
    }

    /**
     * jcode::UTF8toSJIS()
     *
     * @param &string $str_UTF8
     * @return string
     */
    function UTF8toSJIS(&$str_UTF8)
    {
        static $table_utf8_jis;
        if (!isset($table_utf8_jis)) {
            include('code_table.ucs2jis.php');
        }

        $str_SJIS = '';
        $b = unpack('C*', $str_UTF8);
        $n = count($b);

        for ($i = 1; $i <= $n; ++$i) {
            if ($b[$i] >= 0x80) { //Not ASCII
                if ($b[$i] <= 0xDF) { //2 Bytes
                    $utf8 = ($b[$i] << 8) + $b[++$i];
                } else { //3 Bytes
                    $utf8 = ($b[$i] << 16) + ($b[++$i] << 8) + $b[++$i];
                }
                if (isset($table_utf8_jis[$utf8])) {
                    $jis = $table_utf8_jis[$utf8];
                    if ($jis < 0xFF) { //Hankaku
                        $str_SJIS .= chr($jis + 0x80);
                    } else { //Zenkaku
                        $b1 = $jis >> 8;
                        $b2 = $jis & 0xFF;
                        if ($b1 & 0x01) {
                            $b1 >>= 1;
                            if ($b1 < 0x2F) $b1 += 0x71; else $b1 -= 0x4F;
                            if ($b2 > 0x5F) $b2 += 0x20; else $b2 += 0x1F;
                        } else {
                            $b1 >>= 1;
                            if ($b1 <= 0x2F) $b1 += 0x70; else $b1 -= 0x50;
                            $b2 += 0x7E;
                        }
                        $str_SJIS .= chr($b1).chr($b2);
                    }
                } else { //Unknown
                    $str_SJIS .= JCODE_UNKNOWN_CHAR;
                }
            } else { //ASCII
                $str_SJIS .= chr($b[$i]);
            }
        }
    
        return $str_SJIS;
    }

    /********************************************************************
     * to JIS (ISO-2022-JP)
     ********************************************************************/

    /**
     * jcode::EUCtoJIS()
     *
     * @param &string $str_EUC
     * @return string
     */
    function EUCtoJIS(&$str_EUC)
    {

        $str_JIS = '';
        $mode = 0;
        $b = unpack('C*', $str_EUC);
        $n = count($b);

        //Escape sequence
        $ESC = array(chr(0x1B).chr(0x28).chr(0x42),
                 chr(0x1B).chr(0x24).chr(0x42),
                 chr(0x1B).chr(0x28).chr(0x49));

        for ($i = 1; $i <= $n; ++$i) {
            $b1 = $b[$i];
            if ($b1 == 0x8E) {
                if ($mode != 2) {
                    $mode = 2;
                    $str_JIS .= $ESC[$mode];
                }
                $str_JIS .= chr($b[++$i] - 0x80);
            } elseif ($b1 > 0x8E) {
                if ($mode != 1) {
                    $mode = 1;
                    $str_JIS .= $ESC[$mode];
                }
                $str_JIS .= chr($b1 - 0x80).chr($b[++$i] - 0x80);
            } else {
                if ($mode != 0) {
                    $mode = 0;
                    $str_JIS .= $ESC[$mode];
                }
                $str_JIS .= chr($b1);
            }
        }
        if ($mode != 0) $str_JIS .= $ESC[0];
    
        return $str_JIS;
    }

    /**
     * jcode::SJIStoJIS()
     *
     * @param &string $str_SJIS
     * @return string
     */
    function SJIStoJIS(&$str_SJIS)
    {
        $str_JIS = '';
    	$mode = 0;
    	$b = unpack('C*', $str_SJIS);
    	$n = count($b);
    
    	//Escape sequence
    	$ESC = array(chr(0x1B).chr(0x28).chr(0x42),
    		     chr(0x1B).chr(0x24).chr(0x42),
    		     chr(0x1B).chr(0x28).chr(0x49));
    
    	for ($i = 1; $i <= $n; ++$i) {
    		$b1 = $b[$i];
    		if (0xA1 <= $b1 && $b1 <= 0xDF) {
    			if ($mode != 2) {
    				$mode = 2;
    				$str_JIS .= $ESC[$mode];
    			}
    			$str_JIS .= chr($b1 - 0x80);
    		} elseif ($b1 >= 0x80) {
    			if ($mode != 1) {
    				$mode = 1;
    				$str_JIS .= $ESC[$mode];
    			}
    			$b2 = $b[++$i];
    			$b1 <<= 1;
    			if ($b2 < 0x9F) {
    				if ($b1 < 0x13F) $b1 -= 0xE1; else $b1 -= 0x61;
    				if ($b2 > 0x7E)  $b2 -= 0x20; else $b2 -= 0x1F;
    			} else {
    				if ($b1 < 0x13F) $b1 -= 0xE0; else $b1 -= 0x60;
    				$b2 -= 0x7E;
    			}
    			$str_JIS .= chr($b1).chr($b2);
    		} else {
    			if ($mode != 0) {
    				$mode = 0;
    				$str_JIS .= $ESC[$mode];
    			}
    			$str_JIS .= chr($b1);
    		}
    	}
    	if ($mode != 0) $str_JIS .= $ESC[0];
    
    	return $str_JIS;
    }

    /**
     * jcode::UTF8toJIS()
     *
     * @param &string $str_UTF8
     * @return string
     */
    function UTF8toJIS(&$str_UTF8)
    {
        static $table_utf8_jis;
        if (!isset($table_utf8_jis)) {
            include('code_table.ucs2jis.php');
        }

        $str_JIS = '';
        $mode = 0;
        $b = unpack('C*', $str_UTF8);
        $n = count($b);
    
        //Escape sequence
        $ESC = array(chr(0x1B).chr(0x28).chr(0x42),
                 chr(0x1B).chr(0x24).chr(0x42),
                 chr(0x1B).chr(0x28).chr(0x49));
    
        for ($i = 1; $i <= $n; ++$i) {
            if ($b[$i] >= 0x80) { //Not ASCII
                if ($b[$i] <= 0xDF) { //2 Bytes
                    $utf8 = ($b[$i] << 8) + $b[++$i];
                } else { //3 Bytes
                    $utf8 = ($b[$i] << 16) + ($b[++$i] << 8) + $b[++$i];
                }
                if (isset($table_utf8_jis[$utf8])) {
                    $jis = $table_utf8_jis[$utf8];
                    if ($jis < 0xFF) { //Hankaku
                        if ($mode != 2) {
                            $mode = 2;
                            $str_JIS .= $ESC[$mode];
                        }
                        $str_JIS .= chr($jis);
                    } else { //Zenkaku
                        if ($mode != 1) {
                            $mode = 1;
                            $str_JIS .= $ESC[$mode];
                        }
                        $str_JIS .= chr($jis >> 8).chr($jis & 0xFF);
                    }
                } else { //Unknown
                    if ($mode != 0) {
                        $mode = 0;
                        $str_JIS .= $ESC[$mode];
                    }
                    $str_JIS .= JCODE_UNKNOWN_CHAR;
                }
            } else { //ASCII
                if ($mode != 0) {
                    $mode = 0;
                    $str_JIS .= $ESC[$mode];
                }
                $str_JIS .= chr($b[$i]);
            }
        }
        if ($mode != 0) $str_JIS .= $ESC[0];
    
        return $str_JIS;
    }

    /********************************************************************
     * to UTF-8
     ********************************************************************/

    /**
     * jcode::EUCtoUTF8()
     *
     * @param &string $str_EUC
     * @return string
     */
    function EUCtoUTF8(&$str_EUC)
    {
        static $table_jis_utf8;
        if (!isset($table_jis_utf8)) {
            include('code_table.jis2ucs.php');
        }

        $str_UTF8 = '';
        $b = unpack('C*', $str_EUC);
        $n = count($b);
    
        for ($i = 1; $i <= $n; ++$i) {
            if ($b[$i] == 0x8E) { //Hankaku
                $b2 = $b[++$i] - 0x40;
                $u2 = 0xBC | (($b2 >> 6) & 0x03);
                $u3 = 0x80 | ($b2 & 0x3F);
                $str_UTF8 .= chr(0xEF).chr($u2).chr($u3);
            } elseif ($b[$i] >= 0x80) { //Zenkaku
                $jis = (($b[$i] - 0x80) << 8) + ($b[++$i] - 0x80);
                if (isset($table_jis_utf8[$jis])) {
                    $utf8 = $table_jis_utf8[$jis];
                    if ($utf8 < 0xFFFF) {
                        $str_UTF8 .= chr($utf8 >> 8).chr($utf8);
                    } else {
                        $str_UTF8 .= chr($utf8 >> 16).chr($utf8 >> 8).chr($utf8);
                    }
                } else { //Unknown
                    $str_UTF8 .= JCODE_UNKNOWN_CHAR;
                }
            } else { //ASCII
                $str_UTF8 .= chr($b[$i]);
            }
        }
    
        return $str_UTF8;
    }

    /**
     * jcode::SJIStoUTF8()
     *
     * @param &string $str_SJIS
     * @return string
     */
    function SJIStoUTF8(&$str_SJIS)
    {
        static $table_jis_utf8;
        if (!isset($table_jis_utf8)) {
            include('code_table.jis2ucs.php');
        }

    	$str_UTF8 = '';
    	$b = unpack('C*', $str_SJIS);
    	$n = count($b);
    
    	for ($i = 1; $i <= $n; ++$i) {
    		if (0xA1 <= $b[$i] && $b[$i] <= 0xDF) { //Hankaku
    			$b2 = $b[$i] - 0x40;
    			$u2 = 0xBC | (($b2 >> 6) & 0x03);
    			$u3 = 0x80 | ($b2 & 0x3F);
    			$str_UTF8 .= chr(0xEF).chr($u2).chr($u3);
    		} elseif ($b[$i] >= 0x80) { //Zenkaku
    			$b1 = $b[$i] << 1;
    			$b2 = $b[++$i];
    			if ($b2 < 0x9F) {
    				if ($b1 < 0x13F) $b1 -= 0xE1; else $b1 -= 0x61;
    				if ($b2 > 0x7E)  $b2 -= 0x20; else $b2 -= 0x1F;
    			} else {
    				if ($b1 < 0x13F) $b1 -= 0xE0; else $b1 -= 0x60;
    				$b2 -= 0x7E;
    			}
    			$b1 &= 0xFF;
    			$jis = ($b1 << 8) + $b2;
    			if (isset($table_jis_utf8[$jis])) {
    				$utf8 = $table_jis_utf8[$jis];
    				if ($utf8 < 0xFFFF) {
    					$str_UTF8 .= chr($utf8 >> 8).chr($utf8);
    				} else {
    					$str_UTF8 .= chr($utf8 >> 16).chr($utf8 >> 8).chr($utf8);
    				}
    			} else {
    				$str_UTF8 .= JCODE_UNKNOWN_CHAR; //Unknown
    			}
    		} else { //ASCII
    			$str_UTF8 .= chr($b[$i]);
    		}
    	}
    
    	return $str_UTF8;
    }
    
    /**
     * jcode::JIStoUTF8()
     *
     * @param &string $str_JIS
     * @return string
     */
    function JIStoUTF8(&$str_JIS)
    {
        static $table_jis_utf8;
        if (!isset($table_jis_utf8)) {
            include('code_table.jis2ucs.php');
        }

    	$str_UTF8 = '';
    	$mode = 0;
    	$b = unpack('C*', $str_JIS);
    	$n = count($b);
    
    	for ($i = 1; $i <= $n; ++$i) {
    
    		//Check escape sequence
    		while ($b[$i] == 0x1B) {
    			if (($b[$i+1] == 0x24 && $b[$i+2] == 0x42)
    				|| ($b[$i+1] == 0x24 && $b[$i+2] == 0x40)) {
    				$mode = 1;
    			} elseif ($b[$i+1] == 0x28 && $b[$i+2] == 0x49) {
    				$mode = 2;
    			} else {
    				$mode = 0;
    			}
    			$i += 3;
    			if (!isset($b[$i])) break 2;
    		}
    
    		if ($mode == 1) { //Zenkaku
    			$jis = ($b[$i] << 8) + $b[++$i];
    			if (isset($table_jis_utf8[$jis])) {
    				$utf8 = $table_jis_utf8[$jis];
    				if ($utf8 < 0xFFFF) {
    					$str_UTF8 .= chr($utf8 >> 8).chr($utf8);
    				} else {
    					$str_UTF8 .= chr($utf8 >> 16).chr($utf8 >> 8).chr($utf8);
    				}
    			} else { //Unknown
    				$str_UTF8 .= JCODE_UNKNOWN_CHAR;
    			}
    		} elseif ($mode == 2) { //Hankaku
    			$b2 = $b[$i] + 0x40;
    			$u2 = 0xBC | (($b2 >> 6) & 0x03);
    			$u3 = 0x80 | ($b2 & 0x3F);
    			$str_UTF8 .= chr(0xEF).chr($u2).chr($u3);
    		} else { //ASCII
    			$str_UTF8 .= chr($b[$i]);
    		}
    	}
    
    	return $str_UTF8;
    }

}
?>