/*
 * This is sample program for CABI system.
 * create accounting object.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <sched.h>
#include <linux/config.h>
#ifdef CONFIG_X86

#include <cabi/cabi_error.h>
#include "common.h"

#define ERR_MSG "Usage: cstest "
#define INIT_COUNT	10 	/* ꤵ뤿˺ǽ˰٥ߡǼ¹Ԥ */
#define TVALMAX 0x7FFFFFFFFFFFFFFFLL


#define RDTSC(X) __asm__ __volatile__ ("rdtsc" : "=A" (X))

#define CS_CREATE		0
#define CS_DESTROY		1
#define CS_BIND_PID		2
#define CS_UNBIND		3
#define CS_BIND_PGID		4
#define CS_GET			5
#define CS_SET			6
#define SYSCALL_FORK		7
#define SYSCALL_GETPID		8
#define BUSYWAIT		9
#define SYSCALL_FORKCB		10
#define END_OF_SYSCALL		11

#define OPT_COUNT	END_OF_SYSCALL
#define LOOPCOUNT	(OPT_COUNT+1)
#define RR		(LOOPCOUNT+1)
#define FIFO		(RR+1)
#define CSV		(FIFO+1)

char *name[END_OF_SYSCALL] = {
	"cabi_account_create",
	"cabi_account_destroy",
	"cabi_account_bind_pid",
	"cabi_account_unbind",
	"cabi_account_bind_pgid",
	"cabi_account_get",
	"cabi_account_set",
	"fork",
	"getpid",
	"busywait",
	"forkcb",
};
long long sum[END_OF_SYSCALL];
long long min[END_OF_SYSCALL];
long long max[END_OF_SYSCALL];
unsigned long long *ctlog[END_OF_SYSCALL];
long long tsc_overhead;

static struct cabi_uaccount ucabi, *ucabip;
static pid_t pid;
static pid_t pgid;

static void calibration()
{
	unsigned long long start, end, tval, min = 0;
	int i = 0, count = 10000;

	RDTSC(start);
	RDTSC(end);
	min = end - start;
	
	for ( i = 0; i < count; i++ ){
		RDTSC(start);
		RDTSC(end);
		tval = end - start;
		if ( tval < min )
			min = tval;
	}
	for ( i = 0; i < count; i++ ){
		RDTSC(start);
		RDTSC(end);
		tval = end - start;
		if ( tval < min )
			min = tval;
	}
	tsc_overhead = min;
}

static void clearLog(int no, int count)
{
	int i;
	
	sum[no] = 0;
	min[no] = TVALMAX;
	max[no] = 0;
	for ( i = 0; i < count; i++ ){
		ctlog[no][i] = 0;
	}
}

static void setMinMaxLog(int no, int count, unsigned long long tval)
{
	if ( min[no] > tval )
		min[no] = tval;
	if ( max[no] < tval )
		max[no] = tval;
	ctlog[no][count] = tval;
}

static inline long long diff(unsigned long long start, unsigned long long end)
{
	return end - start - tsc_overhead;
}


static void create_destroy(int count)
{
	int i, ret;
	unsigned long long start, end, tval;
	
	clearLog(CS_CREATE, count);
	clearLog(CS_DESTROY, count);
	
	for ( i = 0; i < count; i++ ){
		/* create */
		RDTSC(start);
		ret = cabi_account_create (ucabip);
		RDTSC(end);
		tval = diff(start, end);
		sum[CS_CREATE] += tval;
		setMinMaxLog(CS_CREATE, i, tval);
		if ( ret != CABI_SUCCESS ){
			cabi_result(ret, 1);
		}
		
		/* destroy */
		RDTSC(start);
		ret = cabi_account_destroy(ucabip->cabi_id);
		RDTSC(end);
		tval = diff(start, end);
		sum[CS_DESTROY] += tval;
		setMinMaxLog(CS_DESTROY, i, tval);
		if ( ret != CABI_SUCCESS ){
			cabi_result(ret, 1);
		}
	}
}

static void bind_pid(int count)
{
	int i, ret;
	unsigned long long start, end, tval;
	
	clearLog(CS_BIND_PID, count);
	clearLog(CS_UNBIND, count);
	
	ret = cabi_account_create (ucabip);
	if ( ret != CABI_SUCCESS ){
		cabi_result(ret, 1);
	}
	
	for ( i = 0; i < count; i++ ){
		/* bind */
		RDTSC(start);
		ret = cabi_account_bind_pid(ucabip->cabi_id, pid);
		RDTSC(end);
		tval = diff(start, end);
		sum[CS_BIND_PID] += tval;
		setMinMaxLog(CS_BIND_PID, i, tval);
		if ( ret != CABI_SUCCESS ){
			cabi_result(ret, 1);
		}

		/* unbind */
		RDTSC(start);
		ret = cabi_account_unbind(pid);
		RDTSC(end);
		tval = diff(start, end);
		sum[CS_UNBIND] += tval;
		setMinMaxLog(CS_UNBIND, i, tval);
		if ( ret != CABI_SUCCESS ){
			cabi_result(ret, 1);
		}
		
	}
	ret = cabi_account_destroy(ucabip->cabi_id);
	if ( ret != CABI_SUCCESS ){
		cabi_result(ret, 1);
	}
}

static void bind_pgid(int count)
{
	int i, j, ret;
	unsigned long long start, end, tval;
	unsigned int pidcount=0, *pidlist=NULL;
	
	clearLog(CS_BIND_PGID, count);
	
	ret = cabi_account_create (ucabip);
	if ( ret != CABI_SUCCESS ){
		cabi_result(ret, 1);
	}
	
	for ( i = 0; i < count; i++ ){
		/* bind */
		RDTSC(start);
		ret = cabi_account_bind_pgid(ucabip->cabi_id, pgid);
		RDTSC(end);
		tval = diff(start, end);
		sum[CS_BIND_PGID] += tval;
		setMinMaxLog(CS_BIND_PGID, i, tval);
		if ( ret != CABI_SUCCESS ){
			cabi_result(ret, 1);
		}

		/* unbind */
		if ( !pidlist ){
			ret = cabi_get_bind_pid(ucabip->cabi_id, &pidcount, NULL);
			if ( ret != CABI_SUCCESS ){
				cabi_result(ret, 1);
			}
			pidlist = malloc(sizeof(unsigned int)*pidcount);
			ret = cabi_get_bind_pid(ucabip->cabi_id, &pidcount, pidlist);
			if ( ret != CABI_SUCCESS ){
				cabi_result(ret, 1);
			}
		}
		for ( j = 0; j < pidcount; j++ ){
			ret = cabi_account_unbind(pidlist[j]);
		}
	}
	ret = cabi_account_destroy(ucabip->cabi_id);
	if ( ret != CABI_SUCCESS ){
		cabi_result(ret, 1);
	}
}

static void get(int count)
{
	int i, ret;
	unsigned long long start, end, tval;
	struct cabi_uaccount tmpcabi;
	
	clearLog(CS_GET, count);

	/* create cabi */
	ret = cabi_account_create (ucabip);
	if ( ret != CABI_SUCCESS ){
		cabi_result(ret, 1);
	}

	/* bind process */
	ret = cabi_account_bind_pid(ucabip->cabi_id, pid);
	if ( ret != CABI_SUCCESS ){
		cabi_result(ret, 1);
	}
	
	for ( i = 0; i < count; i++ ){
		RDTSC(start);
		ret = cabi_account_get(ucabip->cabi_id, &tmpcabi);
		RDTSC(end);
		tval = diff(start, end);
		sum[CS_GET] += tval;
		setMinMaxLog(CS_GET, i, tval);
		if ( ret != CABI_SUCCESS ){
			cabi_result(ret, 1);
		}
		
	}

	/* unbind process */
	ret = cabi_account_unbind(pid);

	/* destroy cabi */
	ret = cabi_account_destroy(ucabip->cabi_id);
	if ( ret != CABI_SUCCESS ){
		cabi_result(ret, 1);
	}
}

static void set(int count)
{
	int i, ret;
	unsigned long long start, end, tval;
	
	clearLog(CS_SET, count);
	
	/* create cabi */
	ret = cabi_account_create (ucabip);
	if ( ret != CABI_SUCCESS ){
		cabi_result(ret, 1);
	}

	/* bind process */
	ret = cabi_account_bind_pid(ucabip->cabi_id, pid);
	if ( ret != CABI_SUCCESS ){
		cabi_result(ret, 1);
	}
	
	/* ߤƤ */
	ret = cabi_account_get(ucabip->cabi_id, ucabip);
	
	for ( i = 0; i < count; i++ ){
		RDTSC(start);
		ret = cabi_account_set(ucabip->cabi_id, ucabip);
		RDTSC(end);
		tval = diff(start, end);
		sum[CS_SET] += tval;
		setMinMaxLog(CS_SET, i, tval);
		if ( ret != CABI_SUCCESS ){
			cabi_result(ret, 1);
		}
		
	}

	/* unbind process */
	ret = cabi_account_unbind(pid);
	
	/* destroy cabi */
	ret = cabi_account_destroy(ucabip->cabi_id);
	if ( ret != CABI_SUCCESS ){
		cabi_result(ret, 1);
	}
}

static void syscallfork(int count)
{
	int ret, i;
	unsigned long long start, end, tval;

	clearLog(SYSCALL_FORK, count);
	
	for ( i = 0; i < count; i++ ){
		RDTSC(start);
		ret = fork();
		RDTSC(end);
		tval = diff(start, end);
		sum[SYSCALL_FORK] += tval;
		setMinMaxLog(SYSCALL_FORK, i, tval);
		switch( ret ){
		case -1: /* error */
			fprintf(stderr, "fork fail");
			exit(1);
			break;
		case 0:	/* child */
			exit(0);
			break;
		default:/* parent */
			waitpid(ret, NULL, 0);
			//sleep(0);
			break;
		}
	}
}

static void syscallgetpid(int count)
{
	int i;
	unsigned long long start, end, tval;

	clearLog(SYSCALL_GETPID, count);
	
	for ( i = 0; i < count; i++ ){
		RDTSC(start);
		getpid();
		RDTSC(end);
		tval = diff(start, end);
		sum[SYSCALL_GETPID] += tval;
		setMinMaxLog(SYSCALL_GETPID, i, tval);
	}
}

static void busywait(int count, int loopcount)
{
	int i, j, k;
	unsigned long long start, end, tval;
	volatile int a,b,c;

	clearLog(BUSYWAIT, count);
	
	for ( i = 0; i < count; i++ ){
		RDTSC(start);
		for ( j = 0; j < loopcount; j++ ){
			for ( k = 0; k < 10000; k++ ){
				a = b = c = 0;
				a = a * b * c;
				a = a + b + c;
			}
		}
		RDTSC(end);
		tval = diff(start, end);
		sum[BUSYWAIT] += tval;
		setMinMaxLog(BUSYWAIT, i, tval);
	}
}

//#define RDTSCLOG /* show debug log */
//#define CHILD_COUNT 3
#define CHILD_COUNT 499
#define CABI_COUNT  (CHILD_COUNT+1)
static void syscallforkcb(int count)
{
	int ret, i;
	unsigned int cabiid[CABI_COUNT];
	pid_t pid[CHILD_COUNT];
#ifdef RDTSCLOG
	unsigned long long *rtlog[CABI_COUNT];
	int child=0;
#else /*RDTSCLOG*/
	unsigned long long start, end, tval;
#endif /*RDTSCLOG*/

#ifdef RDTSCLOG
	for ( i = 0; i < CABI_COUNT; i++ ){
		rtlog[i]=(unsigned long long *)malloc(sizeof(unsigned long long)*count);
		if (rtlog[i] == NULL) program_fail("nomem");
		memset(rtlog[i], 0, sizeof(unsigned long long)*count);
	}
#endif /*RDTSCLOG*/

	clearLog(SYSCALL_FORKCB, count);

	if (ucabip->cabi_param.term_act) {
		for ( i = 0; i < CABI_COUNT; i++ ){
			ret = cabi_account_create (ucabip);
			if ( ret != CABI_SUCCESS ){
				cabi_result(ret, 1);
			}
			cabiid[i] = ucabip->cabi_id;
		}
	}

	for ( i = 0; i < CHILD_COUNT; i++ ){
		pid[i] = fork();
		switch( pid[i] ){
		case -1: /* error */
			fprintf(stderr, "fork fail");
			exit(1);
			break;
		case 0:	/* child */
			goto child_start;
		default:/* parent */
			if (ucabip->cabi_param.term_act) {
				ret = cabi_account_bind_pid(cabiid[i+1], pid[i]);
				if ( ret != CABI_SUCCESS ){
					cabi_result(ret, 1);
				}
			}
			break;
		}
	}

	if (ucabip->cabi_param.term_act) {
		/* bind parent process */
		ret = cabi_account_bind_pid(cabiid[0], getpid());
		if ( ret != CABI_SUCCESS ){
			cabi_result(ret, 1);
		}
	}

	sched_yield();
	for ( i = 0; i < count; i++ ){
#ifdef RDTSCLOG
		RDTSC(rtlog[0][i]);
		sched_yield();
#else /*RDTSCLOG*/
//		fprintf(stderr, "P");
		RDTSC(start);
		sched_yield();
		RDTSC(end);
		tval = diff(start, end);
		sum[SYSCALL_FORKCB] += tval;
		setMinMaxLog(SYSCALL_FORKCB, i, tval);
#endif /*RDTSCLOG*/
	}

	for ( i = 0; i < CHILD_COUNT; i++ ){
		waitpid(pid[i], NULL, 0);
	}

	if (ucabip->cabi_param.term_act) {
		ret = cabi_account_unbind(getpid());
		if ( ret != CABI_SUCCESS ){
			cabi_result(ret, 1);
		}
		for ( i = 0; i < CABI_COUNT; i++ ){
			ret = cabi_account_destroy(cabiid[i]);
			if ( ret != CABI_SUCCESS ){
				cabi_result(ret, 1);
			}
		}
	}

#ifdef RDTSCLOG
	goto child_exit;
#else /*RDTSCLOG*/
	return;
#endif /*RDTSCLOG*/

child_start:
	for ( i = 0; i < count; i++ ){
#ifdef RDTSCLOG
		RDTSC(rtlog[child][i]);
#endif /*RDTSCLOG*/
//		fprintf(stderr, "C");
		sched_yield();
	}

#ifdef RDTSCLOG
child_exit:
	for ( i = 0; i < count; i++ ){
		fprintf(stderr, "[%llu] PID[%08u] count[%08d]\n", rtlog[child][i], getpid(), i);
	}
	for ( i = 0; i < CABI_COUNT; i++ ){
		free(rtlog[i]);
	}
#endif /*RDTSCLOG*/

	sched_yield();
	exit(0);
}

int main (int argc, char **argv)
{
	int i, j, c, index = 0, count = 1, loopcount = 1000000;
	struct option *options;
	static struct option cscreate_options[] = {
		{"count", required_argument, 0, OPT_COUNT},
		{"create", no_argument, 0, CS_CREATE},
		{"destroy", no_argument, 0, CS_DESTROY},
		{"bind_pid", no_argument, 0, CS_BIND_PID},
		{"bind_pgid", no_argument, 0, CS_BIND_PGID},
		{"get", no_argument, 0, CS_GET},
		{"set", no_argument, 0, CS_SET},
		{"fork", no_argument, 0 ,SYSCALL_FORK},
		{"getpid", no_argument, 0 ,SYSCALL_GETPID},
		{"busywait", no_argument, 0 ,BUSYWAIT},
		{"forkcb", no_argument, 0 ,SYSCALL_FORKCB},
		{"loopcount", required_argument, 0 ,LOOPCOUNT},
		{"csv", required_argument, 0 ,CSV},
#ifdef _POSIX_PRIORITY_SCHEDULING
		{"rr", required_argument, 0, RR},	// RR塼
		{"fifo", required_argument, 0, FIFO},	// FIFO塼
#endif /* _POSIX_PRIORITY_SCHEDULING */
		{0,0,0,0},
	};
	int csvfd=0;

	for ( i = 0; i < END_OF_SYSCALL; i++ ){
		sum[i] = 0;
		min[i] = TVALMAX;
		max[i] = 0;
	}
	calibration();


	/* clear cabi_uaccount */
	memset(&ucabi, 0, sizeof(struct cabi_uaccount));
	ucabip = &ucabi;

	/* make option table */
	options = append_common_options(cscreate_options, 0);
	options = append_pid_options(options, 1);
	options = append_pgid_options(options, 1);
	options = append_ucabi_options(options, 1);

        for ( i = 0; i < END_OF_SYSCALL; i++ ){
                ctlog[i] = malloc(sizeof(unsigned long long) * count);
                if (ctlog[i] == NULL) program_fail("nomem");
        }

	c = 0;
	while( c != -1 ){
		c = getopt_long_only (argc, argv, "", options, &index);

		switch( c ){
		case OPT_COUNT:
			{
				int k;
				
				count = parse_int(options[index].name, optarg);
				for ( k = 0; k < END_OF_SYSCALL; k++ ){
                                        free(ctlog[k]);
                                        ctlog[k] = malloc(sizeof(unsigned long long) * count);
                                        if (ctlog[k] == NULL) program_fail("nomem");
				}
			}
			break;
		case CS_CREATE:
		case CS_DESTROY:
			check_setarg_ucabi_term_act(1,  ERR_MSG "--TERM_XXX");
			check_setarg_ucabi_cpu_time(1,  ERR_MSG "--cpu_time=x");
			check_setarg_ucabi_cpu_period(1, ERR_MSG "--cpu_period=x");
			create_destroy(INIT_COUNT);
			create_destroy(count);
			break;
		case CS_BIND_PID:
			check_setarg_pid(1, "Usage: ctbindpid --pid=x");
			check_setarg_ucabi_term_act(1,  ERR_MSG "--TERM_XXX");
			check_setarg_ucabi_cpu_time(1,  ERR_MSG "--cpu_time=x");
			check_setarg_ucabi_cpu_period(1, ERR_MSG "--cpu_period=x");
			bind_pid(INIT_COUNT);
			bind_pid(count);
			break;
		case CS_BIND_PGID:
			check_setarg_pgid(1, "Usage: ctbindpgid --pid=x");
			check_setarg_ucabi_term_act(1,  ERR_MSG "--TERM_XXX");
			check_setarg_ucabi_cpu_time(1,  ERR_MSG "--cpu_time=x");
			check_setarg_ucabi_cpu_period(1, ERR_MSG "--cpu_period=x");
			bind_pgid(INIT_COUNT);
			bind_pgid(count);
			break;
		case CS_GET:
			check_setarg_pid(1, "Usage: ctbindpid --pid=x");
			check_setarg_ucabi_term_act(1,  ERR_MSG "--TERM_XXX");
			check_setarg_ucabi_cpu_time(1,  ERR_MSG "--cpu_time=x");
			check_setarg_ucabi_cpu_period(1, ERR_MSG "--cpu_period=x");
			get(INIT_COUNT);
			get(count);
			break;
		case CS_SET:
			check_setarg_pid(1, "Usage: ctbindpid --pid=x");
			check_setarg_ucabi_term_act(1,  ERR_MSG "--TERM_XXX");
			check_setarg_ucabi_cpu_time(1,  ERR_MSG "--cpu_time=x");
			check_setarg_ucabi_cpu_period(1, ERR_MSG "--cpu_period=x");
			set(INIT_COUNT);
			set(count);
			break;
		case SYSCALL_FORK:
			syscallfork(INIT_COUNT);
			syscallfork(count);
			break;
		case SYSCALL_FORKCB:
			syscallforkcb(count);
			break;
		case SYSCALL_GETPID:
			syscallgetpid(INIT_COUNT);
			syscallgetpid(count);
			break;
		case LOOPCOUNT:
			loopcount = parse_int(options[index].name, optarg);
			break;
		case BUSYWAIT:
			busywait(count, loopcount);
			break;
#ifdef _POSIX_PRIORITY_SCHEDULING
		case RR: {
			int min, max;
			struct sched_param param;
			int priority = 1;

			min = sched_get_priority_min(SCHED_RR);
			max = sched_get_priority_max(SCHED_RR);

			if ( !strcmp("MAX", optarg) ) {
				priority = max;

			} else if ( !strcmp("MIN", optarg) ) {
				priority = min;

			} else {
				priority = parse_int("rr", optarg);
				if ( priority < min || max < priority ){
					program_fail("illegal argument:rr priority:%d <= %d <= %d", min, priority, max);
				}
			}
			param.sched_priority = priority;
			sched_setscheduler(getpid(), SCHED_RR, &param);
			break;
		}
		case FIFO: {
			int min, max;
			struct sched_param param;
			int priority = 1;

			min = sched_get_priority_min(SCHED_FIFO);
			max = sched_get_priority_max(SCHED_FIFO);

			if ( !strcmp("MAX", optarg) ) {
				priority = max;

			} else if ( !strcmp("MIN", optarg) ) {
				priority = min;

			} else {
				priority = parse_int("rr", optarg);
				if ( priority < min || max < priority ){
					program_fail("illegal argument:fifo priority:%d <= %d <= %d", min, priority, max);
				}
			}
			param.sched_priority = priority;
			sched_setscheduler(getpid(), SCHED_FIFO, &param);
			break;
		}
#endif /* _POSIX_PRIORITY_SCHEDULING */
		case CSV:
			csvfd = open(optarg, O_WRONLY|O_CREAT);
                        if (csvfd < 0) {
                                program_fail("open fail");
                        }
			break;
		case -1:	// end of argument
			break;
		default:
			if ( !check_common(c, options, index) ){
				// accept argument : NOP
			} else if ( !check_ucabi(c, options, index, &ucabip) ){
				// accept argument : NOP
			} else if ( !check_pid(c, options, index, &pid) ){
				// accept argument : NOP
			} else if ( !check_pgid(c, options, index, &pgid) ){
				// accept argument : NOP
			} else {
				unknown_option(index, c, optarg);
			}
		}
	}

	/* ɽ */
	for ( i = 0; i < END_OF_SYSCALL; i++ ){
		if ( sum[i] > 0 ){
                        static char buf[1024];
                        int wcount;
                        if (csvfd) {
                                wcount = sprintf(buf, "%s,\n", name[i]);
                                write(csvfd, buf, wcount);
                                wcount = sprintf(buf, "count,tsccount\n");
                                write(csvfd, buf, wcount);
                        }
			for ( j = 0; j < count; j++ ){
                                if (csvfd) {
                                        wcount = sprintf(buf, "%d,%llu\n", j, ctlog[i][j]);
                                        write(csvfd, buf, wcount);
                                } else {
                                        cabi_information("  count=%5d : %7llu\n", j, ctlog[i][j]);
                                }
			}
			printf("syscall %s min=%lld max=%lld ave=%lld\n",
			       name[i],
			       min[i],
			       max[i],
			       (sum[i]/count));
		}
		
	}

        for ( i = 0; i < END_OF_SYSCALL; i++ ){
                free(ctlog[i]);
        }

        if (csvfd) {
                close(csvfd);
        }

	/*
	 * ޤãϡ٤Ƥ뤿Υå
	 */
	cabi_result(0, 1);
	
	return 0;	
}
#else
int main(void)
{
        fprintf(stderr, "cstest only exec x86 environment\n");
        return 1;
}
#endif /* CONFIG_X86 */
