#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <unistd.h>
#include <sched.h>
#include <string.h>
#include <fcntl.h>
#include <time.h>
#include <sys/time.h>
#include <cabi/cabi_error.h>
#include "common.h"

#define ENV_CTLOOP_LEVEL	"CTLEVEL"
#define ENV_CT_LEVEL_PID	"CTLEVEL_PID"
#define ENV_CT_FORKSIG_PID	"CTFORKSIG_PID"
#define CTPID_FILE		"/dev/shm/ctloop.pid"

#define ID		0x01
#define LEVEL		0x02
#define CHILDREN	0x03
#define SLEEP		0x04
#define SLEEP_X		0x05
#define WORK		0x06
#define WORK_X		0x07
#define DIEALL		0x08
#define SIG		0x09
#define SIGALL		0x0a
#define DIE		0x0b
#define BIND		0x0c
#ifdef _POSIX_PRIORITY_SCHEDULING
#define RR		0xA0
#define FIFO		0xA1
#endif /* _POSIX_PRIORITY_SCHEDULING */
#define NULLACCESS	0xB0
#define NULLSIG		0xB1
#define BUSYSIG		0xC0
#define BUSYENDSIG	0xC1
#define BINDSIG		0xD0
#define UNBINDSIG	0xD1
#define FORKSIG		0xE0
#define OLDESTROYSIG	0xF0

#define LEVEL_MAX	5
#define CHILED_MIN	1
#define CHILED_MAX	9
#define CHILED_NUM_BASE	48 /* 0 */

static unsigned long cabiid = -1;
static char **argv_bk;

#define local_cabi_info(arg) {\
      cabi_information("[%d] LEVEL-%s ", getpid(), get_current_level_string()); \
      cabi_information(arg); }

static void child_check(int child)
{
	int num = child - CHILED_NUM_BASE;

	if (num < CHILED_MIN ||
	    num > CHILED_MAX) {
		cabi_information("children %d...%d\n", CHILED_MIN, CHILED_MAX);
		program_fail("child_check");
	}
}

/* check level max and child max */
static void level_check(char *optarg)
{
	int i=0;

	if (!optarg) program_fail("level_check");

	if (strlen(optarg) > LEVEL_MAX) {
		cabi_information("level max %d\n", LEVEL_MAX);
		program_fail("level_check");
	}

	while (optarg[i]) {
		child_check((int)optarg[i]);
		i++;
	}
}

/* convert 111 to 1_1_1 */
static void level_conv(char *output, char *level)
{
	int i=0, j=0, flag=0;
	char *buf;
	if (!optarg||!level) program_fail("level_conv");

	buf = output;
	while(level[i]) {
		if (flag) {
			buf[j++] = '_';
			flag=0;
		} else {
			buf[j++] = level[i++];
			flag=1;
		}
	}
	buf[j++]= '\0';
}

/* get my name */
static char *get_current_level_string(void)
{
	char *p;

	/* Υץ Level Ķѿ */
	p = getenv(ENV_CTLOOP_LEVEL);

	if ( p == NULL ){
		return "1";
	}

	return p;
}

void fork_child(char **argv, int count)
{
	int ret;

	if (!argv) program_fail("fork_child");

	if ( (ret = fork()) < 0 ) {
		program_fail("fork fail");
	}

	if (ret == 0) {
		/* child */
		char buf[4096];

		/* ĶѿLEVELֹ򥻥åȤ
		   get_current_level_stringǼǤͤ */
		memset(buf, 0, sizeof(buf));
		sprintf(buf, "%s_%d", get_current_level_string(), count);
		setenv(ENV_CTLOOP_LEVEL, buf, 1);

		/* ҤPIDexportեؽ */
		memset(buf, 0, sizeof(buf));
		sprintf(buf, "%s_%s", ENV_CT_LEVEL_PID, get_current_level_string());
		local_cabi_info("forked\n");
		cabi_file_export_int(CTPID_FILE, 0, buf, getpid());
		local_cabi_info("exec\n");

		execvp(argv[0], argv);
	} else {
		/* parent */
	}

	return;
}

/*
 * bind signal handler
 *
 * ʥAO˼ʬbind
 * 餫--cabiid=<AOID>bindAOꤷƤɬפ
 */
static void cabi_bindsighandler(int sig)
{
	if ( cabiid == -1 ) {
		cabi_information("option --cabiid==<AOID>\n");
	}
	if ( cabi_account_bind_pid(cabiid, getpid()) ) {
		cabi_information("bind fail\n");
	}

	fprintf(stderr, "[%d] CATCH SIGNAL %s(%d) [BIND]\n",
		getpid(), get_signal_name(sig, 0), sig);
}

/* unbind signal handler */
static void cabi_unbindsighandler(int sig)
{
	if ( cabi_account_unbind(getpid()) ) {
		cabi_information("unbind fail\n");
	}

	fprintf(stderr, "[%d] CATCH SIGNAL %s(%d) [UNBIND]\n",
		getpid(), get_signal_name(sig, 0), sig);
}

/*
 * fork signal handler
 *
 * ʥϥɥfork
 * forkҥץPIDctloop.pidCTFORKSIG_PID˽񤭽Ф
 * forkϺǽΥʥȤԤ2ܰʹߤϲ⤻꥿󤹤
 * ξCATCH SIGNALåϤǤʤФ⤷ʤ
 */
static void cabi_forksighandler(int sig)
{
	int ret;
	static int forkcount=0;

	if (forkcount) return;

	if (!argv_bk) program_fail("fork_child");

	if ( (ret = fork()) < 0 ) {
		program_fail("fork fail");
	}

	if (ret == 0) {
		/* child */
		char buf[4096];

		/* ĶѿLEVELֹ򥻥åȤ
		   get_current_level_stringǼǤͤ */
		memset(buf, 0, sizeof(buf));
		sprintf(buf, "%s_%d", get_current_level_string(), 1);
		setenv(ENV_CTLOOP_LEVEL, buf, 1);

		/* ҤPIDexportեؽ */
		memset(buf, 0, sizeof(buf));
		local_cabi_info("forked\n");
		cabi_file_export_int(CTPID_FILE, 0, ENV_CT_FORKSIG_PID, getpid());
		local_cabi_info("exec\n");

		fprintf(stderr, "[%d] CATCH SIGNAL %s(%d) [FORK]\n",
			getppid(), get_signal_name(sig, 0), sig);

		execvp(argv_bk[0], argv_bk);
	} else {
		/* parent */
		forkcount++;
	}
}

/* overload destroy signal handler */
static void cabi_oldestroysighandler(int sig)
{
	if ( cabi_overload_destroy() ) {
		cabi_information("overload destroy fail\n");
	}

	fprintf(stderr, "[%d] CATCH SIGNAL %s(%d) [OVERLOAD DESTROY]\n",
		getpid(), get_signal_name(sig, 0), sig);
}

int main ( int argc, char **argv)
{
	int c, index = 0, option_is_valid = 0;
	int childrens=0;
	int loop_flag=1;
	int bind_flag=0;
	static struct timeval sleep_time, work_time;
	struct cabi_uaccount ucabi, *ucabip;
#ifdef _POSIX_PRIORITY_SCHEDULING
	int rrOrFifo = SCHED_OTHER;	// 0:TS, 1:RR, 2:FIFO
	int priority = 1;
#endif /* _POSIX_PRIORITY_SCHEDULING */
	long long nullaccess = 0;	/* null ץǡnull ޤǤ usec ñ̤Ԥ */
	struct option *options;
	static struct option ctloop_options[] = {
		{"level", required_argument, 0, LEVEL},		// ʹߤΥץо
		{"children", required_argument, 0, CHILDREN},	// Ҷο
		{"sleep", required_argument, 0, SLEEP},
		{"sleep_x", required_argument, 0, SLEEP_X},
		{"work", required_argument, 0, WORK},
		{"work_x", required_argument, 0, WORK_X},
		{"dieall", no_argument, 0, DIEALL},
		{"sig", required_argument, 0, SIG},
		{"sigall", required_argument, 0, SIGALL},
		{"die", no_argument, 0, DIE},
		{"bind", no_argument, 0, BIND},
#ifdef _POSIX_PRIORITY_SCHEDULING
		{"rr", required_argument, 0, RR},	// RR塼
		{"fifo", required_argument, 0, FIFO},	// FIFO塼
#endif /* _POSIX_PRIORITY_SCHEDULING */
		{"null", required_argument, 0, NULLACCESS},	// ָ NULL 
		{"nullsig", required_argument, 0, NULLSIG},	// signal  NULL 
		{"busysig", required_argument, 0, BUSYSIG},	// busy signal ꤹ
		{"busyendsig", required_argument, 0, BUSYENDSIG},//busy signal λ signal ꤹ
		{"bindsig", required_argument, 0, BINDSIG},	// ʬbindsignalꤹ
		{"unbindsig", required_argument, 0, UNBINDSIG},	// ʬunbindsignalꤹ
		{"forksig", required_argument, 0, FORKSIG},	// ʬforksignalꤹ
		{"oldestroysig", required_argument, 0, OLDESTROYSIG},// СAOsignalꤹ

		{0,0,0,0},
	};

	argv_bk = argv;

	/* clear cabi_uaccount */
	memset(&ucabi, 0, sizeof(struct cabi_uaccount));
	ucabip = &ucabi;

	/* make option table */
	options = append_common_options(ctloop_options, 0);
	options = append_cabiid_options(options, 1);
	options = append_ucabi_options(options, 1);

	/* noverbose */
	check_common(OPT_CODE_NOVERBOSE, options, 0);

	/* check before fork */
	c = 0;
	while( c != -1 ){
		c = getopt_long_only (argc, argv, "", options, &index);
		switch( c ){
		case LEVEL:
			level_check(optarg);
			break;
		case CHILDREN:
			/* max check */
			if (optarg[1]) program_fail("children");
			child_check(optarg[0]);
			childrens += parse_int("children", optarg);
			break;
		default:
			check_common(c, options, index);
			break;
		}
	}
	optind = 0;

	if ( !strcmp(get_current_level_string(), "1") ){
#ifdef _POSIX_PRIORITY_SCHEDULING
		struct sched_param param;
		int schedtype;

		/* ޤRTץȤưƤ */
//		  schedtype = SCHED_FIFO;
		schedtype = SCHED_RR;
		param.sched_priority = sched_get_priority_max(SCHED_RR);
		sched_setscheduler(getpid(), schedtype, &param);
#endif /* _POSIX_PRIORITY_SCHEDULING */

//		option_is_valid = 1;
		/* ƤPIDexportեؽ */
		cabi_file_export_int(CTPID_FILE, 1, ENV_CT_PID, getpid());
		cabi_file_export_int(CTPID_FILE, 0, ENV_CT_PGID, getpgrp());
		cabi_file_export_int(CTPID_FILE, 0, ENV_CT_LEVEL_PID"_1", getpid());
		
		/* ǽ level Υץ -l=1 Ф륪ץȤƽ */
		option_is_valid = 1;
	}

	c = 0;
	while( c != -1 ){
		struct timespec tmp_time;
		c = getopt_long_only (argc, argv, "", options, &index);

		switch( c ){

		/* level local options */
		case LEVEL: {
			char buf[1024];
			memset(buf, 0, sizeof(buf));
			level_conv(buf, optarg);
			if ( !strcmp(get_current_level_string(), buf) ){
				option_is_valid = 1;
			} else {
				option_is_valid = 0;
			}
			break;
		}
		case CHILDREN: {
			int i, num;

			if ( !option_is_valid ) break;

			num = parse_int("children", optarg);
			for ( i = 1; i < num+1; i++ ){
				fork_child(argv, i);
			}

			break;
		}
		case SLEEP:
			if ( !option_is_valid ) break;
			parse_time("sleep_time", &tmp_time, optarg);
			sleep_time.tv_sec = tmp_time.tv_sec;
			sleep_time.tv_usec = tmp_time.tv_nsec/(NANOSEC/MICROSEC);
			break;
		case SLEEP_X:
			if ( !option_is_valid ) break;
			parse_time_x("sleep_time_x", &tmp_time, optarg);
			sleep_time.tv_sec = tmp_time.tv_sec;
			sleep_time.tv_usec = tmp_time.tv_nsec/(NANOSEC/MICROSEC);
			break;
		case WORK:
			if ( !option_is_valid ) break;
			parse_time("work_time", &tmp_time, optarg);
			work_time.tv_sec = tmp_time.tv_sec;
			work_time.tv_usec = tmp_time.tv_nsec/(NANOSEC/MICROSEC);
			break;
		case WORK_X:
			if ( !option_is_valid ) break;
			parse_time_x("work_time_x", &tmp_time, optarg);
			work_time.tv_sec = tmp_time.tv_sec;
			work_time.tv_usec = tmp_time.tv_nsec/(NANOSEC/MICROSEC);
			break;

#ifdef _POSIX_PRIORITY_SCHEDULING
		case RR: {
			int min, max;

			if ( !option_is_valid ) break;

			min = sched_get_priority_min(SCHED_RR);
			max = sched_get_priority_max(SCHED_RR);

			if ( !strcmp("MAX", optarg) ) {
				priority = max;

			} else if ( !strcmp("MIN", optarg) ) {
				priority = min;

			} else {
				priority = parse_int("rr", optarg);
				if ( priority < min || max < priority ){
					program_fail("illegal argument:rr priority:%d <= %d <= %d", min, priority, max);
				}
			}

			rrOrFifo = SCHED_RR;
			break;
		}
		case FIFO: {
			int min, max;

			if ( !option_is_valid ) break;

			min = sched_get_priority_min(SCHED_FIFO);
			max = sched_get_priority_max(SCHED_FIFO);

			if ( !strcmp("MAX", optarg) ) {
				priority = max;

			} else if ( !strcmp("MIN", optarg) ) {
				priority = min;

			} else {
				priority = parse_int("rr", optarg);
				if ( priority < min || max < priority ){
					program_fail("illegal argument:fifo priority:%d <= %d <= %d", min, priority, max);
				}
			}

			rrOrFifo = SCHED_FIFO;
			break;
		}
#endif /* _POSIX_PRIORITY_SCHEDULING */

		case SIG:
			if ( !option_is_valid ) break;
						/* through next case */
		case SIGALL:
			if ( !strcmp("ALL", optarg) ) {
				int i;
				for (i=1; i<32; i++) {
					cabi_signal(i, NULL);
				}
				cabi_information("[%d] signal handler ALL SIGNAL\n",
						 getpid());
			} else {
				int sigcode;
				const char *signame;

				sigcode = parse_signal(options[index].name, optarg);
				signame = get_signal_name(sigcode, 0);
				cabi_signal(sigcode, NULL);
				cabi_information("[%d] signal handler %s(%d)\n",
						 getpid(),
						 signame, sigcode);
			}
			break;

		/* global options */
		case DIEALL:
			loop_flag=0;
			break;
		/* local option */
		case DIE:
			if ( !option_is_valid ) break;
			loop_flag=0;
			break;
		case BIND:
			if ( !option_is_valid ) break;
			bind_flag=1;
			break;
		case NULLACCESS:
			if ( !option_is_valid ) break;
			{
				/*
				 * nullaccess ϰΥեޥåȾ nsec ñ̤
				 * ǽǤ뤬ͭ٤ usec Ȥʤ
				 */
				struct timespec tmp;
				
				parse_time("null", &tmp, optarg);
				nullaccess = (long long)(tmp.tv_sec * 1000000) + (tmp.tv_nsec/1000);
			}
			break;

                case NULLSIG:
                        if ( !option_is_valid ) break;
			
			{
			    int sigcode;
			    const char *signame;

			    sigcode = parse_signal(options[index].name, optarg);
			    signame = get_signal_name(sigcode, 0);
			    cabi_signal(sigcode, cabi_nullsighandler);
			    cabi_information("[%d] busy signal handler %s(%d)\n",
					     getpid(),
					     signame, sigcode);
                        }
                        break;
                case BUSYSIG:
                        if ( !option_is_valid ) break;
			
			{
			    int sigcode;
			    const char *signame;

			    sigcode = parse_signal(options[index].name, optarg);
			    signame = get_signal_name(sigcode, 0);
			    cabi_signal(sigcode, cabi_busysighandler);
			    cabi_information("[%d] busy signal handler %s(%d)\n",
					     getpid(),
					     signame, sigcode);
                        }
                        break;
                case BUSYENDSIG:
                        if ( !option_is_valid ) break;
			
			{
			    int sigcode;
			    const char *signame;

			    sigcode = parse_signal(options[index].name, optarg);
			    signame = get_signal_name(sigcode, 0);
			    cabi_signal(sigcode, cabi_busyendsighandler);
			    cabi_information("[%d] busy end signal handler %s(%d)\n",
					     getpid(),
					     signame, sigcode);
                        }
                        break;
                case BINDSIG:
                        if ( !option_is_valid ) break;
			
			{
			    int sigcode;
			    const char *signame;

			    sigcode = parse_signal(options[index].name, optarg);
			    signame = get_signal_name(sigcode, 0);
			    cabi_signal(sigcode, cabi_bindsighandler);
			    cabi_information("[%d] bind signal handler %s(%d)\n",
					     getpid(),
					     signame, sigcode);
                        }
                        break;
                case UNBINDSIG:
                        if ( !option_is_valid ) break;
			
			{
			    int sigcode;
			    const char *signame;

			    sigcode = parse_signal(options[index].name, optarg);
			    signame = get_signal_name(sigcode, 0);
			    cabi_signal(sigcode, cabi_unbindsighandler);
			    cabi_information("[%d] unbind signal handler %s(%d)\n",
					     getpid(),
					     signame, sigcode);
                        }
                        break;
                case FORKSIG:
                        if ( !option_is_valid ) break;
			
			{
			    int sigcode;
			    const char *signame;

			    sigcode = parse_signal(options[index].name, optarg);
			    signame = get_signal_name(sigcode, 0);
			    cabi_signal(sigcode, cabi_forksighandler);
			    cabi_information("[%d] bind signal handler %s(%d)\n",
					     getpid(),
					     signame, sigcode);
                        }
                        break;
                case OLDESTROYSIG:
                        if ( !option_is_valid ) break;
			
			{
			    int sigcode;
			    const char *signame;

			    sigcode = parse_signal(options[index].name, optarg);
			    signame = get_signal_name(sigcode, 0);
			    cabi_signal(sigcode, cabi_oldestroysighandler);
			    cabi_information("[%d] overload destroy signal handler %s(%d)\n",
					     getpid(),
					     signame, sigcode);
                        }
                        break;
		case -1:	// end of argument
			break;
		default:
			if ( !check_common(c, options, index) ){
				// accept argument : NOP
			} else if ( !check_cabiid(c, options, index, &cabiid) ){
				if ( !option_is_valid ) cabiid = -1;
			} else if ( !check_ucabi(c, options, index, &ucabip) ){
				// accept argument : NOP
			} else {
				unknown_option(index, c, optarg);
			}
		}
	}

	/* wait all childrens wake up */
	if (childrens && cabi_export()) {
		FILE *fd = fopen(CTPID_FILE, "r");

		if (!fd) {
			program_fail("wait wakeup childrens");
		} else {
			int i;
			char buf[1024];

			for (;;) {
				i = -3;

				while (fgets(buf, sizeof(buf), fd) != NULL) {
					i++;
				}
				if ( !strcmp(get_current_level_string(), "1") ) {
					cabi_information("[%s] c[%d] == i[%d]\n",
							 get_current_level_string(),
							 childrens,
							 i);
				}
				if (i != childrens) {
					sleep(1);
					rewind(fd);
					continue;
				}

				break;
			}
		}

		fclose(fd);
	}

	/* print loop status */
	local_cabi_info("LOOP ");
	cabi_information("loop_flag[%d] ", loop_flag);
	if ( work_time.tv_sec || work_time.tv_usec ) {
		cabi_information("work[%d.%06ds(%lds.%3ldms/%6ldus)] ",
				 work_time.tv_sec, work_time.tv_usec,
				 work_time.tv_sec,
				 work_time.tv_usec/MILISEC,
				 work_time.tv_usec);
	} else {
		cabi_information("nocheck ");
	}
	if ( sleep_time.tv_sec || sleep_time.tv_usec ) {
		cabi_information("sleep[%d.%06ds(%lds.%3ldms/%6ldus)] ",
				 sleep_time.tv_sec, sleep_time.tv_usec,
				 sleep_time.tv_sec,
				 sleep_time.tv_usec/MILISEC,
				 sleep_time.tv_usec);
	} else {
		cabi_information("nowait ");
	}
	cabi_information("\n");

	/* print result ok */
	if ( !strcmp(get_current_level_string(), "1") ){
		/* parent only */
		cabi_result(0, 1);
	}

        /* schedule change type */
#ifdef _POSIX_PRIORITY_SCHEDULING
	switch ( rrOrFifo ) {
	case SCHED_RR: /* fall down */
	case SCHED_FIFO: {
		struct sched_param param;
		param.sched_priority = priority;
		sched_setscheduler(getpid(), rrOrFifo, &param);
		break;
	}
	case SCHED_OTHER: /* fall down */
	default: {
		/* ꤵƤʤФTSץˤɤ */
		struct sched_param param;
		param.sched_priority = 0;
		sched_setscheduler(getpid(), SCHED_OTHER, &param);
		break;
	}
	}
#endif /* _POSIX_PRIORITY_SCHEDULING */

        /* bind myself */
        if (bind_flag) {
                if ( cabiid == -1 ) {
                        /* create */
                        if ( cabi_account_create (&ucabi) != CABI_SUCCESS ) {
                                program_fail("cabi create fail\n");
                        }
                        cabi_export_int(ENV_CT_CABI_ID, ucabi.cabi_id);
                        cabiid = ucabi.cabi_id;
                        cabi_information("create AO[%d]\n", cabiid);
                }

                /* bind */
                if ( cabi_account_bind_pid(cabiid, getpid()) != CABI_SUCCESS) {
                        program_fail("cabi bind fail\n");
                }
                local_cabi_info("");
                cabi_information("bind to AO[%d]\n", cabiid);
        }

	/* loop */
{
	int cnt = 0;
	int work_flag  = (work_time.tv_sec || work_time.tv_usec);
	int sleep_flag = (sleep_time.tv_sec || sleep_time.tv_usec);
	struct timeval old, new;
	long long  start_usec;
	int mark=0;

	if (cabi_verbose())
		mark = getpid() % 10;

	{
		struct timeval tmp_time;
		gettimeofday(&tmp_time, NULL);
		start_usec = (long long)(tmp_time.tv_sec * 1000000) + tmp_time.tv_usec;
	}
	
	while (loop_flag) {

		/* NOP ??? */


		if ( nullaccess != 0 ){
			struct timeval tmp_time;
			long long cur;
			gettimeofday(&tmp_time, NULL);
			cur = (long long)(tmp_time.tv_sec * 1000000) + tmp_time.tv_usec;

			if ( cur >= (start_usec + nullaccess) ){
				*(char*)0 = 0;
			}
		}
		
		/* work */
		if ( work_flag ) {
			gettimeofday(&old, NULL); /* get start time */
		}

		for (;;) {
			cnt++;
			if ( !sleep_flag || work_flag ) {
				if ( !(cnt % 500001) ){
					cabi_information("(%d#)", mark);
				} else if ( !(cnt % 10000) ){
					cabi_information("%d", mark);
				}
				cabi_information_flush();
			}

			if ( !work_flag ) {
				break; /* goto sleep */
			}

			gettimeofday(&new, NULL); /* get now time */
			if ( new.tv_sec > old.tv_sec+work_time.tv_sec ) {
				break; /* goto sleep */
			}
			if ( new.tv_sec == old.tv_sec+work_time.tv_sec &&
			     new.tv_usec > old.tv_usec+work_time.tv_usec ) {
				break; /* goto sleep */
			}
		}

		/* sleep */
		if ( sleep_flag ) {
			if ( work_flag ) {
				cabi_information("(%dZ)", mark);
				cabi_information_flush();
			} else {
				cabi_information("(%ds)", mark);
				cabi_information_flush();
			}

			if (sleep_time.tv_sec) {
				sleep(sleep_time.tv_sec);
			}
			if (sleep_time.tv_usec) {
				usleep(sleep_time.tv_usec);
			}
			/* wakeup */
		}
	}
}

	local_cabi_info("die\n");
	return 0;
}
