package com.limegroup.gnutella.gui.download;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.Comparator;

import javax.swing.JPopupMenu;

import com.limegroup.gnutella.Assert;
import com.limegroup.gnutella.Downloader;
import com.limegroup.gnutella.Endpoint;
import com.limegroup.gnutella.RemoteFileDesc;
import com.limegroup.gnutella.RouterService;
import com.limegroup.gnutella.downloader.AlreadyDownloadingException;
import com.limegroup.gnutella.gui.GUIMediator;
import com.limegroup.gnutella.gui.PaddedPanel;
import com.limegroup.gnutella.gui.search.SearchMediator;
import com.limegroup.gnutella.gui.tables.AbstractTableMediator;
import com.limegroup.gnutella.gui.tables.ColumnPreferenceHandler;
import com.limegroup.gnutella.gui.tables.DataLine;
import com.limegroup.gnutella.gui.tables.DragManager;
import com.limegroup.gnutella.gui.tables.LimeJTable;
import com.limegroup.gnutella.gui.tables.LimeTableColumn;
import com.limegroup.gnutella.gui.tables.SimpleColumnListener;
import com.limegroup.gnutella.gui.tables.TableSettings;
import com.limegroup.gnutella.gui.themes.ThemeFileHandler;
import com.limegroup.gnutella.gui.themes.ThemeMediator;
import com.limegroup.gnutella.gui.themes.ThemeSettings;
import com.limegroup.gnutella.settings.QuestionsHandler;
import com.limegroup.gnutella.settings.SharingSettings;
import com.limegroup.gnutella.util.CommonUtils;
import com.limegroup.gnutella.util.ProcessingQueue;

/**
 * This class acts as a mediator between all of the components of the
 * download window.  It also constructs all of the download window
 * components.
 */
public final class DownloadMediator extends AbstractTableMediator {

	/**
	 * Variable for the total number of downloads that have been added in this
	 * session.
	 */
	private static int _totalDownloads = 0;

	/**
	 * Flag for whether or not an mp3 file has been launched from the download
	 * window.
	 */
    private static boolean _audioLaunched = false;

    /**
     * instance, for singleton acces
     */
    private static DownloadMediator _instance = new DownloadMediator();

    /**
     * Variable for whether or not the currently selected host has chat
     * enabled.
     */
    private static boolean _chatEnabled;

    /**
     * Variable for whether or not the currently selected host has browse
     * host enabled.
     */
    private static boolean _browseEnabled;
    
    /**
     * The queue that launched items are processed in.
     */
    private static final ProcessingQueue QUEUE = 
        new ProcessingQueue("DownloadLauncher");


    public static DownloadMediator instance() { return _instance; }

    /**
     * Variables so only one ActionListener needs to be created for both
     * the buttons & popup menu.
     */
    public ActionListener CHAT_LISTENER;
    public ActionListener CLEAR_LISTENER;
    public ActionListener BROWSE_LISTENER;
    public ActionListener LAUNCH_LISTENER;
    public ActionListener RESUME_LISTENER;
    public ActionListener PAUSE_LISTENER;
    public ActionListener PRIORITY_UP_LISTENER;
    public ActionListener PRIORITY_DOWN_LISTENER;

    /** The actual download buttons instance.
     */
    private DownloadButtons _downloadButtons;
    
    /**
     * Overriden to have different default values for tooltips.
     */
    protected void buildSettings() {
        SETTINGS = new TableSettings(ID) {
            public boolean getDefaultTooltips() {
                return false;
            }
        };
    }
    
    /**
     * Sets up drag & drop for the table.
     */
    protected void setupDragAndDrop() {
        DragManager.install(TABLE);
    }    

    /**
     * Build some extra listeners
     */
    protected void buildListeners() {
        super.buildListeners();
        
        CHAT_LISTENER = new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                chatWithSelectedDownloads();
            }
        };
        
        CLEAR_LISTENER = new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                clearCompletedDownloads();
            }
        };
        
        BROWSE_LISTENER = new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                browseSelectedDownloads();
            }
        };
        
        LAUNCH_LISTENER = new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                launchSelectedDownloads();
            }
        };
        
        RESUME_LISTENER = new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                resumeSelectedDownloads();
            }
        };
        
        PAUSE_LISTENER = new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                pauseSelectedDownloads();
            }
        };
        
        PRIORITY_UP_LISTENER = new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                bumpPriority(true);
            }
        };
        
        PRIORITY_DOWN_LISTENER = new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                bumpPriority(false);
            }
        };
    }

	/**
	 * Set up the necessary constants.
	 */
	protected void setupConstants() {
		MAIN_PANEL =
		    new PaddedPanel(GUIMediator.getStringResource("DOWNLOAD_TITLE"));
		DATA_MODEL = new DownloadModel();
		TABLE = new LimeJTable(DATA_MODEL);
        _downloadButtons = new DownloadButtons(this);
		BUTTON_ROW = _downloadButtons.getComponent();
    }
    
    /**
     * Sets up the table headers.
     */
    protected void setupTableHeaders() {
        super.setupTableHeaders();
        
        // set the queue panel to be visible depending on whether or 
        // not the priority column is visible.
        Object pId = DATA_MODEL.getColumnId(DownloadDataLine.PRIORITY_INDEX);
        _downloadButtons.setQueuePanelVisible(TABLE.isColumnVisible(pId));
        
        // add a listener to keep the queue panel in synch with the priority column
        ColumnPreferenceHandler cph = TABLE.getColumnPreferenceHandler();
        cph.setSimpleColumnListener(new SimpleColumnListener() {
            public void columnAdded(LimeTableColumn ltc, LimeJTable table) {
                Assert.that(table == TABLE);
                if(ltc.getModelIndex() == DownloadDataLine.PRIORITY_INDEX)
                    _downloadButtons.setQueuePanelVisible(true);
            }
            
            public void columnRemoved(LimeTableColumn ltc, LimeJTable table) {
                Assert.that(table == TABLE);
                if(ltc.getModelIndex() == DownloadDataLine.PRIORITY_INDEX)
                    _downloadButtons.setQueuePanelVisible(false);
            }
        });
    }

    /**
     * Update the splash screen.
     */
	protected void updateSplashScreen() {
		GUIMediator.setSplashScreenString(
            GUIMediator.getStringResource("SPLASH_STATUS_DOWNLOAD_WINDOW"));
    }

	/**
	 * Constructs all of the elements of the download window, including
	 * the table, the buttons, etc.
	 */
	private DownloadMediator() {
	    super("DOWNLOAD_TABLE");
		GUIMediator.addRefreshListener(this);
		ThemeMediator.addThemeObserver(this);
		
        if(SETTINGS.REAL_TIME_SORT.getValue())
            DATA_MODEL.sort(DownloadDataLine.PRIORITY_INDEX); // ascending
	}

	/**
	 * Override the default refreshing so that we can
	 * set the clear button appropriately.
	 */
	public void doRefresh() {
	    boolean inactivePresent =
	        ((Boolean)DATA_MODEL.refresh()).booleanValue();
	    setButtonEnabled(DownloadButtons.CLEAR_BUTTON, inactivePresent);
        int[] selRows = TABLE.getSelectedRows();
        if (selRows.length > 0) {
            DownloadDataLine dataLine = 
                (DownloadDataLine)DATA_MODEL.get(selRows[0]);
            if (dataLine.getState() == Downloader.WAITING_FOR_USER)
                _downloadButtons.transformResumeButton();
            else
                _downloadButtons.transformSourcesButton();
            boolean inactive = dataLine.isDownloaderInactive();
            boolean pausable = !dataLine.getDownloader().isPaused() &&
                               !dataLine.getDownloader().isCompleted();                
            setButtonEnabled(DownloadButtons.RESUME_BUTTON, inactive);
            setButtonEnabled(DownloadButtons.PAUSE_BUTTON, pausable);
            setButtonEnabled(DownloadButtons.UP_BUTTON, inactive && pausable);
            setButtonEnabled(DownloadButtons.DOWN_BUTTON, inactive && pausable);
        }
	}

	/**
	 * Returns the total number of Downloads that have occurred in this session.
	 *
	 * @return the total number of Downloads that have occurred in this session
	 */
    public int getTotalDownloads() {
        return _totalDownloads;
    }

 	/**
	 * Returns the total number of current Downloads.
	 *
	 * @return the total number of current Downloads
	 */
    public int getCurrentDownloads() {
        return ((DownloadModel)DATA_MODEL).getCurrentDownloads();
    }

 	/**
	 * Returns the total number of active Downloads.
     * This includes anything that is still viewable in the Downloads view.
	 *
	 * @return the total number of active Downloads
	 */
    public int getActiveDownloads() {
        return ((DownloadModel)DATA_MODEL).getRowCount();
    }

    /**
     * Overrides the default add.
     *
	 * Adds a new Downloads to the list of Downloads, obtaining the necessary
	 * information from the supplied <tt>Downloader</tt>.
	 *
	 * If the download is not already in the list, then it is added.
     *  <p>
	 */
    public void add(Object downloader) {
        if ( !DATA_MODEL.contains(downloader) ) {
            _totalDownloads++;
            super.add(downloader);
        }
    }

	/**
	 * Overrides the default remove.
	 *
	 * Takes action upon downloaded theme files, asking if the user wants to
	 * apply the theme.
	 *
	 * Removes a download from the list if the user has configured their system
	 * to automatically clear completed download and if the download is
	 * complete.
	 *
	 * @param downloader the <tt>Downloader</tt> to remove from the list if it is
	 *  complete.
	 */
    public void remove(Object downloader) {
        Downloader dloader = (Downloader)downloader;
        int state = dloader.getState();
        
        if(state == Downloader.COMPLETE &&
         isThemeFile(dloader.getFileName())) {
            File themeFile = dloader.getDownloadFragment();
            themeFile = copyToThemeDir(themeFile);
            int response = GUIMediator.showYesNoMessage(
                "DOWNLOAD_APPLY_NEW_THEME_START",
                ThemeSettings.formatName(dloader.getFileName()),
                "DOWNLOAD_APPLY_NEW_THEME_END",
                QuestionsHandler.THEME_DOWNLOADED);
            if( response == GUIMediator.YES_OPTION ) {
		ThemeMediator.changeTheme(themeFile);
	    }
        }
        
        if(SharingSettings.CLEAR_DOWNLOAD.getValue()
		   && ( state == Downloader.COMPLETE ||
		        state == Downloader.ABORTED ) ) {
            super.remove(downloader);
		} else {
		    DownloadDataLine ddl = (DownloadDataLine)DATA_MODEL.get(downloader);
		    if (ddl != null) ddl.setEndTime(System.currentTimeMillis());
		}
    }
    
    private File copyToThemeDir(File themeFile) {
        File themeDir = ThemeSettings.THEME_DIR_FILE;
        File realLoc = new File(themeDir, themeFile.getName());
        // if they're the same, just use it.
        if( realLoc.equals(themeFile) )
            return themeFile;

        // otherwise, if the file already exists in the theme dir, remove it.
        realLoc.delete();
        
        // copy from shared to theme dir.
        CommonUtils.copy(themeFile, realLoc);
        return realLoc;
    }
    
    private boolean isThemeFile(String name) {
        return name.toLowerCase().endsWith(ThemeSettings.EXTENSION);
    }
    

	/**
	 * Launches the selected files in the <tt>Launcher</tt> or in the built-in
	 * media player.
	 */
	void launchSelectedDownloads() {
		final DataLine[] lines = TABLE.getSelectedDataLines();
        _audioLaunched = false;
		for(int i = 0; i < lines.length; i++) {
		    final Downloader dl = (Downloader)lines[i].getInitializeObject();
		    QUEUE.add(new Runnable() {
		        public void run() {
                    File toLaunch = dl.getDownloadFragment();
    				if (toLaunch == null) {
                        GUIMediator.showMessage("NO_PREVIEW_BEGIN", 
                                                dl.getFileName(),
                                                "NO_PREVIEW_END",
                                                QuestionsHandler.NO_PREVIEW_REPORT);
                        return;
                    }
    				if (!_audioLaunched && GUIMediator.isValidPlaylistFile(toLaunch)) {
    					GUIMediator.instance().launchAudio(toLaunch);
    					_audioLaunched = true;
    				} else {
    					try {
    						GUIMediator.launchFile(toLaunch);
    					} catch (IOException ignored) {}
    				}
                }
            });
        }
	}
	
	/**
	 * Pauses all selected downloads.
	 */
	void pauseSelectedDownloads() {
		DataLine[] lines = TABLE.getSelectedDataLines();
		for(int i = 0; i < lines.length; i++)
			((Downloader)lines[i].getInitializeObject()).pause();
	}
	
	/**
	 * Changes the priority of the selected downloads.
	 */
	void bumpPriority(final boolean up) {
	    DataLine[] lines = TABLE.getSelectedDataLines();
	    
	    // sort the lines by priority.
	    // this is necessary so that they move in the correct order
        Arrays.sort(lines, new Comparator() {
            public int compare(Object a, Object b) {
                int pa = ((Downloader)((DataLine)a).getInitializeObject()).
                            getInactivePriority();
                int pb = ((Downloader)((DataLine)b).getInitializeObject()).
                            getInactivePriority();
                return (pa < pb ? -1 : pa > pb ? 1 : 0) * ( up ? 1 : -1 );
            }
        });
        
	    for(int i = 0; i < lines.length; i++) {
            Downloader dl = (Downloader)lines[i].getInitializeObject();
            RouterService.getDownloadManager().bumpPriority(dl, up);
        }
    }
	
	/**
	 * Forces the selected downloads in the download window to resume.
	 */
	void resumeSelectedDownloads() {
		DataLine[] lines = TABLE.getSelectedDataLines();
		for(int i = 0; i < lines.length; i++) {
		    DownloadDataLine dd = (DownloadDataLine)lines[i];
			Downloader downloader = dd.getDownloader();
			try {
			    if(!dd.isCleaned())
				    downloader.resume();
			} catch (AlreadyDownloadingException e) {
				GUIMediator.showError("ERROR_ALREADY_DOWNLOADING",
									  "\""+e.getFilename()+"\".",
									  QuestionsHandler.ALREADY_DOWNLOADING);
			}
		}
	}

	/**
	 * Opens up a chat session with the selected hosts in the download
	 * window.
	 */
	void chatWithSelectedDownloads() {
		DataLine[] lines = TABLE.getSelectedDataLines();
		for(int i = 0; i < lines.length; i++) {
		    DataLine dl = lines[i];
			Downloader downloader=(Downloader)dl.getInitializeObject();
			Endpoint end = downloader.getChatEnabledHost();
			if (end!=null)
				RouterService.createChat(end.getAddress(), end.getPort());
		}
	}

	/**
	 * Opens up a browse session with the selected hosts in the download
	 * window.
	 */
	void browseSelectedDownloads() {
		DataLine[] lines = TABLE.getSelectedDataLines();
		for(int i = 0; i < lines.length; i++) {
		    DataLine dl = lines[i];
			Downloader downloader=(Downloader)dl.getInitializeObject();
			RemoteFileDesc end = downloader.getBrowseEnabledHost();
            if (end != null)
                SearchMediator.doBrowseHost(end);
		}
	}

	/**
	 * Handles a double-click event in the table.
	 */
	public void handleActionKey() {
		launchSelectedDownloads();
	}

	/**
	 * Clears the downloads in the download window that have completed.
	 */
	void clearCompletedDownloads() {
		((DownloadModel)DATA_MODEL).clearCompleted();
		clearSelection();
        setButtonEnabled(DownloadButtons.CLEAR_BUTTON, false);
	}

    // inherit doc comment
    protected JPopupMenu createPopupMenu() {
        JPopupMenu menu = (new DownloadPopupMenu(this)).getComponent();
        boolean selection = !TABLE.getSelectionModel().isSelectionEmpty();
		menu.getComponent(DownloadPopupMenu.KILL_INDEX).
            setEnabled(selection);
		menu.getComponent(DownloadPopupMenu.RESUME_INDEX).
            setEnabled(selection);
		menu.getComponent(DownloadPopupMenu.LAUNCH_INDEX).
            setEnabled(selection);
		menu.getComponent(DownloadPopupMenu.CHAT_INDEX).
            setEnabled(_chatEnabled);
		menu.getComponent(DownloadPopupMenu.BROWSE_INDEX).
            setEnabled(_browseEnabled);   
        return menu;
    }
    
	/**
	 * Handles the selection of the specified row in the download window,
	 * enabling or disabling buttons and chat menu items depending on
	 * the values in the row.
	 *
	 * @param row the selected row
	 */
	public void handleSelection(int row) {

		DownloadDataLine dataLine = (DownloadDataLine)DATA_MODEL.get(row);
		_chatEnabled = dataLine.getChatEnabled();
        _browseEnabled = dataLine.getBrowseEnabled();
        boolean inactive = dataLine.isDownloaderInactive();
        boolean pausable = !dataLine.getDownloader().isPaused() &&
                           !dataLine.getDownloader().isCompleted();

        if (dataLine.getState() == Downloader.WAITING_FOR_USER)
            _downloadButtons.transformResumeButton();
        else
            _downloadButtons.transformSourcesButton();
		
		setButtonEnabled(DownloadButtons.KILL_BUTTON, true);
		setButtonEnabled(DownloadButtons.RESUME_BUTTON, inactive);
		setButtonEnabled(DownloadButtons.LAUNCH_BUTTON, true);
        setButtonEnabled(DownloadButtons.PAUSE_BUTTON, pausable);
        setButtonEnabled(DownloadButtons.UP_BUTTON, inactive && pausable);
        setButtonEnabled(DownloadButtons.DOWN_BUTTON, inactive && pausable);
		
	}

	/**
	 * Handles the deselection of all rows in the download table,
	 * disabling all necessary buttons and menu items.
	 */
	public void handleNoSelection() {
        _chatEnabled = false;
        _browseEnabled = false;

		setButtonEnabled(DownloadButtons.KILL_BUTTON, false);
		setButtonEnabled(DownloadButtons.RESUME_BUTTON, false);
		setButtonEnabled(DownloadButtons.LAUNCH_BUTTON, false);
        setButtonEnabled(DownloadButtons.PAUSE_BUTTON, false);
        setButtonEnabled(DownloadButtons.UP_BUTTON, false);
        setButtonEnabled(DownloadButtons.DOWN_BUTTON, false);
	}
}
