package com.limegroup.gnutella.gui.library;

import java.awt.Frame;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.IOException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Arrays;

import javax.swing.JComponent;
import javax.swing.JPopupMenu;
import javax.swing.JTable;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import javax.swing.ListSelectionModel;
import javax.swing.event.MouseInputListener;
import javax.swing.table.TableCellEditor;

import com.limegroup.gnutella.FileDesc;
import com.limegroup.gnutella.RouterService;
import com.limegroup.gnutella.URN;
import com.limegroup.gnutella.UploadManager;
import com.limegroup.gnutella.downloader.AlreadyDownloadingException;
import com.limegroup.gnutella.downloader.CantResumeException;
import com.limegroup.gnutella.gui.ButtonRow;
import com.limegroup.gnutella.gui.GUIMediator;
import com.limegroup.gnutella.gui.tables.AbstractTableMediator;
import com.limegroup.gnutella.gui.tables.DragManager;
import com.limegroup.gnutella.gui.tables.LimeJTable;
import com.limegroup.gnutella.gui.themes.ThemeMediator;
import com.limegroup.gnutella.gui.tables.DataLine;
import com.limegroup.gnutella.gui.xml.MetaEditorFrame;
import com.limegroup.gnutella.gui.xml.editor.MetaEditor;
import com.limegroup.gnutella.settings.ConnectionSettings;
import com.limegroup.gnutella.settings.QuestionsHandler;
import com.limegroup.gnutella.util.Launcher;
import com.limegroup.gnutella.util.NetworkUtils;
import com.limegroup.gnutella.xml.LimeXMLUtils;
import com.limegroup.gnutella.licenses.License;
import com.limegroup.gnutella.licenses.VerificationListener;
import com.limegroup.gnutella.gui.LicenseWindow;

/**
 * This class wraps the JTable that displays files in the library,
 * controlling access to the table and the various table properties.
 * It is the Mediator to the Table part of the Library display.
 */
//2345678|012345678|012345678|012345678|012345678|012345678|012345678|012345678|
final class LibraryTableMediator extends AbstractTableMediator
                                implements VerificationListener {

	/**
	 * Reference to the <tt>LibraryMediator</tt> that acts as the hub of
	 * communication for library components.
	 */
	private static final LibraryMediator LIBRARY_MEDIATOR =
	    LibraryMediator.instance();

    /**
     * Variables so the PopupMenu & ButtonRow can have the same listeners
     */
    public static ActionListener LAUNCH_LISTENER;
    public static ActionListener ADD_PLAY_LIST_LISTENER;
    public static ActionListener ANNOTATE_LISTENER;
    public static ActionListener BITZI_LOOKUP_LISTENER;
    public static ActionListener MAGNET_LOOKUP_LISTENER;
    public static ActionListener HTTP_LOOKUP_LISTENER;
    public static ActionListener RESUME_LISTENER;
    public static ActionListener LICENSE_LISTENER;
    public static ActionListener RENAME_LISTENER;

    /**
     * Whether or not annotation is allowed (when the XML is finished loading)
     */
    private boolean _allowAnnotate;
    
    /**
     * Whether or not the incomplete directory is selected.
     */
    private boolean _isIncomplete;

    /**
     * instance, for singelton access
     */
    private static LibraryTableMediator _instance = new LibraryTableMediator();

    public static LibraryTableMediator instance() { return _instance; }

    /** useful for other library components to use
     */
    public static boolean shouldShowHTTPOption() {
        return RouterService.acceptedIncomingConnection() &&
               !(NetworkUtils.isPrivate() &&
                 ConnectionSettings.LOCAL_IS_PRIVATE.getValue());
    }

    /**
     * Build some extra listeners
     */
    protected void buildListeners() {
        super.buildListeners();
        LAUNCH_LISTENER = new LaunchListener();
        ADD_PLAY_LIST_LISTENER = new AddPLFileListener();
        ANNOTATE_LISTENER = new AnnotateListener();
        BITZI_LOOKUP_LISTENER = new BitziLookupListener();
        MAGNET_LOOKUP_LISTENER = new MagnetLookupListener();
        HTTP_LOOKUP_LISTENER = new HttpPageListener();
        RESUME_LISTENER = new ResumeListener();
        LICENSE_LISTENER = new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                showLicenseWindow();
            }
        };
        RENAME_LISTENER = new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                startRename();
            }
        };
    }

    /**
     * Set up the constants
     */
    protected void setupConstants() {
		MAIN_PANEL = null;
		DATA_MODEL = new LibraryTableModel();
		TABLE = new LimeJTable(DATA_MODEL);
		((LibraryTableModel)DATA_MODEL).setTable(TABLE);
		BUTTON_ROW = (new LibraryTableButtons(this)).getComponent();
    }

    // inherit doc comment
    protected JPopupMenu createPopupMenu() {
        JPopupMenu menu = (new LibraryTablePopupMenu(this)).getComponent();
        
        boolean selection = !TABLE.getSelectionModel().isSelectionEmpty();
        
        if ( GUIMediator.isPlaylistVisible() ) {
            menu.getComponent(LibraryTablePopupMenu.PLAYLIST_INDEX).
                setEnabled(selection);
        }
        
        DataLine dl = TABLE.getSelectedDataLine();
        boolean license = dl != null && ((LibraryTableDataLine)dl).isLicensed();
        

        menu.getComponent(LibraryTablePopupMenu.ANNOTATE_INDEX).
            setEnabled(selection && _allowAnnotate && !_isIncomplete);
        menu.getComponent(LibraryTablePopupMenu.RENAME_INDEX).
            setEnabled(selection && !_isIncomplete);

        menu.getComponent(LibraryTablePopupMenu.RESUME_INDEX).
            setEnabled(selection && _isIncomplete);

        menu.getComponent(LibraryTablePopupMenu.LAUNCH_INDEX).
            setEnabled(selection);
        menu.getComponent(LibraryTablePopupMenu.DELETE_INDEX).
            setEnabled(selection);
        menu.getComponent(LibraryTablePopupMenu.BITZI_INDEX).
            setEnabled(selection);            
        menu.getComponent(LibraryTablePopupMenu.MAGNET_INDEX).
            setEnabled(selection);
        menu.getComponent(LibraryTablePopupMenu.LICENSE_INDEX).
            setEnabled(license);

        if (shouldShowHTTPOption())
            menu.getComponent(LibraryTablePopupMenu.HTTP_INDEX).
                setEnabled(selection);

        return menu;
    }

    /**
     * Upgrade getScrolledTablePane to public access for the LibraryConstructor
     */
    public JComponent getScrolledTablePane() {
        return super.getScrolledTablePane();
    }

    /* Don't display anything for this.  The LibraryMediator will do it. */
	protected void updateSplashScreen() {}

    /**
     * Note: This is set up for this to work.
     * Polling is not needed though, because updates
     * already generate update events.
     */
    private LibraryTableMediator() {
        super("LIBRARY_TABLE");
        //GUIMediator.addRefreshListener(this);
        ThemeMediator.addThemeObserver(this);
    }
    
    /**
     * Sets up drag & drop for the table.
     */
    protected void setupDragAndDrop() {
        DragManager.install(TABLE);
    }

	/**
	 * there is no actual component that holds all of this table.
	 * The LibraryMediator is real the holder.
	 */
	public JComponent getComponent() {
		return null;
	}
	
    /**
     * Sets the default editors.
     */
    protected void setDefaultEditors() {
        TableColumnModel model = TABLE.getColumnModel();
        TableColumn tc = model.getColumn(LibraryTableDataLine.NAME_IDX);
        tc.setCellEditor(new LibraryTableCellEditor(this));
    }


	/**
	 * Cancels all editing of fields in the tree and table.
	 */
	void cancelEditing() {
		if(TABLE.isEditing()) {
			TableCellEditor editor = TABLE.getCellEditor();
			editor.cancelCellEditing();
		}	    
	}

	/**
	 * Adds the mouse listeners to the wrapped <tt>JTable</tt>.
	 *
	 * @param listener the <tt>MouseInputListener</tt> that handles mouse events
	 *                 for the library
	 */
	void addMouseInputListener(final MouseInputListener listener) {
        TABLE.addMouseListener(listener);
        TABLE.addMouseMotionListener(listener);
	}

	/**
	 * Allows or disallows annotation
	 *
	 * @param enabled whether or not annotation is allowed
	 */
	public void setAnnotateEnabled(boolean enabled) {
	    _allowAnnotate = enabled;
	    LibraryTableDataLine.setXMLEnabled(enabled);
	    DATA_MODEL.refresh();
	    
	    //disable the annotate buttons if we are turning annotation off.
	    if ( ! enabled ) {
		    setButtonEnabled(LibraryTableButtons.ANNOTATE_BUTTON, false);
		//if it's turning on and something is selected, enable the annotate button.
	    } else if ( !TABLE.getSelectionModel().isSelectionEmpty() ) {
            setButtonEnabled(LibraryTableButtons.ANNOTATE_BUTTON, true && !_isIncomplete);
        }        
	}

	/**
	 * Notifcation that the incomplete directory is selected (or not)
	 *
	 * @param enabled whether or not incomplete is showing
	 */
	void setIncompleteSelected(boolean enabled) {
	    _isIncomplete = enabled;
	    //disable the resume buttons if we're not incomplete
	    if ( ! enabled ) {
	        setButtonEnabled(LibraryTableButtons.RESUME_BUTTON, false);
	    } else if ( !TABLE.getSelectionModel().isSelectionEmpty() ) {
	        setButtonEnabled(LibraryTableButtons.RESUME_BUTTON, true);
	    }
	    
	    // enable the annotate button appropriately
        setButtonEnabled(LibraryTableButtons.ANNOTATE_BUTTON,
                         _allowAnnotate && !enabled);
	}


    /**
	 * Returns the <tt>File</tt> stored at the specified row in the list.
	 *
	 * @param row the row of the desired <tt>File</tt> instance in the
	 *            list
	 *
	 * @return a <tt>File</tt> instance associated with the specified row
	 *         in the table
	 */
    File getFile(int row) {
		return ((LibraryTableModel)DATA_MODEL).getFile(row);
    }
    
    /**
	 * Accessor for the table that this class wraps.
	 *
	 * @return The <tt>JTable</tt> instance used by the library.
	 */
    JTable getTable() {
        return TABLE;
    }

    ButtonRow getButtonRow() {
        return BUTTON_ROW;
    }

	/**
	 * Accessor for the <tt>ListSelectionModel</tt> for the wrapped
	 * <tt>JTable</tt> instance.
	 */
	ListSelectionModel getSelectionModel() {
		return TABLE.getSelectionModel();
	}


    /**
     * shows the user a meta-data for the file(if any) and allow the user
     * to edit it.
     */
    void editMeta(){
        //get the selected file. If there are more than 1 we just use the
        // last one.
        int[] rows = TABLE.getSelectedRows();
        int k = rows.length;
        if(k == 0)
            return;
        int index = rows[k-1];//this is the index of the last row selected
        FileDesc fd = ((LibraryTableModel)DATA_MODEL).getFileDesc(index);
        if( fd == null ) // oh well
            return;

        String fullName = "";
        try{
            fullName = fd.getFile().getCanonicalPath();
        }catch(IOException ee){//if there is an exception return
            return;
        }
        //Now docsOfFile has all LimeXMLDocuments pertainint to selected file
        Frame mainFrame = GUIMediator.getAppFrame();
        
        if (LimeXMLUtils.isSupportedAudioFormat(fullName)) {
            MetaEditor metaEditor = new MetaEditor(fd, fullName, mainFrame);
            metaEditor.show();
        } else {
            MetaEditorFrame metaEditor = new MetaEditorFrame(fd, fullName, mainFrame);
            metaEditor.show();
        }
    }

	/**
	 * Performs the Bitzi lookup for the selected files in the library.
	 */
    void doBitziLookup() {
        // get the selected file. If there are more than 1 we just use the
        // last one.
        int[] rows = TABLE.getSelectedRows();
        int k = rows.length;
        if(k == 0)
            return;
        int index = rows[k-1];//this is the index of the last row selected
        FileDesc fd = ((LibraryTableModel)DATA_MODEL).getFileDesc(index);
        if (fd==null) {
            // noop
            return;
        }
        URN urn = fd.getSHA1Urn();
		if(urn==null) {
            // unable to do lookup -- perhaps SHA1 not calculated yet?;
            // TODO could show dialog suggesting user try again later but won't for now
            return;
        }
        String urnStr = urn.toString();
        int hashstart = 1+urnStr.indexOf(":",4);
        // TODO: grab this lookup URL from a template somewhere
        String lookupUrl = "http://bitzi.com/lookup/"+urnStr.substring(hashstart)+"?ref=limewire";
        try {
            Launcher.openURL(lookupUrl);
        } catch (IOException ioe) {
            // do nothing
        }
    }
    
    /**
     * Programatically starts a rename of the selected item.
     */
    void startRename() {
        int row = TABLE.getSelectedRow();
        if(row == -1)
            return;
        int viewIdx = TABLE.convertColumnIndexToView(LibraryTableDataLine.NAME_IDX);
        TABLE.editCellAt(row, viewIdx, LibraryTableCellEditor.EVENT);
    }
    
    /**
     * Shows the license window.
     */
    void showLicenseWindow() {
        DataLine dl = TABLE.getSelectedDataLine();
        if(dl == null)
            return;

        LibraryTableDataLine ldl = (LibraryTableDataLine)dl;
        FileDesc fd = ldl.getFileDesc();
        License license = fd.getLicense();
        URN urn = fd.getSHA1Urn();
        LicenseWindow window = new LicenseWindow(license, urn, this);
        window.show();
    }
    
    public void licenseVerified(License license) {
        DATA_MODEL.refresh();
    }

    /**
     * Prepare a detail page of magnet link info for selected files 
     * in the library.
     */
    void doMagnetLookup() {
        doMagnetCommand("/magcmd/detail?");
    }
    /**
     * Fire a local lookup with file/magnet details
     */
    void doMagnetCommand(String cmd) {
        // get the selected files.  Build up a url to display details.
        int[] rows = TABLE.getSelectedRows();
        int k = rows.length;
        if(k == 0)
            return;

        boolean haveValidMagnet = false;

        int    count     = 0;
        int    port      = RouterService.getHTTPAcceptor().getPort();
        int    eport     = RouterService.getAcceptor().getPort(true);
        byte[] eaddr     = RouterService.getAcceptor().getAddress(true);
        String lookupUrl = "http://localhost:"+port+
          cmd+
          "addr="+NetworkUtils.ip2string(eaddr)+":"+eport;
        for(int i=0; i<k; i++) {
            FileDesc fd = ((LibraryTableModel)DATA_MODEL).getFileDesc(rows[i]);
            if (fd==null) {
                // Only report valid files
                continue;
            }
            URN    urn       = fd.getSHA1Urn();
            String urnStr    = urn.toString();
            int    hashstart = 1+urnStr.indexOf(":",4);
            if(urn==null) {
                // Only report valid sha1s
                continue;
            } 
            String sha1=urnStr.substring(hashstart);
            lookupUrl +=
              "&n"+count+"="+URLEncoder.encode(fd.getName())+
              "&u"+count+"="+sha1;
            count++;
            haveValidMagnet = true;
        }
        if (haveValidMagnet) {
            try {
                Launcher.openURL(lookupUrl);
            } catch (IOException ioe) {
                // do nothing
            }
        }
    }

    /**
     * Fire a local lookup with file/magnet details
     */
    void doHttpPageCommand() {
        // get the selected files.  Build up a url and display it.

        // sanity check
        int[] rows = TABLE.getSelectedRows();
        int k = rows.length;
        if(k == 0)
            return;
        if (!shouldShowHTTPOption())
            return;

        // set up the URL
        final int port = RouterService.getPort();
        final String addr = NetworkUtils.ip2string(RouterService.getAddress());

        StringBuffer lookupURL = new StringBuffer();
        lookupURL.append("http://" + addr + ":" + port);
        lookupURL.append(UploadManager.FV_REQ_BEGIN + "/");
        lookupURL.append(UploadManager.FV_PASS + "/");
        // add indices
        for(int i=0; i<k; i++) {
            FileDesc fd = ((LibraryTableModel)DATA_MODEL).getFileDesc(rows[i]);
            if (fd == null) // Only report valid files
                continue;
            if (fd.getIndex() > -1) {
                lookupURL.append(fd.getIndex());
                if ((i+1) < k) lookupURL.append("&");
            }
        }

        try {
            Launcher.openURL(lookupURL.toString());
        } catch (IOException ioe) {
            // do nothing
        }
    }


    /**
     * Override the default removal so we can actually stop sharing
     * and delete the file.
	 * Deletes the selected rows in the table.
	 * CAUTION: THIS WILL DELETE THE FILE FROM THE DISK.
	 */
	public void removeSelection() {
		String msgKey = "MESSAGE_CONFIRM_FILE_DELETE";
		int response = GUIMediator.showYesNoMessage(msgKey);
		if(response != GUIMediator.YES_OPTION) return;

        int[] rows = TABLE.getSelectedRows();
		if(rows.length <= 0) return;

		Arrays.sort(rows);

		if(TABLE.isEditing()) {
			TableCellEditor editor = TABLE.getCellEditor();
			editor.cancelCellEditing();
		}
		
		ArrayList errors = new ArrayList();

		for(int i = rows.length - 1; i >= 0; i--) {
			File file = ((LibraryTableModel)DATA_MODEL).getFile(rows[i]);
            RouterService.getFileManager().removeFileIfShared(file);
            boolean removed = file.delete();
            if(!removed) {
                // try again, telling UploadManager to kill any uploads
                FileDesc fd = ((LibraryTableModel)DATA_MODEL).getFileDesc(rows[i]);
                if(fd != null) {
                    RouterService.getUploadManager().killUploadsForFileDesc(fd);
                    removed = file.delete();
                }
            }
            
			if(removed)
			    DATA_MODEL.remove(rows[i]);
			else
			    errors.add(file.getName());
		}

		clearSelection();		
		
		// go through the errors and tell them what couldn't be deleted.
		for(int i = 0; i < errors.size(); i++) {
		    String name = (String)errors.get(i);
			final String key1 = "MESSAGE_UNABLE_TO_DELETE_FILE_START";
			final String key2 = "MESSAGE_UNABLE_TO_DELETE_FILE_END";
			final String msg = "'" + name + "'.";
			// notify the user that deletion failed
			GUIMediator.showError(key1, msg, key2);
		}
    }
    
	/**
	 * Handles a name change of one of the files displayed.
	 *
	 * @param newName The new name of the file
	 *
	 * @return A <tt>String</tt> that is the name of the file
	 *         after this method is called. This is the new name if
	 *         the name change succeeded, and the old name otherwise.
	 */
	String handleNameChange(String newName) {
		int row = TABLE.getEditingRow();
		LibraryTableModel ltm = (LibraryTableModel)DATA_MODEL;
		
		
		File oldFile = ltm.getFile(row);
		String parent = oldFile.getParent();
		String nameWithExtension = newName + "." + ltm.getType(row);
		File newFile = new File(parent, nameWithExtension);
		if(!ltm.getName(row).equals(newName)) {
			if(oldFile.renameTo(newFile)) {
				ltm.handleFileChangeInternal(row, newFile);			    
				RouterService.getFileManager().renameFileIfShared(oldFile, newFile);
				return newName;
			} else {
				// notify the user that renaming failed
				GUIMediator.showError("MESSAGE_UNABLE_TO_RENAME_FILE_START",
				                      "'" + ltm.getName(row) + "'.",
				                      "MESSAGE_UNABLE_TO_RENAME_FILE_END");
				return ltm.getName(row);
			}
		} else {
		    return newName; // newName == currentName
        }
	}


    public void handleActionKey() {
        launch();
    }

    /**
     * Resume incomplete downloads
     */    
    void resumeIncomplete() {        
        //For each selected row...
        int[] rows = TABLE.getSelectedRows();
        boolean startedDownload=false;
        ArrayList errors = new ArrayList();
        for (int i=0; i<rows.length; i++) {
            //...try to download the incomplete
            File incomplete=((LibraryTableModel)DATA_MODEL).getFile(rows[i]);
            try {
                RouterService.download(incomplete);
                startedDownload=true;
            } catch (AlreadyDownloadingException e) { 
                // we must cache errors to display later so we don't wait
                // while the table might change in the background.
                errors.add(e);
            } catch(CantResumeException e) {
                errors.add(e);
            }
        }
        
        // traverse back through the errors and show them.
        for(int i = 0; i < errors.size(); i++) {
            Exception e = (Exception)errors.get(i);
            if( e instanceof AlreadyDownloadingException ) {
                GUIMediator.showError("ERROR_ALREADY_DOWNLOADING",
                    "\""+((AlreadyDownloadingException)e).getFilename()+"\".",
                    QuestionsHandler.ALREADY_DOWNLOADING);
            } else if ( e instanceof CantResumeException ) {
                GUIMediator.showError("ERROR_CANT_RESUME_START",
                    "\""+((CantResumeException)e).getFilename()+"\"",
                    "ERROR_CANT_RESUME_END",
                    QuestionsHandler.CANT_RESUME);
            }
        }       

        //Switch to download tab (if we actually started anything).
        if (startedDownload)
            GUIMediator.instance().setWindow(GUIMediator.SEARCH_INDEX);
    }

    

    /**
	 * Launches the associated applications for each selected file
	 * in the library if it can.
	 */
    void launch() {
        int[] rows = TABLE.getSelectedRows();
        boolean audioLaunched = false;
		for(int i = 0, l = rows.length; i < l; i++) {
            File currFile = ((LibraryTableModel)DATA_MODEL).getFile(rows[i]);
            if (!audioLaunched && GUIMediator.isValidPlaylistFile(currFile)) {
					LIBRARY_MEDIATOR.launchAudio(currFile);
					audioLaunched = true;
            } else {
                try {
                    GUIMediator.launchFile(currFile);
                } catch(IOException ignored) {}
            }
		}
    }

	/**
	 * Handles the selection of the specified row in the library window,
	 * enabling or disabling buttons and chat menu items depending on
	 * the values in the row.
	 *
	 * @param row the selected row
	 */
	public void handleSelection(int row) {
		setButtonEnabled(LibraryTableButtons.LAUNCH_BUTTON, true);
		setButtonEnabled(LibraryTableButtons.DELETE_BUTTON, true);

		if ( GUIMediator.isPlaylistVisible() ) {
		    setButtonEnabled(LibraryTableButtons.PLAYLIST_BUTTON, true);
		}

		if ( _allowAnnotate && !_isIncomplete ) {
		    setButtonEnabled(LibraryTableButtons.ANNOTATE_BUTTON, true);
		}

		if ( _isIncomplete ) {
		    setButtonEnabled(LibraryTableButtons.RESUME_BUTTON, true);
		}
	}

	/**
	 * Handles the deselection of all rows in the library table,
	 * disabling all necessary buttons and menu items.
	 */
	public void handleNoSelection() {
		setButtonEnabled(LibraryTableButtons.LAUNCH_BUTTON, false);
		setButtonEnabled(LibraryTableButtons.DELETE_BUTTON, false);

		if ( GUIMediator.isPlaylistVisible() ) {
		    setButtonEnabled(LibraryTableButtons.PLAYLIST_BUTTON, false);
		}

        setButtonEnabled(LibraryTableButtons.ANNOTATE_BUTTON, false);
		setButtonEnabled(LibraryTableButtons.RESUME_BUTTON, false);
	}

    ///////////////////////
    // A collection of private listeners
    //////////////////////

    private final class AnnotateListener implements ActionListener {
    	public void actionPerformed(ActionEvent ae) {
    		editMeta();
    	}
    }

    private final class ResumeListener implements ActionListener {
    	public void actionPerformed(ActionEvent ae) {
            resumeIncomplete();
    	}
    }

    private final class BitziLookupListener implements ActionListener {
        public void actionPerformed(ActionEvent e){
            doBitziLookup();
        }
    }

    private final class MagnetLookupListener implements ActionListener {
        public void actionPerformed(ActionEvent e){
            doMagnetLookup();
        }
    }

    private final class HttpPageListener implements ActionListener {
        public void actionPerformed(ActionEvent e){
            doHttpPageCommand();
        }
    }

    private final class LaunchListener implements ActionListener {

    	public void actionPerformed(ActionEvent ae) {
    		launch();
    	}
    }


    private final class AddPLFileListener implements ActionListener{
        public void actionPerformed(ActionEvent ae){
			//get the selected file. If there are more than 1 we add all
			int[] rows = TABLE.getSelectedRows();
			for (int i = 0; i < rows.length; i++) {
				int index = rows[i]; // current index to add
				File file = ((LibraryTableModel)DATA_MODEL).getFile(index);
				LIBRARY_MEDIATOR.addFileToPlayList(file);
			}
        }
    }
}
