package com.limegroup.gnutella.gui.xml;

import java.awt.CardLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Insets;
import java.awt.Rectangle;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;

import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.Scrollable;
import javax.swing.SwingConstants;
import javax.swing.text.Document;

import com.limegroup.gnutella.gui.AutoCompleteTextField;
import com.limegroup.gnutella.gui.BoxPanel;
import com.limegroup.gnutella.gui.GUIMediator;
import com.limegroup.gnutella.gui.search.DisplayManager;
import com.limegroup.gnutella.gui.search.SearchField;
import com.limegroup.gnutella.xml.LimeXMLSchema;
import com.limegroup.gnutella.xml.LimeXMLUtils;
import com.limegroup.gnutella.xml.SchemaFieldInfo;
import com.limegroup.gnutella.xml.XMLStringUtils;
import com.limegroup.gnutella.util.NameValue;

/**
 * This panel is used to indent the labels and text fields of a schema.
 * <p>
 * This class is extended by the InputPanel and the Output panel.
 *
 * @author Sumeet Thadani 
 */
public abstract class IndentingPanel extends JPanel implements Scrollable {

    public static final String CHECKBOX_STRING =
        GUIMediator.getStringResource("XML_SEARCH_MORE_OPTIONS_LABEL");
    public static final String MINI_VIEW = "Mini-View";
    public static final String MAX_VIEW  = "Max-View";

    protected HashMap nameToField; // maps the names to TextFields
    
    /**
     * Constant for the number of pixels to offset fields for each successive
     * indentation if indenting is turned on.
     */
    private final static int PIXEL_OFFSET = 6;

    private JPanel expandablePanel;
    
    /**
     * Handler for the "more options" panel that is not always visible.  The 
     * handler is necessary to update theme changes which otherwise would
     * not happen since they're not visible.
     */
    private final MoreOptionsHandler MORE_OPTIONS_HANDLER =
        new MoreOptionsHandler();
        
    private CardLayout advancedCards;

    private int defaultFields = 4;
    
    private String ancestorCanonicalized = "";
    
    /**
     * The checkbox for more options, so that other things can add
     * listeners to its state changing.
     */
    private JCheckBox _moreOptions;
    
    /**
     * The first TextField for input, used to quickly set the focus on 
     * the text field when this panel wants focus.
     */
    private Component focusRequestor;

    // Constructor
    public IndentingPanel(LimeXMLSchema schema, ActionListener listener, 
                          Document document, boolean expand, boolean indent, boolean searching) {
        DisplayManager dispMan = DisplayManager.instance();

        if (expand) {
            int s = schema.getCanonicalizedFields().size();
            defaultFields = s;
        }
        this.setOpaque(false);
        this.setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
        nameToField = new HashMap();
        List fields = schema.getCanonicalizedFields();
        int indentsToClose = 0;
        String currField;
        int common =0;
        int z = fields.size();
        Component filler = null;
        boolean firstTextFieldAdded = false;
        
        for (int i = 0; i < z ; i++) {
            SchemaFieldInfo infoField = (SchemaFieldInfo)fields.get(i);
            if(!searching && !infoField.isEditable())
                continue;
                
            currField = infoField.getCanonicalizedFieldName();
            
            List currFieldList = XMLStringUtils.split(currField);
            // remove top level since its like audios, but keep it around in
            // the ancestor string
            ancestorCanonicalized = (String)currFieldList.remove(0);
            int numParents = currFieldList.size() - 1;
            if (i > 0) { //after the first case
                String prev;
                String now;
                SchemaFieldInfo p = (SchemaFieldInfo)fields.get(i-1);
                prev = p.getCanonicalizedFieldName();
                SchemaFieldInfo n = (SchemaFieldInfo)fields.get(i);
                now = n.getCanonicalizedFieldName();
                List currFields = XMLStringUtils.split(prev);
                currFields.remove(0); // remove top level - like audios
                List prevField = XMLStringUtils.split(now);
                prevField.remove(0); // remove top level - like audios
                common = getCommonCount(currFields, prevField);
            }
            indentsToClose = common;
            // deal with parents
            JPanel fieldPanel;
            JPanel outerField = new JPanel(new FlowLayout(FlowLayout.LEFT, 0, 0));
            outerField.setOpaque(false);
            // get the fieldname to look up internationalized properties
            for (int k = 0; k < common; k++) {
                ancestorCanonicalized +=
                XMLStringUtils.DELIMITER + (String)currFieldList.get(k);
            }
            for (int j = common; j < numParents; j++) { // parents dont have text field
                fieldPanel = new JPanel(new FlowLayout(FlowLayout.LEFT, 0, 0)); // create new panel
                fieldPanel.setOpaque(false);
                String backupFieldName = (String)currFieldList.get(j);
                String key = ancestorCanonicalized +
                             XMLStringUtils.DELIMITER +
                             backupFieldName;
                String fieldName = 
                    dispMan.getDisplayName(key, schema.getDescription());
                // Note: There should be no need to strip preferred size.
                JLabel label = new JLabel(LimeXMLUtils.
                                          capitalizeFirst(fieldName));
                if (indent) {
                    for (int k = 0; k < indentsToClose; k++) {
                        filler = Box.createHorizontalStrut(PIXEL_OFFSET);
                        outerField.add(filler);
                    }
                }
                // label.setForeground(Color.black);
                //fieldPanel.add(label);
                outerField.add(fieldPanel);
                if (i < defaultFields)
                    this.add(outerField);
                else  {
                    //MORE_OPTIONS_PANEL.add(outerField);
                }
                indentsToClose++;
            } // end of parents

            // create new pane
            outerField = new JPanel(new FlowLayout(FlowLayout.LEFT, 0, 0));
            fieldPanel = new BoxPanel(BoxPanel.Y_AXIS);
            fieldPanel.setOpaque(false);
            outerField.setOpaque(false);
            if (indent){
                for (int l = 0; l < indentsToClose; l++) {
                    filler = Box.createHorizontalStrut(PIXEL_OFFSET);
                    outerField.add(filler);
                }
            }

            String backupFieldName = (String)currFieldList.get(numParents);
            String fieldName = 
                dispMan.getDisplayName(currField, schema.getDescription());
            fieldName = DisplayManager.stripPreferredSize(fieldName);
            int type = infoField.getFieldType();
            String title = LimeXMLUtils.capitalizeFirst(fieldName);

            if (type == SchemaFieldInfo.TEXTFIELD) {
                AutoCompleteTextField textField;
                if(searching)
                    textField = new SearchField(14);
                else
                    textField = new AutoCompleteTextField(30);
                if(!firstTextFieldAdded) {
                    firstTextFieldAdded = true;
                    focusRequestor = textField;
                    if(document != null)
                        textField.setDocument(document);
                }
                textField.setMargin(new Insets(0, 0, 0, 0));
                nameToField.put(currField.toLowerCase(Locale.US),
                                textField); 
                if (i >= defaultFields)  {
                    MORE_OPTIONS_HANDLER.addTextField(textField, title);
                }  else  {
                    fieldPanel.add(createLabel(title));
                    fieldPanel.add(Box.createVerticalStrut(3));
                    fieldPanel.add(textField);
                }
                addTextFieldToList(textField);
                if (listener != null) {
                    textField.addActionListener(listener);
                }
            } else if (type == SchemaFieldInfo.OPTIONS) {
                List values = infoField.getEnumerationList();
                int d = values.size();
                ComboBoxValue[] vals = new ComboBoxValue[d + 1];
                vals[0] = new ComboBoxValue("", "");
                for (int m = 0; m < d; m++)
                    vals[m + 1] = new ComboBoxValue((NameValue)values.get(m));
                Arrays.sort(vals);
                JComboBox comboBox = new JComboBox(vals);
                comboBox.setOpaque(false);
                nameToField.put(currField.toLowerCase(Locale.US), comboBox);
                fieldPanel.add(createLabel(title));
                fieldPanel.add(Box.createVerticalStrut(3));
                fieldPanel.add(comboBox);
                addComboBoxToList(comboBox);
                if (i >= defaultFields)
                    MORE_OPTIONS_HANDLER.addComboBox(comboBox, title);
            }
            
            outerField.add(fieldPanel);

            if (i < defaultFields) 
                this.add(outerField);

            if (i == defaultFields) {
                // set up switchable card layout....
                advancedCards = new CardLayout(0,0);
                expandablePanel = new JPanel(advancedCards);
                expandablePanel.setOpaque(false);
                
                // create switcher checkbox...
                JPanel tempPanel = new JPanel(new FlowLayout(FlowLayout.LEFT, 0, 0));
                tempPanel.setOpaque(false);
                final JCheckBox moreOptions = new JCheckBox(CHECKBOX_STRING);
                moreOptions.setPreferredSize(new Dimension(600, 20));
                moreOptions.setOpaque(false);
                //moreOptions.setBorder(BorderFactory.createEmptyBorder());
                moreOptions.addActionListener(new java.awt.event.ActionListener() {
                    public void actionPerformed(java.awt.event.ActionEvent e) {
                        if (moreOptions.isSelected()) {
                            expandablePanel.add(
                                MORE_OPTIONS_HANDLER.createPanel(), 
                                MAX_VIEW);
                            //expandablePanel.add(MORE_OPTIONS_PANEL, MAX_VIEW);                
                            advancedCards.show(expandablePanel, MAX_VIEW);
                            expandablePanel.revalidate();
                        } else {
                            expandablePanel.remove(
                                MORE_OPTIONS_HANDLER.getPanel());
                            advancedCards.show(expandablePanel, MINI_VIEW);
                            expandablePanel.revalidate();
                        }
                    }
                });
                _moreOptions = moreOptions;

                // put it in the panel....
                tempPanel.add(moreOptions);
                this.add(tempPanel);
                
                // continue set up switchable panel....
                JPanel dummyPanel = new JPanel();
                dummyPanel.setOpaque(false);
                expandablePanel.add(dummyPanel, MINI_VIEW);                
                advancedCards.show(expandablePanel, MINI_VIEW);
                
                this.add(expandablePanel);
            }

           // if (i >= defaultFields)
             //   MORE_OPTIONS_PANEL.add(outerField);

        } // end of for
    }
    
    /**
     * Adds an actionlistener to the more options checkbox.
     */
    public void addMoreOptionsListener(ActionListener l) {
        if(_moreOptions != null)
            _moreOptions.addActionListener(l);
    }
    
    /**
     * Requests focus for the focusRequestor instead of this.
     */
    public void requestFirstFocus() {
        if(focusRequestor != null)
            focusRequestor.requestFocus();
        else
            super.requestFocus();
    }

    static JPanel createLabel(String text) {
        JPanel labelPanel = new BoxPanel(BoxPanel.X_AXIS);
        labelPanel.setOpaque(false);
        JLabel inner = new JLabel(text);
        inner.setPreferredSize(new Dimension(100, 20));
        labelPanel.add(inner);
        labelPanel.add(Box.createHorizontalGlue());		
        return labelPanel;
    }
     

    private int getCommonCount(List currField, List prevField) {
        int smaller;
        int commonCount = 0;
        if (currField.size() < prevField.size())
            smaller = currField.size();
        else 
            smaller  = prevField.size();
        boolean common = true;
        while (common && commonCount < smaller) {
            if (currField.get(commonCount).equals(
                prevField.get(commonCount)))
                commonCount++;
            else
                common = false;
        }        
        return commonCount;
    }

    /**
     * OK, so the JTextFields are created on the fly and we need to access 
     * them. So we store it in a hashMap as we create them. 
     * <p>
     * When we have the field name we need to have access to the TextField 
     * that is associated with that fieldName.
     */
    
    public JComponent getField(String fieldName) {
        return (JComponent)nameToField.get(
            fieldName.toLowerCase(Locale.US));
    }

    private List textFields = new ArrayList();

    private final void addTextFieldToList(JTextField tf) {
        textFields.add(tf);
    }

    private final void addComboBoxToList(JComboBox cb) {
        textFields.add(cb);
    }

    /**
     * Clears the text fields.
     */
    public void clear() {
        Iterator iter = textFields.iterator();
        while (iter.hasNext()) {
            Object currObject = iter.next();
            clearField(currObject);
        }
    }

    public void clearField(Object field) {
        if (field instanceof JTextField)
            ((JTextField)field).setText(null);
        else if (field instanceof JComboBox)
            ((JComboBox)field).setSelectedIndex(0);
        else
            com.limegroup.gnutella.Assert.that(false, "Unknown Object!");  
    }
    
	/**
	 * Implement a Scrollable interface to properly scroll by some
	 * sane amount and not pixels.
	 */
	public Dimension getPreferredScrollableViewportSize() {
		return getPreferredSize();
	}
	
	public int getScrollableUnitIncrement(Rectangle visibleRect, int orientation,
		int direction) {
		switch (orientation) {
		case SwingConstants.HORIZONTAL:
			return visibleRect.width / 10;
		case SwingConstants.VERTICAL:
			return visibleRect.height / 10;
		default:
			throw new IllegalArgumentException("Unknown orientation " + orientation);
		}
	}
	
	public int getScrollableBlockIncrement( Rectangle visibleRect, int orientation,
		int direction) {
		switch (orientation) {
		case SwingConstants.HORIZONTAL:
			return visibleRect.width;
		case SwingConstants.VERTICAL:
			return visibleRect.height;
		default:
			throw new IllegalArgumentException("Unknown orientation " + orientation);
		}
	}
	
	public boolean getScrollableTracksViewportWidth() {
		return false;
	}

	public boolean getScrollableTracksViewportHeight() {
		return false;
	}
}
