package com.limegroup.gnutella.gui.xml;

import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.StringTokenizer;

import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JTextField;
import javax.swing.text.Document;

import com.limegroup.gnutella.gui.AutoCompleteTextField;
import com.limegroup.gnutella.util.I18NConvert;
import com.limegroup.gnutella.util.NameValue;
import com.limegroup.gnutella.xml.LimeXMLDocument;
import com.limegroup.gnutella.xml.LimeXMLSchema;
import com.limegroup.gnutella.xml.SchemaFieldInfo;

/**
 * A panel that is used to gather information from the user about
 * what the search criterion based on a particular schema.
 * <p>
 * This Panel is popped up everytime the user want to enter a query and has
 * specified the schema that she would like to base her search on
 *
 * @author Sumeet Thadani
 */
public class InputPanel extends IndentingPanel {
   
    private LimeXMLSchema _schema; // the schema that this Panel pertains to

    public InputPanel(LimeXMLSchema schema, ActionListener listener,
                      Document document) {
        this(schema, listener, document, false, false, true);
    }
    
    public InputPanel(LimeXMLSchema schema, ActionListener listener,
                      Document document, boolean expand, boolean indent, boolean search) {
        super(schema, listener, document, expand, indent, search);
        _schema = schema;
    }
        

    /**
     * @return The Schema URI associated with this InputPanel
     */
    public String getSchemaURI() {
        return _schema.getSchemaURI();
    }

    public String getInput() {
        return getInput(false);
    }

    /**
     * Looks at the textFields and creates a string that can be converted 
     * into LimeXMLDocument, so that the client that receives the search 
     * sting is can compare it with documents in its repository.
     * 
     * @param normalize true if the returned string should be normalized, thisis
     * the case when the user is doing a rich query. Otherwise if annotating,
     * metadata the string need not be normalized.
     */
    public String getInput(boolean normalize) {
        ArrayList namValList = new ArrayList();
        List list = _schema.getCanonicalizedFields();
        int t = list.size();
        String fields[] = new String[t];
        for (int i = 0; i < t; i++) {
            SchemaFieldInfo info = (SchemaFieldInfo)list.get(i);
            fields[i] = info.getCanonicalizedFieldName();
                
        }
        int size = fields.length;
        for (int i = 0; i < size; i++) {
            String currField = fields[i];
            JComponent comp = (JComponent)
                nameToField.get(currField.toLowerCase(Locale.US));
            String value = "";
            if (comp instanceof JTextField) {
                JTextField theField = (JTextField)comp;
                value = theField.getText();
            } else if (comp instanceof JComboBox) {
                JComboBox theBox = (JComboBox)comp;
                value = ((ComboBoxValue)theBox.getSelectedItem()).getValue();
            }
            if (value != null && !value.equals("")) {
                NameValue namValue = 
                    new NameValue(currField,
                                  normalize?
                                  I18NConvert.instance().getNorm(value): value);
                namValList.add(namValue);
            }
        }
        String schemaURI = _schema.getSchemaURI();
        String str = constructXML(namValList, schemaURI);
        return str;
    }

    /**
     * Scan through all the AutoTextField components
     * and store the input into their dictionaries.
     */
    public void storeInput() {
        List list = _schema.getCanonicalizedFields();
        int listLength = list.size();
        String fields[] = new String[listLength];
        for (int i = 0; i < listLength; i++) {
            SchemaFieldInfo info = (SchemaFieldInfo)list.get(i);
            fields[i] = info.getCanonicalizedFieldName();
        }
        int size = fields.length;
        for (int i = 0; i < size; i++) {
            JComponent comp = (JComponent)
                nameToField.get(fields[i].toLowerCase(Locale.US));
            if (comp instanceof AutoCompleteTextField) {
                AutoCompleteTextField theField = (AutoCompleteTextField)comp;
                if (!theField.getText().equals(""))
                    theField.addToDictionary();
            }
        }
    }
    
    /**
     * @return A string the represents a standard query (as opposed to a rich
     * query).
     * <p>
     * The order in which it checks for fields is schema specific.
     */
    public String getStandardQuery() {
        List list = _schema.getCanonicalizedFields();
        final int size = list.size();
        String fields[] = new String[size];
        for (int i = 0; i < size; i++) {
            SchemaFieldInfo info = (SchemaFieldInfo)list.get(i);
            fields[i] = info.getCanonicalizedFieldName();
        }

        StringBuffer retString = new StringBuffer();
        int numWords = 0;
        for (int i = 0; i < size && numWords < 3; i++) {
            String currField = fields[i];
            JComponent comp = (JComponent)
                nameToField.get(currField.toLowerCase(Locale.US));
            String value = "";
            if (comp instanceof JTextField) {
                JTextField theField = (JTextField)comp;
                value = theField.getText();
            } else if (comp instanceof JComboBox) {
                JComboBox theBox = (JComboBox)comp;
                value = ((ComboBoxValue)theBox.getSelectedItem()).toString();
            }
            if (value != null && value.trim().length() > 1) {
                StringTokenizer st = new StringTokenizer(value);
                numWords += st.countTokens();
                retString.append(value + " ");
            }
        }
        return retString.toString();
    }

    /**
     * Deligates to the the static method in LimeXMLDocument
     */
    public String constructXML(List namValList, String uri) {
        if(namValList == null || namValList.isEmpty())
            return null;
        else
            return new LimeXMLDocument(namValList, uri).getXMLString();
    }
}
