/*
 * check.c
 *
 * # echo 'noptrace-objs := check.o probe.o' > noptrace/Makefile
 * # echo 'obj-m += noptrace.o' >> noptrace/Makefile
 * # make -s SUBDIRS=$PWD/noptrace modules
 * # make -s SUBDIRS=$PWD/noptrace modules_install
 */
#include <linux/module.h>
#include <linux/init.h>
#include <linux/security.h>
#include <linux/version.h>
#include "probe.h"

#if LINUX_VERSION_CODE >= KERNEL_VERSION(3, 5, 0)
static inline _Bool my_check_task(void)
{
	return uid_gte(current_uid(), KUIDT_INIT(1000));
}
#else
#ifndef current_uid
#define current_uid()   (current->uid)
#endif
static inline _Bool my_check_task(void)
{
	return current_uid() >= 1000;
}
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 2, 0)

static int my_ptrace_access_check(struct task_struct *child, unsigned int mode)
{
	return my_check_task() ? -EPERM : 0;
}

static int my_ptrace_traceme(struct task_struct *parent)
{
	return my_check_task() ? -EPERM : 0;
}

#define MY_HOOK_INIT(HEAD, HOOK)				\
	{ .head = &probe_dummy_security_hook_heads.HEAD,	\
			.hook = { .HEAD = HOOK } }

static struct security_hook_list my_hooks[] = {
	MY_HOOK_INIT(ptrace_traceme, my_ptrace_traceme),
	MY_HOOK_INIT(ptrace_access_check, my_ptrace_access_check)
};

static inline void add_hook(struct security_hook_list *hook)
{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 17, 0)
	hlist_add_tail_rcu(&hook->list, hook->head);
#else
	list_add_tail_rcu(&hook->list, hook->head);
#endif
}

#if defined(CONFIG_STRICT_KERNEL_RWX) && !defined(CONFIG_SECURITY_WRITABLE_HOOKS)
#include <linux/uaccess.h> /* probe_kernel_write() */
#define NEED_TO_CHECK_HOOKS_ARE_WRITABLE

#if defined(CONFIG_X86)
#define MAX_RO_PAGES 1024
static struct page *ro_pages[MAX_RO_PAGES] __initdata;
static unsigned int ro_pages_len __initdata;

static bool __init lsm_test_page_ro(void *addr)
{
	unsigned int i;
	int unused;
	struct page *page;

	page = (struct page *) lookup_address((unsigned long) addr, &unused);
	if (!page)
		return false;
	if (test_bit(_PAGE_BIT_RW, &(page->flags)))
		return true;
	for (i = 0; i < ro_pages_len; i++)
		if (page == ro_pages[i])
			return true;
	if (ro_pages_len == MAX_RO_PAGES)
		return false;
	ro_pages[ro_pages_len++] = page;
	return true;
}

static bool __init check_ro_pages(struct security_hook_heads *hooks)
{
	int i;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 17, 0)
	struct hlist_head *list = &hooks->capable;

	if (!probe_kernel_write(list, list, sizeof(void *)))
		return true;
	for (i = 0; i < ARRAY_SIZE(my_hooks); i++) {
		struct hlist_head *head = my_hooks[i].head;
		struct security_hook_list *shp;

		if (!lsm_test_page_ro(&head->first))
			return false;
		hlist_for_each_entry(shp, head, list)
			if (!lsm_test_page_ro(&shp->list.next) ||
			    !lsm_test_page_ro(&shp->list.pprev))
				return false;
	}
#else
	struct list_head *list = &hooks->capable;

	if (!probe_kernel_write(list, list, sizeof(void *)))
		return true;
	for (i = 0; i < ARRAY_SIZE(my_hooks); i++) {
		struct list_head *head = my_hooks[i].head;
		struct security_hook_list *shp;

		if (!lsm_test_page_ro(&head->next) ||
		    !lsm_test_page_ro(&head->prev))
			return false;
		list_for_each_entry(shp, head, list)
			if (!lsm_test_page_ro(&shp->list.next) ||
			    !lsm_test_page_ro(&shp->list.prev))
				return false;
	}
#endif
	return true;
}
#else
static bool __init check_ro_pages(struct security_hook_heads *hooks)
{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4, 17, 0)
	struct hlist_head *list = &hooks->capable;
#else
	struct list_head *list = &hooks->capable;
#endif

	return !probe_kernel_write(list, list, sizeof(void *));
}
#endif
#endif

static int __init noptrace_init(void)
{
	int idx;
	struct security_hook_heads *hooks = probe_security_hook_heads();

	if (!hooks)
		return -EINVAL;
	for (idx = 0; idx < ARRAY_SIZE(my_hooks); idx++)
		my_hooks[idx].head = ((void *) hooks)
			+ ((unsigned long) my_hooks[idx].head)
			- ((unsigned long) &probe_dummy_security_hook_heads);
#if defined(NEED_TO_CHECK_HOOKS_ARE_WRITABLE)
	if (!check_ro_pages(hooks)) {
		printk(KERN_INFO "Can't update security_hook_heads due to write protected. Retry with rodata=0 kernel command line option added.\n");
		return -EINVAL;
	}
#endif
#if defined(NEED_TO_CHECK_HOOKS_ARE_WRITABLE) && defined(CONFIG_X86)
	for (idx = 0; idx < ro_pages_len; idx++)
		set_bit(_PAGE_BIT_RW, &(ro_pages[idx]->flags));
#endif
	for (idx = 0; idx < ARRAY_SIZE(my_hooks); idx++)
		add_hook(&my_hooks[idx]);
#if defined(NEED_TO_CHECK_HOOKS_ARE_WRITABLE) && defined(CONFIG_X86)
	for (idx = 0; idx < ro_pages_len; idx++)
		clear_bit(_PAGE_BIT_RW, &(ro_pages[idx]->flags));
#endif
	printk(KERN_INFO "ptrace restriction enabled.\n");
	return 0;
}

#else

/* Function pointers originally registered by register_security(). */
static struct security_operations original_security_ops /* = *security_ops; */;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 32)

static int my_ptrace_access_check(struct task_struct *child, unsigned int mode)
{
	if (my_check_task())
		return -EPERM;
	while (!original_security_ops.ptrace_access_check)
		smp_rmb();
	return original_security_ops.ptrace_access_check(child, mode);
}

#elif LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 27)

static int my_ptrace_may_access(struct task_struct *child, unsigned int mode)
{
	if (my_check_task())
		return -EPERM;
	while (!original_security_ops.ptrace_may_access)
		smp_rmb();
	return original_security_ops.ptrace_may_access(child, mode);
}

#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 27)

static int my_ptrace_traceme(struct task_struct *parent)
{
	if (my_check_task())
		return -EPERM;
	while (!original_security_ops.ptrace_traceme)
		smp_rmb();
	return original_security_ops.ptrace_traceme(parent);
}

#else

static inline int my_ptrace(struct task_struct *parent,
			    struct task_struct *child)
{
	if (my_check_task())
		return -EPERM;
	while (!original_security_ops.ptrace)
		smp_rmb();
	return original_security_ops.ptrace(parent, child);
}

#endif

static int my_file_open(struct file *f, const struct cred *cred)
{
	if (my_check_task()) {
		struct dentry *dentry = f->f_path.dentry;

		if (dentry && dentry->d_sb && dentry->d_sb->s_magic == PROC_SUPER_MAGIC &&
		    dentry->d_name.len == 3 && !memcmp(dentry->d_name.name, "mem", 3))
			return -EPERM;
	}
	while (!original_security_ops.file_open)
		smp_rmb();
	return original_security_ops.file_open(f, cred);
}

#define swap_security_ops(op)		    \
	original_security_ops.op = ops->op; \
	smp_wmb(); ops->op = my_##op

static int __init noptrace_init(void)
{
	struct security_operations *ops = probe_security_ops();

	if (!ops)
		return -EINVAL;
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 32)
	swap_security_ops(ptrace_traceme);
	swap_security_ops(ptrace_access_check);
#elif LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 27)
	swap_security_ops(ptrace_traceme);
	swap_security_ops(ptrace_may_access);
#else
	swap_security_ops(ptrace);
#endif
	swap_security_ops(file_open);
	printk(KERN_INFO "ptrace and /proc/*/mem restriction enabled.\n");
	return 0;
}

#undef swap_security_ops

#endif

module_init(noptrace_init);
MODULE_LICENSE("GPL");
