/* Copyright (C) 2003, 2010 TSUTSUMI Kikuo.
   This file is part of the CCUnit Library.

   The CCUnit Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   The CCUnit Library is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty
   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the CCUnit Library; see the file COPYING.LESSER.
   If not, write to the Free Software Foundation, Inc., 59 Temple
   Place - Suite 330, Boston, MA 02111-1307, USA.  
*/

/*
 * $Id: CCUnitTestSuite.c,v 1.11 2010/08/16 05:07:00 tsutsumi Exp $
 */
/** @file
 * TestSuite module implementation.
 */
#include <assert.h>
#include <ccunit/CCUnitTestSuite.h>
#include <ccunit/CCUnitTestResult.h>

/** @addtogroup CCUnitTestSuite
 * @{
 */

/*
 * Runs the test cases and collects their result in a TestResult.
 * @param testSuite test suite.
 * @param result result container.
 */
static void run (CCUnitTest* testSuite, CCUnitTestResult* result)
{
  CCUnitTestSuite* suite;
  CCUnitListIterator itor;
  CCUnitTest* test;
  assert (testSuite->type == ccunitTypeSuite);
  suite = (CCUnitTestSuite*)testSuite;
  ccunit_initListIterator (&suite->tests, &itor);
  while ((test = ccunit_nextListIterator (&itor)) != NULL)
    {
      if (result->shouldStop)
	break;
      test->run (test, result);
    }
}

/*
 * destruct test suite.
 * @param test destruct object.
 */
static void destroy (CCUnitTest* test)
{
  CCUnitTestSuite *suite;
  assert (test->type == ccunitTypeSuite);
  suite = (CCUnitTestSuite*)test;
  safe_free (suite->name);
  ccunit_deleteList (&suite->tests, (void (*)(void*))ccunit_deleteTest);
}

inline CCUnitTestSuite* ccunit_newTestSuite (const char* name)
{
  CCUnitTestSuite* suite = calloc (1, sizeof (*suite));
  ccunit_initTest (&suite->test, ccunitTypeSuite, run, destroy);
  suite->name = safe_strdup (name);
  ccunit_initList (&suite->tests);
  return suite;
}

CCUnitTestSuite* ccunit_newTestSuiteFromDfn (const CCUnitTestSuiteDfn* sdp)
{
  CCUnitTestSuite* suite;
  CCUnitTestDfn** tdpp;
  if (!sdp)
    return NULL;
  suite = ccunit_newTestSuite (sdp->name);
  if (!suite)
    return NULL;
  for (tdpp = (CCUnitTestDfn**)sdp->tests; *tdpp; tdpp ++)
    {
      const CCUnitTestDfn* tdp = *tdpp;
      if (tdp->type == ccunitTypeSuite)
	{
	  CCUnitTestSuite* sp;
	  sp = ccunit_newTestSuiteFromDfn ((CCUnitTestSuiteDfn*)tdp);
	  if (sp)
	    ccunit_addTestSuite (suite, sp);
	}
      else if (tdp->type == ccunitTypeFixture)
	{
	  CCUnitTestFixtureDfn* fdp = (CCUnitTestFixtureDfn*)tdp;
	  CCUnitTestFixture* fp;
	  CCUnitTestCase* cd;
	  fp = ccunit_newTestFixture (fdp->name,
				      ccunit_newTestFunc (fdp->setUp.name,
							  fdp->setUp.desc,
							  fdp->setUp.runTest),
				      ccunit_newTestFunc (fdp->tearDown.name,
							  fdp->tearDown.desc,
							  fdp->tearDown.runTest));
	  if (fp)
	    {
	      ccunit_addTestFixture (suite, fp);
	      for (cd = (CCUnitTestCase*)fdp->testCases; cd->name; cd ++)
		{
		  CCUnitTestCase* cp;
		  cp = ccunit_newTestCase (cd->name, cd->desc, cd->runTest);
		  if (cp)
		    ccunit_addTestCase (fp, cp);
		}
	    }
	}
    }
  return suite;
}

inline void ccunit_deleteTestSuite (CCUnitTestSuite* suite)
{
  ccunit_deleteTest (&suite->test);
}

inline void ccunit_addTest (CCUnitTestSuite* suite, CCUnitTest* test)
{
  ccunit_addList (&suite->tests, test);
}

inline void ccunit_addTestSuite (CCUnitTestSuite* suite,
				 CCUnitTestSuite* testSuite)
{
  ccunit_addTest (suite, &testSuite->test);
}

inline void ccunit_addTestFixture (CCUnitTestSuite* suite,
				   CCUnitTestFixture* fixture)
{
  ccunit_addTest (suite, &fixture->test);
}

inline CCUnitTestResult* ccunit_runTestSuite (CCUnitTestSuite* suite,
					      CCUnitTestResult* result)
{
  if (!result)
    {
      result = ccunit_newTestResult ();
      if (!result)
	return NULL;
    }
  suite->test.run (&suite->test, result);
  return result;
}

/** @} */
