/* -*- C -*- */
/* Copyright (C) 2003 TSUTSUMI Kikuo.
   This file is part of the CCUnit Library.

   The CCUnit Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   The CCUnit Library is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty
   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the CCUnit Library; see the file COPYING.LESSER.
   If not, write to the Free Software Foundation, Inc., 59 Temple
   Place - Suite 330, Boston, MA 02111-1307, USA.  
*/
/*
 * $Id: CCUnitMakeSuite.h,v 1.12 2003/10/24 20:05:22 tsutsumi Exp $
 */

/** @file
 * Make test suite object code from test source code.
 */
#ifndef CCUNITMAKESUITE_H
#define CCUNITMAKESUITE_H

#include <ccunit/CCUnitConfig.h>
#include <ccunit/CCUnitList.h>
#include <ccunit/CCUnitTest.h>

#include <stdio.h>

/**
 * @ingroup ModuleHierarchy
 * @defgroup CCUnitMakeSuite MakeSuite
 *
 * @~english
 * To generate a code to create test suite code from the test case
 * source codes, the test case source codes must be formatted by
 * following pseudo-BNF:
 * @~japanese
 * ƥȥɤƥȥȥɤ륳ɤ
 * 뤿ˡƥȥɤϲ BNF ˤäƥեޥ
 * ȤʤƤϤʤޤ
 * @~
 *
 * @code
 * SOURCEFILE ::= [SUITE]... | FIXTURE...
 *
 * SUITE ::= SUITE_HEADER
 *           [ any-C-code ]...
 *           FIXTURE...
 *           SUITE_END
 *
 * SUITE_HEADER ::= JAVADOC_COMMENT_BEGIN 'test suite:' SUITE_NAME COMMENT_END
 * SUITE_END ::= JAVADOC_COMMENT_BEGIN 'end test suite' wsp string COMMENT_END
 *
 * FIXTURE ::= FIXTURE_HEADER
 *             [FIXTURE_CODE]...
 *             [SETUP_FUNC]
 *             [TEARDOWN_FUNC]
 *             [TESTCASE]...
 *             FIXTURE_END
 *
 * FIXTURE_HEADER ::= JAVADOC_COMMENT_BEGIN
 *                    'test case:' TESTFIXTURE_NAME
 *                    COMMENT_END
 * FIXTURE_CODE ::= any C language codes.
 *
 * SETUP_FUNC ::= [ FUNC_DESC ] 'void setUp'[A-Za-z0-9_]* '()' FUNC_BODY
 *
 * TEARDOWN_FUNC ::= [ FUNC_DESC ] 'void tearDown'[A-Za-z0-9_]* '()' FUNC_BODY
 *
 * TESTCASE ::= [ FUNC_DESC ] 'void test'[A-Za-z0-9_]* '()' FUNC_BODY
 *
 * FUXTURE_END ::= JAVADOC_COMMENT_BEGIN 'end test case' wsp string COMMENT_END
 *
 * FUNC_DESC ::= JAVADOC_COMMENT_BEGIN string COMMENT_END
 *
 * FUNC_BODY ::= '{' C language codes... '}'
 *
 * JAVADOC_COMMENT_BEGIN ::= '/' '*' '*'
 * COMMENT_END ::= '*' '/'
 * @endcode
 *
 * @{
 */

/**
 * @defgroup CCUnitTestDef _TestDef
 * Test definition
 * @{
 */

/**
 * Test definition information.
 */
typedef struct _CCUnitTestDef
{
  CCUnitTestType_t type;			/**< test type */
  char* name;					/**< test name */
  char* idname;					/**< id name */
  void (*dtor) (struct _CCUnitTestDef*);	/**< test destcuctor  */
} _CCUnitTestDef;

/** @} */

/**
 * @defgroup CCUnitTestSuiteDef _TestSuiteDef
 * Test suite definition
 * @{
 */

/**
 * Test suite definition information.
 */
typedef struct _CCUnitTestSuiteDef
{
  _CCUnitTestDef testdef;			/**< super class */
  CCUnitList testdefs;				/**< test defs list */
} _CCUnitTestSuiteDef;

/**
 * create new test suite def.
 *
 * @param name suite name.
 * @return created testdef.
 */
extern _CCUnitTestSuiteDef* ccunit_newTestSuiteDef (const char* name);

/**
 * delete test suite def.
 *
 * @param suite test suite def.
 */
extern inline void ccunit_deleteTestSuiteDef (_CCUnitTestSuiteDef* suite);

/** @} */

/**
 * @defgroup CCUnitTestCaseDef _TestCaseDef
 * Test case definition.
 * @{
 */

/**
 * Test case definition information.
 */
typedef struct _CCUnitFuncDef
{
  char* scope;					/**< scope */
  char* type;					/**< return type */
  char* name;					/**< function name */
  char* desc;					/**< description */
} _CCUnitFuncDef;

/** @} */

/**
 * @defgroup CCUnitTestFixtureDef _TestFixtureDef
 * Test fixture definition.
 * @{
 */

/**
 * Test fixture definition information.
 */
typedef struct _CCUnitTestFixtureDef
{
  _CCUnitTestDef testdef;			/**< super class */
  _CCUnitFuncDef* ctor;				/**< constructor */
  _CCUnitFuncDef* dtor;				/**< destructor */
  _CCUnitFuncDef* setUp;			/**< test setup function */
  _CCUnitFuncDef* tearDown;			/**< test tearDown function */
  CCUnitList testCases;				/**< test case list */
} _CCUnitTestFixtureDef;

/** @} */

/** @ingroup CCUnitMakeSuite
 * @defgroup CCUnitReadSuite _ReadSuite
 * Read test case definitions from test source code.
 * @{
 */

/**
 * read test unit suite from specified stream.
 *
 * @param fname test unit source code file.
 * @param parent [out] test suite to adding read test.
 */
extern void ccunit_readSuite (const char* fname, _CCUnitTestSuiteDef* parent);

/** @} */

/**
 * @defgroup CCUnitWriteSuite _WriteSuite
 * Write the source code to making the test suite.
 * @{
 */

/**
 * output test suite making function code.
 *
 * @param ofp output stream.
 * @param name add function name.
 * @param suite test suite to print.
 */
extern void ccunit_writeSuite (FILE* ofp,
			       const char* name,
			       _CCUnitTestSuiteDef* suite);

/**
 * output test suite making function code.
 *
 * @param ofp output stream.
 * @param name add function name.
 * @param suite test suite to print.
 */
extern void ccunit_printSuite (FILE* ofp,
			       const char* name,
			       _CCUnitTestSuiteDef* suite);

/** @} */

/**
 * @english
 * Generate a code to creating test suite code from the test case
 * source codes.
 * @japanese
 * ƥȥΥɤ顢ƥȥĤ륳ɤ롣
 * @endif
 *
 * @param ac arg count.
 * @param av array of arg string.
 *
 * @code
 * av[] = { PRORGNAME, [OPTIONS]..., FILES... }
 * OPTIONS:
 *      --output OUTFILE
 *      -o OUTFILE
 *          output file name (default stdout)
 *      --function FUNCTION
 *      -f FUNCTION
 *          creating function name
 *      --verbose
 *      -v
 *          output verbose message
 *      --debug
 *      -d
 *          output debug message
 *      --version
 *      -V
 *          print version
 *      --help
 *      -h
 *          print this message
 * FILES: the special name '-' means input from stdin.
 * @endcode
 *
 * @~english
 * To generate a code to create test suite code from the test case
 * source codes, the test case source codes must be formatted by
 * following pseudo-BNF:
 * @~japanese
 * ƥȥɤƥȥȥɤ륳ɤ
 * 뤿ˡƥȥɤϲ BNF ˤäƥեޥ
 * ȤʤƤϤʤޤ
 * @~
 * @copydoc CCUnitMakeSuite
 *
 * @return 0 if succeeded, else error occured.
 * @ingroup CreatingTestSuite
 */
extern int ccunit_makeSuite (int ac, char** av);

/**
 * Generate a code to creating test suite code from the test case
 * source codes. 
 * @param prg program name
 * @param ... arguments. The last arg must be NULL.
 * @return 0 if succeeded, else error occured.
 * @see ccunit_makeSuite
 * @ingroup CreatingTestSuite
 */
extern int ccunit_va_makeSuite (const char* prg, ...);

/** @} */

#endif
