/* Copyright (C) 2003, 2010 TSUTSUMI Kikuo.
   This file is part of the CCUnit Library.

   The CCUnit Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   The CCUnit Library is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty
   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the CCUnit Library; see the file COPYING.LESSER.
   If not, write to the Free Software Foundation, Inc., 59 Temple
   Place - Suite 330, Boston, MA 02111-1307, USA.  
*/

/*
 * $Id: CCUnitTestCase.c,v 1.12 2010/08/21 04:12:01 tsutsumi Exp $
 */

/**
 * @file
 * TestCase module implementation.
 */

#include <ccunit/CCUnitTestCase.h>
#include <ccunit/CCUnitTestResult.h>
#include <assert.h>
#include <setjmp.h>

/**
 * run tests exception environment.
 */
jmp_buf _ccunit_runTest_env;

/**
 * run tests exception.
 */
CCUnitTestFailure* _ccunit_testFailure;

extern void _ccunit_startTest (CCUnitTestResult* result,
			       CCUnitTestFunc* test);
extern void _ccunit_endTest (CCUnitTestResult* result,
			     CCUnitTestFunc* test);

void ccunit_addTestFunc (CCUnitTestCase* testCase, CCUnitTestFunc* f)
{
  static const char* ssu = "setup_setUp";
  static const char* std = "setup_tearDown";
  static const char* su = "setUp";
  static const char* td = "tearDown";
  static const char* ts = "test";
  if (strncmp (f->name, ts, strlen (ts)) == 0)
    ccunit_addList (&testCase->testFuncs, f);
  else if (strncmp (f->name, ssu, strlen (ssu)) == 0)
    {
      if (testCase->setup_setUp)
	ccunit_deleteTestFunc (testCase->setup_setUp);
      testCase->setup_setUp = f;
    }
  else if (strncmp (f->name, std, strlen (std)) == 0)
    {
      if (testCase->setup_tearDown)
	ccunit_deleteTestFunc (testCase->setup_tearDown);
      testCase->setup_tearDown = f;
    }
  else if (strncmp (f->name, su, strlen (su)) == 0)
    {
      if (testCase->setUp)
	ccunit_deleteTestFunc (testCase->setUp);
      testCase->setUp = f;
    }
  else if (strncmp (f->name, td, strlen (td)) == 0)
    {
      if (testCase->tearDown)
	ccunit_deleteTestFunc (testCase->tearDown);
      testCase->tearDown = f;
    }
  else
    ccunit_addList (&testCase->testFuncs, f);
}

CCUnitTestFunc* ccunit_addNewTestFunc (CCUnitTestCase* testCase,
				       const char* name,
				       const char* desc,
				       void (*runTest)())
{
  CCUnitTestFunc* f;
  if (!testCase)
    return NULL;
  f = ccunit_newTestFunc (name, desc, runTest);
  if (!f)
    return NULL;
  ccunit_addTestFunc (testCase, f);
  return f;
}

/**
 * Runs the bare test sequence.
 *
 * @return failure
 */
static CCUnitTestFailure* runTest (CCUnitTestFunc* testFunc,
				   CCUnitTestFunc* setUp,
				   CCUnitTestFunc* tearDown,
				   CCUnitTestResult* result)
{
  CCUnitTestFailure* runFailure = NULL;
  _ccunit_testFailure = NULL;
  if (setUp && setUp->runTest)
    {
      if (setjmp (_ccunit_runTest_env) == 0)
	setUp->runTest ();
      else
	{
	  _ccunit_testFailure->testFunc = setUp;
	  ccunit_addFailure (result, _ccunit_testFailure);
	}
    }
  if (_ccunit_testFailure)
    ;
  else if (setjmp (_ccunit_runTest_env) == 0)
    {
      result->runCount ++;
      testFunc->runTest ();
    }
  else
    {
      runFailure = _ccunit_testFailure;
      runFailure->testFunc = testFunc;
      _ccunit_testFailure = NULL;
    }
  if (tearDown && tearDown->runTest)
    {
      if (setjmp (_ccunit_runTest_env) == 0)
	tearDown->runTest ();
      else
	{
	  _ccunit_testFailure->testFunc = tearDown;
	  ccunit_addFailure (result, _ccunit_testFailure);
	}
    }
  return runFailure;
}

/**
 * Runs the test case and collects the results in CCUnitTestResult.
 * @param test A test to run.
 * @param result A result container.
 */
static void run (CCUnitTest* test, CCUnitTestResult* result)
{
  CCUnitTestCase* testCase = (CCUnitTestCase*)test;
  CCUnitListIterator itor;
  CCUnitTestFunc* testFunc;
  assert (test->type == ccunitTypeTestCase);
  _ccunit_testFailure = NULL;
  if (testCase->setup_setUp && testCase->setup_setUp->runTest)
    {
      if (setjmp (_ccunit_runTest_env) == 0)
	testCase->setup_setUp->runTest ();
      else
	{
	  _ccunit_testFailure->testFunc = testCase->setup_setUp;
	  ccunit_addFailure (result, _ccunit_testFailure);
	}
    }
  if (!_ccunit_testFailure)
    {
      ccunit_initListIterator (&testCase->testFuncs, &itor);
      while ((testFunc = ccunit_nextListIterator (&itor)) != NULL)
	{
	  CCUnitTestFailure* failure;
	  _ccunit_startTest (result, testFunc);
	  failure = runTest (testFunc, testCase->setUp, testCase->tearDown, result);
	  if (failure)
	    ccunit_addFailure (result, failure);
	  _ccunit_endTest (result, testFunc);
	}
    }
  if (testCase->setup_tearDown && testCase->setup_tearDown->runTest)
    {
      if (setjmp (_ccunit_runTest_env) == 0)
	testCase->setup_tearDown->runTest ();
      else
	{
	  _ccunit_testFailure->testFunc = testCase->setup_tearDown;
	  ccunit_addFailure (result, _ccunit_testFailure);
	}
    }
}

/**
 * Destruct test case.
 * @param test destruct test.
 */
static void destroy (CCUnitTest* test)
{
  CCUnitTestCase* testCase;
  assert (test->type == ccunitTypeTestCase);
  testCase = (CCUnitTestCase*)test;
  safe_free (testCase->name);
  ccunit_deleteTestFunc (testCase->setup_setUp);
  ccunit_deleteTestFunc (testCase->setup_tearDown);
  ccunit_deleteTestFunc (testCase->setUp);
  ccunit_deleteTestFunc (testCase->tearDown);
  ccunit_deleteList (&testCase->testFuncs,
		     (void (*)(void*))ccunit_deleteTestFunc);
}

CCUnitTestCase* ccunit_newTestCase (const char* name)
{
  CCUnitTestCase* testCase = calloc (1, sizeof (*testCase));
  ccunit_initTest (&testCase->test, ccunitTypeTestCase, run, destroy);
  testCase->name = safe_strdup (name);
  ccunit_initList (&testCase->testFuncs);
  return testCase;
}

inline void ccunit_deleteTestCase (CCUnitTestCase* testCase)
{
  ccunit_deleteTest (&testCase->test);
}

inline struct CCUnitTestResult* ccunit_runTestCase (CCUnitTestCase* testCase)
{
  CCUnitTestResult* result;
  if (!testCase)
    return NULL;
  result = ccunit_newTestResult ();
  if (!result)
    return NULL;
  testCase->test.run (&testCase->test, result);
  return result;
}
