/* -*- C -*- */
/* Copyright (C) 2003 TSUTSUMI Kikuo.
   This file is part of the CCUnit Library.

   The CCUnit Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   The CCUnit Library is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty
   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the CCUnit Library; see the file COPYING.LESSER.
   If not, write to the Free Software Foundation, Inc., 59 Temple
   Place - Suite 330, Boston, MA 02111-1307, USA.  
*/

/*
 * $Id: CCUnitTestSuite.h,v 1.10 2003/10/05 11:29:59 tsutsumi Exp $
 */
/** @file
 * TestSuite module.
 */
#ifndef CCUNITTESTSUITE_H
#define CCUNITTESTSUITE_H

#include <ccunit/CCUnitConfig.h>
#include <ccunit/CCUnitList.h>
#include <ccunit/CCUnitTest.h>
#include <ccunit/CCUnitTestFixture.h>
#include <ccunit/CCUnitTestResult.h>

/**
 * @ingroup CCUnitTest
 * @defgroup CCUnitTestSuite TestSuite
 *
 * A <code>Composite</code> class of Tests.
 * It runs a collection of test cases.
 * @{
 */

/**
 * A <code>Composite</code> class of Tests.
 * It runs a collection of test cases. Here is an example.
 *
 * @code
 * CCUnitTestSuite* suite = ccunit_newTestSuite ();
 * ccunit_addTestCase (suite, TESTCASE_1);
 * ccunit_addTestCase (suite, TESTCASE_2);
 * @endcode
 *
 * @see CCUnitTest, CCUnitTestCase.
 * @ingroup CreatingTestSuite
 */
typedef struct CCUnitTestSuite
{
  CCUnitTest test;		    /**< super class */
  const char* name;		    /**< test suite name */
  CCUnitList tests;		    /**< added test case/suite list */
} CCUnitTestSuite;

typedef struct CCUnitTestSuiteDfn
{
  CCUnitTestDfn test;		    /**< super class */
  const char* name;		    /**< test suite name */
  CCUnitTestDfn** tests;	    /**< added test case/suite list */
} CCUnitTestSuiteDfn;

/**
 * Constructs an empty TestSuite.
 * @param name test suite name.
 * @return new test suite.
 * @ingroup CreatingTestSuite
 */
extern inline CCUnitTestSuite* ccunit_newTestSuite(const char* name);

/**
 * Constructs a TestSuite from definition struct.
 * @param sdp test suite definition.
 * @return new test suite.
 */
extern CCUnitTestSuite* ccunit_newTestSuiteFromDfn (const CCUnitTestSuiteDfn* sdp);

/**
 * Destructs test suite.
 * @param suite deleting suite.
 * @ingroup CreatingTestSuite
 */
extern inline void ccunit_deleteTestSuite (CCUnitTestSuite* suite);

/**
 * Adds a test to the suite.
 * @param suite test suite.
 * @param test test to add.
 * @ingroup CreatingTestSuite
 */
extern inline void ccunit_addTest (CCUnitTestSuite* suite, CCUnitTest* test);

/**
 * Adds a test suite to the suite.
 * @param suite test suite.
 * @param testSuite test to add.
 * @ingroup CreatingTestSuite
 */
extern inline void ccunit_addTestSuite (CCUnitTestSuite* suite,
					CCUnitTestSuite* testSuite);

/**
 * Adds a test fixture to the suite.
 * @param suite test suite.
 * @param fixture test to add.
 * @ingroup CreatingTestSuite
 */
extern inline void ccunit_addTestFixture (CCUnitTestSuite* suite,
					  CCUnitTestFixture* fixture);

/**
 * run test suite and collect its results.
 * @param suite test suite.
 * @param result test result. if NULL, create a new result object and return it.
 * @return test result.
 * @ingroup ExecutingTest
 */
extern inline CCUnitTestResult* ccunit_runTestSuite (CCUnitTestSuite* suite,
						     CCUnitTestResult* result);

/**
 * Create a test suite from test source file.
 * @param name test suite name.
 * @return new test suite.
 * @ingroup CreatingTestSuite
 */
extern CCUnitTestSuite* ccunit_suite (const char* name);

/** @} */
#endif
