#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <omp.h>

#include <gsl/gsl_rng.h>
#include <cdescent.h>

const int		size = 500;
const double	thres = 0.5;
const double	eps = 1.e-5;
gsl_rng			*rng;

static mm_sparse *
mm_real_create_sparse_symm (MMRealSymm symm, const int n)
{
	int			i, j, k;
	int			nnz = n * n;
	mm_sparse	*s = mm_real_new (MM_REAL_SPARSE, symm, n, n, nnz); 

	k = 0;
	for (j = 0; j < n; j++) {
		int	i0 = (mm_real_is_upper (s)) ? 0 : j;
		int	i1 = (mm_real_is_upper (s)) ? j : n - 1;
		for (i = i0; i <= i1; i++) {
			double	r = gsl_rng_uniform (rng);
			if (r < thres) continue;
			s->i[k] = i;
			s->data[k++] = 1. - 2. * gsl_rng_uniform (rng);
		}
		s->p[j + 1] = k;
	}
	if (nnz != k) mm_real_realloc (s, k);
	return s;
}

static mm_dense *
mm_real_create_dense_symm (MMRealSymm symm, const int n)
{
	int			i, j;
	int			nnz = n * n;
	mm_dense	*d = mm_real_new (MM_REAL_DENSE, symm, n, n, nnz); 

	for (j = 0; j < n; j++) {
		int	i0 = (mm_real_is_upper (d)) ? 0 : j;
		int	i1 = (mm_real_is_upper (d)) ? j : n - 1;
		for (i = i0; i <= i1; i++) d->data[i + j * d->m] = 1. - 2. * gsl_rng_uniform (rng);
	}
	return d;
}

static mm_sparse *
mm_real_create_sparse (const int m, const int n)
{
	int			i, j, k;
	int			nnz = m * n;
	mm_sparse	*s = mm_real_new (MM_REAL_SPARSE, MM_REAL_GENERAL, m, n, nnz);

	k = 0;
	for (j = 0; j < n; j++) {
		for (i = 0; i < m; i++) {
			double	r = gsl_rng_uniform (rng);
			if (r < thres) continue;
			s->i[k] = i;
			s->data[k++] = 1. - 2. * gsl_rng_uniform (rng);
		}
		s->p[j + 1] = k;
	}
	if (nnz != k) mm_real_realloc (s, k);
	return s;
}

static mm_dense *
mm_real_create_dense (const int m, const int n)
{
	int			i, j;
	int			nnz = m * n;
	mm_dense	*d = mm_real_new (MM_REAL_DENSE, MM_REAL_GENERAL, m, n, nnz); 

	for (j = 0; j < n; j++) {
		for (i = 0; i < m; i++) d->data[i + j * d->m] = 1. - 2. * gsl_rng_uniform (rng);
	}
	return d;
}

/* symmetric -> general */
static mm_dense *
mm_real_symm_to_general_dense (const mm_dense *x)
{
	int			i, j;
	mm_dense	*d = mm_real_copy (x);
	for (j = 0; j < x->n; j++) {
		int	i0 = (mm_real_is_upper (d)) ? 0 : j;
		int	i1 = (mm_real_is_upper (d)) ? j : x->n;
		for (i = i0; i < i1; i++) d->data[j + i * d->m] = d->data[i + j * d->m];
	}
	return d;
}

static mm_dense *
mm_real_symm_to_general_sparse (const mm_sparse *x)
{
	int			i, j;
	mm_dense	*d = mm_real_sparse_to_dense (x);
	if (mm_real_is_symmetric (d)) {
		mm_dense	*tmp = mm_real_symm_to_general_dense (d);
		mm_real_free (d);
		d = mm_real_copy (tmp);
		mm_real_free (tmp);
	}
	return d;
}

static void
mm_real_fprintf_dense_0 (FILE *stream, const mm_dense *d, const char *format)
{
	int		i, j, k = 0;
	for (i = 0; i < d->m; i++) {
		for (j = 0; j < d->n; j++) {
			fprintf (stream, format, d->data[k++]);
			if (j <= d->n - 1) fprintf (stream, " ");
		}
		fprintf (stream, "\n");
	}
	return;
}

static void
mm_real_fprintf_sparse (FILE *stream, const mm_sparse *x, const char *format)
{
	mm_dense	*d;
	if (mm_real_is_symmetric (x)) d = mm_real_symm_to_general_sparse (x);
	else d = mm_real_sparse_to_dense (x);
	mm_real_fprintf_dense_0 (stream, d, format);
	mm_real_free (d);
	return;
}

static void
mm_real_fprintf_dense (FILE *stream, const mm_sparse *x, const char *format)
{
	mm_dense	*d;
	if (mm_real_is_symmetric (x)) d = mm_real_symm_to_general_dense (x);
	else d = mm_real_copy (x);
	mm_real_fprintf_dense_0 (stream, d, format);
	mm_real_free (d);
	return;
}

void
mm_real_fprintf (FILE *stream, const mm_real *x, const char *format)
{
	(mm_real_is_sparse (x)) ? mm_real_fprintf_sparse (stream, x, format) : mm_real_fprintf_dense (stream, x, format);
	return;
}

/* test_xj_sum_ssq_nrm2 */
bool
test_sj_sum_ssq_nrm2 (FILE *stream, const long seed)
{
	int			j;
	bool		symm;

	mm_sparse	*s;
	mm_dense	*x;
	double		z0, z1;

	bool		result;

	gsl_rng_set (rng, seed);

	symm = (gsl_rng_uniform (rng) <= 0.5) ? false : true;

	if (symm) {
		int		n = gsl_rng_uniform_int (rng, size) + 100;
		bool	upper = (gsl_rng_uniform (rng) <= 0.5) ? false : true;
		MMRealSymm	symm = (upper) ? MM_REAL_SYMMETRIC_UPPER : MM_REAL_SYMMETRIC_LOWER;
		s = mm_real_create_sparse_symm (symm, n);
	} else {
		int		m = gsl_rng_uniform_int (rng, size) + 100;
		int		n = gsl_rng_uniform_int (rng, size) + 100;
		s = mm_real_create_sparse (m, n);
	}

	z0 = 0.;
	for (j = 0; j < s->n; j++) {
		z0 += mm_real_xj_asum (s, j);
		z0 += mm_real_xj_sum (s, j);
		z0 += mm_real_xj_ssq (s, j);
		z0 += mm_real_xj_nrm2 (s, j);
	}

	if (symm) x = mm_real_symm_to_general_sparse (s);
	else x = mm_real_sparse_to_dense (s);
	mm_real_free (s);

	z1 = 0.;
	for (j = 0; j < x->n; j++) {
		z1 += mm_real_xj_asum (x, j);
		z1 += mm_real_xj_sum (x, j);
		z1 += mm_real_xj_ssq (x, j);
		z1 += mm_real_xj_nrm2 (x, j);
	}
	mm_real_free (x);

	result = (fabs (z1 - z0) < eps);

	return result;
}

bool
test_dj_sum_ssq_nrm2 (FILE *stream, const long seed)
{
	int			j;
	bool		symm;

	mm_dense	*d;
	mm_dense	*x;
	double		z0, z1;

	bool		result;

	gsl_rng_set (rng, seed);

	symm = (gsl_rng_uniform (rng) <= 0.5) ? false : true;

	if (symm) {
		int		n = gsl_rng_uniform_int (rng, size) + 100;
		bool	upper = (gsl_rng_uniform (rng) <= 0.5) ? false : true;
		MMRealSymm	symm = (upper) ? MM_REAL_SYMMETRIC_UPPER : MM_REAL_SYMMETRIC_LOWER;
		d = mm_real_create_dense_symm (symm, n);
	} else {
		int		m = gsl_rng_uniform_int (rng, size) + 100;
		int		n = gsl_rng_uniform_int (rng, size) + 100;
		d = mm_real_create_sparse (m, n);
	}

	z0 = 0.;
	for (j = 0; j < d->n; j++) {
		z0 += mm_real_xj_asum (d, j);
		z0 += mm_real_xj_sum (d, j);
		z0 += mm_real_xj_ssq (d, j);
		z0 += mm_real_xj_nrm2 (d, j);
	}

	if (symm) x = mm_real_symm_to_general_dense (d);
	else x = mm_real_copy (d);
	mm_real_free (d);

	z1 = 0.;
	for (j = 0; j < x->n; j++) {
		z1 += mm_real_xj_asum (x, j);
		z1 += mm_real_xj_sum (x, j);
		z1 += mm_real_xj_ssq (x, j);
		z1 += mm_real_xj_nrm2 (x, j);
	}
	mm_real_free (x);

	result = (fabs (z1 - z0) < eps);

	return result;
}

/* test_xj_trans_dot_y */
bool
test_sj_trans_dot_y (FILE *stream, const long seed)
{
	int			j;
	bool		symm;

	mm_sparse	*s;
	mm_dense	*x;
	mm_dense	*y;
	double		z0, z1;

	bool		result;

	gsl_rng_set (rng, seed);

	symm = (gsl_rng_uniform (rng) <= 0.5) ? false : true;

	if (symm) {
		int		n = gsl_rng_uniform_int (rng, size) + 100;
		bool	upper = (gsl_rng_uniform (rng) <= 0.5) ? false : true;
		MMRealSymm	symm = (upper) ? MM_REAL_SYMMETRIC_UPPER : MM_REAL_SYMMETRIC_LOWER;
		s = mm_real_create_sparse_symm (symm, n);
		y = mm_real_create_dense (n, 1);
	} else {
		int		m = gsl_rng_uniform_int (rng, size) + 100;
		int		n = gsl_rng_uniform_int (rng, size) + 100;
		s = mm_real_create_sparse (m, n);
		y = mm_real_create_dense (m, 1);
	}

	z0 = 0.;
	for (j = 0; j < s->n; j++) z0 += mm_real_xj_trans_dot_y (s, j, y);

	if (symm) x = mm_real_symm_to_general_sparse (s);
	else x = mm_real_sparse_to_dense (s);
	mm_real_free (s);

	z1 = 0.;
	for (j = 0; j < x->n; j++) z1 += mm_real_xj_trans_dot_y (x, j, y);
	mm_real_free (x);
	mm_real_free (y);

	result = (fabs (z1 - z0) < eps);

	return result;
}

bool
test_dj_trans_dot_y (FILE *stream, const long seed)
{
	int			j;
	bool		symm;

	mm_dense	*d;
	mm_dense	*x;
	mm_dense	*y;
	double		z0, z1;

	bool		result;

	gsl_rng_set (rng, seed);

	symm = (gsl_rng_uniform (rng) <= 0.5) ? false : true;

	if (symm) {
		int		n = gsl_rng_uniform_int (rng, size) + 100;
		bool	upper = (gsl_rng_uniform (rng) <= 0.5) ? false : true;
		MMRealSymm	symm = (upper) ? MM_REAL_SYMMETRIC_UPPER : MM_REAL_SYMMETRIC_LOWER;
		d = mm_real_create_dense_symm (symm, n);
		y = mm_real_create_dense (n, 1);
	} else {
		int		m = gsl_rng_uniform_int (rng, size) + 100;
		int		n = gsl_rng_uniform_int (rng, size) + 100;
		d = mm_real_create_dense (m, n);
		y = mm_real_create_dense (m, 1);
	}

	z0 = 0.;
	for (j = 0; j < d->n; j++) z0 += mm_real_xj_trans_dot_y (d, j, y);

	if (symm) x = mm_real_symm_to_general_dense (d);
	else x = mm_real_copy (d);
	mm_real_free (d);

	z1 = 0.;
	for (j = 0; j < x->n; j++) z1 += mm_real_xj_trans_dot_y (x, j, y);
	mm_real_free (x);
	mm_real_free (y);

	result = (fabs (z1 - z0) < eps);

	return result;
}

/* test_x_dot_y */
bool
test_s_dot_y (FILE *stream, const long seed)
{
	bool		symm;
	bool		trans;

	mm_sparse	*s;
	mm_dense	*x;
	mm_dense	*y;
	mm_dense	*z0, *z1;

	double		alpha, beta;

	bool		result;

	gsl_rng_set (rng, seed);

	alpha = 1. - 2. * gsl_rng_uniform (rng);
	beta = 1. - 2. * gsl_rng_uniform (rng);

	symm = (gsl_rng_uniform (rng) <= 0.5) ? false : true;
	trans = (gsl_rng_uniform (rng) <= 0.5) ? false : true;

	if (symm) {
		int		n = gsl_rng_uniform_int (rng, size) + 100;
		bool	upper = (gsl_rng_uniform (rng) <= 0.5) ? false : true;
		MMRealSymm	symm = (upper) ? MM_REAL_SYMMETRIC_UPPER : MM_REAL_SYMMETRIC_LOWER;
		s = mm_real_create_sparse_symm (symm, n);
		y = mm_real_create_dense (n, 1);
		z0 = mm_real_new (MM_REAL_DENSE, MM_REAL_GENERAL, n, 1, n);
		z1 = mm_real_new (MM_REAL_DENSE, MM_REAL_GENERAL, n, 1, n);
	} else {
		int		k, l;
		int		m = gsl_rng_uniform_int (rng, size) + 100;
		int		n = gsl_rng_uniform_int (rng, size) + 100;
		k = (trans) ? m : n;
		l = (trans) ? n : m;
		s = mm_real_create_sparse (m, n);
		y = mm_real_create_dense (k, 1);
		z0 = mm_real_new (MM_REAL_DENSE, MM_REAL_GENERAL, l, 1, l);
		z1 = mm_real_new (MM_REAL_DENSE, MM_REAL_GENERAL, l, 1, l);
	}

	mm_real_set_all (z0, 1.);
	mm_real_x_dot_y (trans, alpha, s, y, beta, z0);

	if (symm) x = mm_real_symm_to_general_sparse (s);
	else x = mm_real_sparse_to_dense (s);
	mm_real_free (s);

	mm_real_set_all (z1, 1.);
	mm_real_x_dot_y (trans, alpha, x, y, beta, z1);
	mm_real_free (x);
	mm_real_free (y);

	mm_real_axjpy (-1., z1, 0, z0);
	mm_real_free (z1);
	result = (mm_real_xj_nrm2 (z0, 0) < eps);
	mm_real_free (z0);

	return result;
}

bool
test_d_dot_y (FILE *stream, const long seed)
{
	bool		symm;
	bool		trans;

	mm_dense	*d;
	mm_dense	*x;
	mm_dense	*y;
	mm_dense	*z0, *z1;

	double		alpha, beta;

	bool		result;

	gsl_rng_set (rng, seed);

	alpha = 1. - 2. * gsl_rng_uniform (rng);
	beta = 1. - 2. * gsl_rng_uniform (rng);

	symm = (gsl_rng_uniform (rng) <= 0.5) ? false : true;
	trans = (gsl_rng_uniform (rng) <= 0.5) ? false : true;

	if (symm) {
		int		n = gsl_rng_uniform_int (rng, size) + 100;
		bool	upper = (gsl_rng_uniform (rng) <= 0.5) ? false : true;
		MMRealSymm	symm = (upper) ? MM_REAL_SYMMETRIC_UPPER : MM_REAL_SYMMETRIC_LOWER;
		d = mm_real_create_dense_symm (symm, n);
		y = mm_real_create_dense (n, 1);
		z0 = mm_real_new (MM_REAL_DENSE, MM_REAL_GENERAL, n, 1, n);
		z1 = mm_real_new (MM_REAL_DENSE, MM_REAL_GENERAL, n, 1, n);
	} else {
		int		k, l;
		int		m = gsl_rng_uniform_int (rng, size) + 100;
		int		n = gsl_rng_uniform_int (rng, size) + 100;
		k = (trans) ? m : n;
		l = (trans) ? n : m;
		d = mm_real_create_dense (m, n);
		y = mm_real_create_dense (k, 1);
		z0 = mm_real_new (MM_REAL_DENSE, MM_REAL_GENERAL, l, 1, l);
		z1 = mm_real_new (MM_REAL_DENSE, MM_REAL_GENERAL, l, 1, l);
	}

	mm_real_set_all (z0, 1.);
	mm_real_x_dot_y (trans, alpha, d, y, beta, z0);

	if (symm) x = mm_real_symm_to_general_dense (d);
	else x = mm_real_copy (d);
	mm_real_free (d);

	mm_real_set_all (z1, 1.);
	mm_real_x_dot_y (trans, alpha, x, y, beta, z1);
	mm_real_free (x);
	mm_real_free (y);

	mm_real_axjpy (-1., z1, 0, z0);
	mm_real_free (z1);
	result = (mm_real_xj_nrm2 (z0, 0) < eps);
	mm_real_free (z0);

	return result;
}

/* test_axjpy */
bool
test_asjpy (FILE *stream, const long seed)
{
	int			j;
	bool		symm;

	mm_sparse	*s;
	mm_dense	*x;
	mm_dense	*y;
	mm_dense	*z0, *z1;

	double		alpha;

	bool		result;

	gsl_rng_set (rng, seed);

	alpha = 1. - 2. * gsl_rng_uniform (rng);

	symm = (gsl_rng_uniform (rng) <= 0.5) ? false : true;

	if (symm) {
		int		n = gsl_rng_uniform_int (rng, size) + 100;
		bool	upper = (gsl_rng_uniform (rng) <= 0.5) ? false : true;
		MMRealSymm	symm = (upper) ? MM_REAL_SYMMETRIC_UPPER : MM_REAL_SYMMETRIC_LOWER;
		s = mm_real_create_sparse_symm (symm, n);
		y = mm_real_create_dense (n, 1);
		z0 = mm_real_new (MM_REAL_DENSE, MM_REAL_GENERAL, n, 1, n);
		z1 = mm_real_new (MM_REAL_DENSE, MM_REAL_GENERAL, n, 1, n);
	} else {
		int		m = gsl_rng_uniform_int (rng, size) + 100;
		int		n = gsl_rng_uniform_int (rng, size) + 100;
		s = mm_real_create_sparse (m, n);
		y = mm_real_create_dense (m, 1);
		z0 = mm_real_new (MM_REAL_DENSE, MM_REAL_GENERAL, m, 1, m);
		z1 = mm_real_new (MM_REAL_DENSE, MM_REAL_GENERAL, m, 1, m);
	}

	mm_real_set_all (z0, 0.);
	for (j = 0; j < s->n; j++) mm_real_axjpy (alpha, s, j, z0);
#pragma omp parallel for
	for (j = 0; j < s->n; j++) mm_real_axjpy_atomic (alpha, s, j, z0);

	if (symm) x = mm_real_symm_to_general_sparse (s);
	else x = mm_real_sparse_to_dense (s);
	mm_real_free (s);

	mm_real_set_all (z1, 0.);
	for (j = 0; j < x->n; j++) mm_real_axjpy (alpha, x, j, z1);
#pragma omp parallel for
	for (j = 0; j < x->n; j++) mm_real_axjpy_atomic (alpha, x, j, z1);
	mm_real_free (x);
	mm_real_free (y);

	mm_real_axjpy (-1., z1, 0, z0);
	mm_real_free (z1);
	result = (mm_real_xj_nrm2 (z0, 0) < eps);
	mm_real_free (z0);

	return result;
}

bool
test_adjpy (FILE *stream, const long seed)
{
	int			j;
	bool		symm;

	mm_dense	*d;
	mm_dense	*x;
	mm_dense	*y;
	mm_dense	*z0, *z1;

	double		alpha;

	bool		result;

	gsl_rng_set (rng, seed);

	alpha = 1. - 2. * gsl_rng_uniform (rng);

	symm = (gsl_rng_uniform (rng) <= 0.5) ? false : true;

	if (symm) {
		int		n = gsl_rng_uniform_int (rng, size) + 100;
		bool	upper = (gsl_rng_uniform (rng) <= 0.5) ? false : true;
		MMRealSymm	symm = (upper) ? MM_REAL_SYMMETRIC_UPPER : MM_REAL_SYMMETRIC_LOWER;
		d = mm_real_create_dense_symm (symm, n);
		y = mm_real_create_dense (n, 1);
		z0 = mm_real_new (MM_REAL_DENSE, MM_REAL_GENERAL, n, 1, n);
		z1 = mm_real_new (MM_REAL_DENSE, MM_REAL_GENERAL, n, 1, n);
	} else {
		int		m = gsl_rng_uniform_int (rng, size) + 100;
		int		n = gsl_rng_uniform_int (rng, size) + 100;
		d = mm_real_create_dense (m, n);
		y = mm_real_create_dense (m, 1);
		z0 = mm_real_new (MM_REAL_DENSE, MM_REAL_GENERAL, m, 1, m);
		z1 = mm_real_new (MM_REAL_DENSE, MM_REAL_GENERAL, m, 1, m);
	}

	mm_real_set_all (z0, 0.);
	for (j = 0; j < d->n; j++) mm_real_axjpy (alpha, d, j, z0);
#pragma omp parallel for
	for (j = 0; j < d->n; j++) mm_real_axjpy_atomic (alpha, d, j, z0);

	if (symm) x = mm_real_symm_to_general_dense (d);
	else x = mm_real_copy (d);
	mm_real_free (d);

	mm_real_set_all (z1, 0.);
	for (j = 0; j < x->n; j++) mm_real_axjpy (alpha, x, j, z1);
#pragma omp parallel for
	for (j = 0; j < x->n; j++) mm_real_axjpy_atomic (alpha, x, j, z1);
	mm_real_free (x);
	mm_real_free (y);

	mm_real_axjpy (-1., z1, 0, z0);
	mm_real_free (z1);
	result = (mm_real_xj_nrm2 (z0, 0) < eps);
	mm_real_free (z0);

	return result;
}

#include <time.h>

int
main (void)
{
	int		i;

	rng = gsl_rng_alloc (gsl_rng_default);
	gsl_rng_set (rng, time (NULL));

#pragma omp parallel for
	for (i = 0; i < 1000; i++) {
		long seed = gsl_rng_uniform_int (rng, 10000);

		/* test xj_sum_ssq_nrm2 */
		if (!test_sj_sum_ssq_nrm2 (stdout, seed)) fprintf (stderr, "test_sj_sum_ssq_nrm2 failed.\n");
		if (!test_dj_sum_ssq_nrm2 (stdout, seed)) fprintf (stderr, "test_dj_sum_ssq_nrm2 failed.\n");

		/* test xj_trans_dot_y */
		if (!test_sj_trans_dot_y (stdout, seed)) fprintf (stderr, "test_sj_trans_dot_y failed.\n");
		if (!test_dj_trans_dot_y (stdout, seed)) fprintf (stderr, "test_dj_trans_dot_y failed.\n");

		/* test x_dot_y */
		if (!test_s_dot_y (stdout, seed)) fprintf (stderr, "test_s_dot_y failed.\n");
		if (!test_d_dot_y (stdout, seed)) fprintf (stderr, "test_d_dot_y failed.\n");

		/* test axjpy */
		if (!test_asjpy (stdout, seed)) fprintf (stderr, "test_asjpy failed.\n");
		if (!test_adjpy (stdout, seed)) fprintf (stderr, "test_adjpy failed.\n");

	}
	return EXIT_SUCCESS;
}


