package charactermanaj.model;

import java.io.Serializable;
import java.util.AbstractMap;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import charactermanaj.graphics.filters.ColorConvertParameter;

public class PartsColorInfo extends AbstractMap<Layer, ColorInfo> implements Serializable, Cloneable {

	private static final long serialVersionUID = -8639109147043912257L;

	private HashMap<Layer, ColorInfo> colorInfoMap = new HashMap<Layer, ColorInfo>();

	private final PartsCategory partsCategory;
	
	public PartsColorInfo(PartsCategory partsCategory) {
		if (partsCategory == null) {
			throw new IllegalArgumentException();
		}
		this.partsCategory = partsCategory;
		init();
	}
	
	protected PartsColorInfo(PartsColorInfo src) {
		if (src == null) {
			throw new IllegalArgumentException();
		}
		this.partsCategory = src.partsCategory;
		init();
		for (Map.Entry<Layer, ColorInfo> entry : src.colorInfoMap.entrySet()) {
			Layer layer = entry.getKey();
			ColorInfo colorInfo = entry.getValue();
			colorInfoMap.put(layer, colorInfo.clone());
		}
	}
	
	@Override
	public PartsColorInfo clone() {
		return new PartsColorInfo(this);
	}
	
	private void init() {
		for (Layer layer : partsCategory.getLayers()) {
			colorInfoMap.put(layer, createColorInfo(layer));
		}
	}
	
	protected ColorInfo createColorInfo(Layer layer) {
		ColorInfo colorInfo = new ColorInfo();
		colorInfo.setColorGroup(layer.getColorGroup());
		colorInfo.setSyncColorGroup(layer.getColorGroup().isEnabled());
		colorInfo.setColorParameter(new ColorConvertParameter());
		return colorInfo;
	}
	
	public PartsCategory getPartsCategory() {
		return partsCategory;
	}
	
	@Override
	public Set<java.util.Map.Entry<Layer, ColorInfo>> entrySet() {
		return colorInfoMap.entrySet();
	}
	
	@Override
	public ColorInfo put(Layer key, ColorInfo value) {
		if (key == null || value == null) {
			throw new IllegalArgumentException();
		}
		if (!colorInfoMap.containsKey(key)) {
			throw new IllegalArgumentException("invalid layer: " + key);
		}
		return colorInfoMap.put(key, value);
	}
	
	@Override
	public String toString() {
		StringBuilder buf = new StringBuilder();
		buf.append(getClass().getSimpleName() + "@" + Integer.toHexString(System.identityHashCode(this)));
		buf.append("(");
		buf.append(colorInfoMap.toString());
		buf.append(")");
		return buf.toString();
	}
	
}
