/*
 * print.c - print mrphs and paths
 *
 * last modified by A.Kitauchi <akira-k@is.aist-nara.ac.jp>, Sep. 1996
 *               by O.Imaichi <osamu-im@is.aist-nara.ac.jp>, Sep. 1996
 */

#include "chalib.h"
#include "config.h"

#define BASIC_FORM      "ܷ"
#define CHA_OUTPUT_SIZE (1024*16)

static int path_buffer[CHA_INPUT_SIZE];

static int is_bol = 1;

static void (*cha_putc)(), (*cha_puts)(), (*cha_printf)();
static void (*cha_fputc)(), (*cha_fputs)(), (*cha_fprintf)();

/*
 * cha_clputc, cha_clputs, cha_clprintf
 *	- output functions for ChaSen client
 */
static void cha_clputc(c, output)
    int c;
    FILE *output;
{
    if (is_bol && c == '.')
      putc('.', output);

    putc(c, output);

    is_bol = c == '\n' ? 1 : 0;
}

static void cha_clputs(s, output)
    char *s;
    FILE *output;
{
    if (is_bol && s[0] == '.')
      putc('.', output);

    fputs(s, output);

    is_bol = s[strlen(s) - 1] == '\n' ? 1 : 0;    
}

static void cha_clprintf(output, format, a, b, c, d, e, f, g, h)
    FILE *output;
    char *format, *a, *b, *c, *d, *e, *f, *g, *h;
{
    char tmpbuf[CHA_INPUT_SIZE];

    sprintf(tmpbuf, format, a, b, c, d, e, f, g, h);

    if (is_bol && tmpbuf[0] == '.')
      putc('.', output);

    fputs(tmpbuf, output);

    is_bol = tmpbuf[strlen(tmpbuf) - 1] == '\n' ? 1 : 0;
}

/*
 * cha_sputc, cha_sputs, cha_sprintf
 *	- output fuctions to string
 *
 * NOTE: `output' is a dummy argument for compatibility with cha_clputc, etc.
 *
 */

static char *cha_output;
static int cha_output_idx;
static int cha_output_nblock;

static void cha_sputc(c, output)
    int c;
    char *output; /* dummy */
{
    if (cha_output_idx + 1 >= CHA_OUTPUT_SIZE * cha_output_nblock
	&& cha_output) {
	cha_output = realloc(cha_output, CHA_OUTPUT_SIZE * ++cha_output_nblock);
    }

    if (cha_output) {
	cha_output[cha_output_idx++] = c;
	cha_output[cha_output_idx] = '\0';
    }
}

static void cha_sputs(s, output)
    char *s;
    char *output;
{
    int len = strlen(s);

    if (cha_output_idx + len >= CHA_OUTPUT_SIZE * cha_output_nblock
	&& cha_output) {
	cha_output = realloc(cha_output, CHA_OUTPUT_SIZE * ++cha_output_nblock);
    }

    if (cha_output) {
	strcpy(cha_output + cha_output_idx, s);
	cha_output_idx += len;
    }
}

static void cha_sprintf(output, format, a, b, c, d, e, f, g, h)
    char *output;
    char *format, *a, *b, *c, *d, *e, *f, *g, *h;
{
    char tmpbuf[CHA_INPUT_SIZE];
    int len;

    sprintf(tmpbuf, format, a, b, c, d, e, f, g, h);
    cha_sputs(tmpbuf, output);
}

void set_cha_fput(server_mode)
    int server_mode;
{
#ifndef HAVE_PROTO_FPUTC
    extern int fputc(), fputs();
    extern int fprintf();
#endif

    if (server_mode) {
	cha_fputc = (void (*))cha_clputc;
	cha_fputs = (void (*))cha_clputs;
	cha_fprintf = (void (*))cha_clprintf;
    } else {
	cha_fputc = (void (*))fputc;
	cha_fputs = (void (*))fputs;
	cha_fprintf = (void (*))fprintf;
    }
}

void set_cha_output(output)
    FILE *output;
{
    if (output == NULL) {
	/* output to string */
	cha_putc = (void (*))cha_sputc;
	cha_puts = (void (*))cha_sputs;
	cha_printf = (void (*))cha_sprintf;
	/* initialize output buffer */
	if (cha_output_nblock > 1) {
	    free(cha_output);
	    cha_output_nblock = 0;
	}
	if (cha_output_nblock == 0)
	  cha_output = malloc(CHA_OUTPUT_SIZE * ++cha_output_nblock);
	cha_output_idx = 0;
	cha_output[0] = '\0';
    } else {
	/* output to file */
	cha_output = (char *)output;
	cha_putc = (void (*))cha_fputc;
	cha_puts = (void (*))cha_fputs;
	cha_printf = (void (*))cha_fprintf;
    }
}

/*
 * returns cha_output for chasen_[fs]arse_tostr()
 */
char *get_cha_output()
{
    return cha_output;
}

/*
 * printf_mrph - print morpheme using format string
 *
 * format string:
 *	%m     Ф(ѷ)
 *	%M     Ф(ܷ)
 *	%y     ɤ
 *	%Y     ɤ(̤ʤ"̤")
 *      %rAB   Ф(ӤĤ)(ɤߤʸAʸBǰϤ)
 *	%i     ̣
 *	%Ic    ̣(NILʤʸc)
 *	%Pc    ƳؤʻʸcǶڤäʸ(¿ʻΤ)
 *	%Pnc   1n(n:19)ܤޤǤʻʸcǶڤäʸ
 *             (¿ʻΤ)
 *	%h     ʻֹ
 *	%H     ʻʸ(¿ʻξ1)
 *	%Hn    n(n:19)ܤʻʸ(ʤкǤ⿼ؤɽ)
 *             (¿ʻΤ)
 *	%b     ʻʬֹ(¿ʻξ0)
 *	%BB    ʻʬʸ(ʤʻ)
 *	%BM    ʻʬʸ(ʤʻޤ"̤")
 *	%Bc    ʻʬʸ(ʤʸc)
 *	%t     ѷֹ
 *	%Tc    ѷʸ(ʤʸc)
 *	%f     ѷֹ
 *	%Fc    ѷʸ(ʤʸc)
 *	%c     ǤΥ
 *	%pi    ѥֹ
 *	%ps    ѥηʸγϰ
 *	%pe    ѥηʸνλ
 *	%pc    ѥΥ
 *	%ppiC  ˤĤʤѥֹʸCǶڤ
 *	%ppcC  ˤĤʤѥΥȤʸCǶڤ
 *	%%     % Τ
 *	.      ե
 *	-      ե
 *	1-9    ե
 *	\n     ʸ
 *	\t     
 *	\\     \ Τ
 *	\'     ' Τ
 *	\"     " Τ
 *
 *      %Ic,%Bc,%Tc,%Fc ˤĤƤ c  ' ' ()λϲɽʤ
 *
 * example:
 *	"%m %y %M %h %b %t %f\n"                - same as '-c' option
 *	"%m %Y %M %H %h %B* %b %T* %t %F* %f\n" - same as '-e' option
 */
static int check_con_cost(path, con_tbl)
    path_t *path;
    int con_tbl;
{
    int con_cost;

    check_automaton(path->state,
#ifdef KOCHA
		    Path[path->path[0]].state,
		    Mrph[path->mrph_p].midasi,
		    Mrph[path->mrph_p].is_undef,
#endif
		    con_tbl, Undef_con_cost, &con_cost);

    return con_cost;
}

#ifdef VGRAM
static void print_nhinsi(hinsi, c, n)
    int hinsi, c, n;
{
    short *path;
    int i;

    path = Hinsi[hinsi].path;
    for (i = 0; ; i++) {
	cha_puts(Hinsi[*path].name, cha_output);
	if (!*++path || i == n)
	  break;
	cha_putc(c, cha_output);
    }
}
#endif


/***********************************************************************
 * int_to_str - convert an integer to ASCII
 *	by Masanao Izumo <masana-i@is.aist-nara.ac.jp>
 ***********************************************************************/
static char *int_to_str(value)
    int value;
{
    static char buff[32];
    char* p;
    int sign;

    p = buff + 31;
    if (value >= 0)
      sign = 0;
    else {
	if (-value == value) { /* value == INT_MIN */
	    sprintf(buff, "%d", value);
	    return buff;
	}
	value = -value;
	sign = 1;
    }

    do {
	*--p = value % 10 + '0';
	value /= 10;
    } while (value > 0);
    if(sign)
      *--p = '-';

    return p;
}    

/***********************************************************************
 * fputsn
 *	by Masanao Izumo <masana-i@is.aist-nara.ac.jp>
 ***********************************************************************/
static void fputsn(str, fp, n)
    char *str;
    FILE* fp;
    int n;
{
    char buff[256];
    int len;

    while (n > 0) {
	len = (n <= 255 ? n : 255);
	memcpy(buff, str, len);
	buff[len] = '\0';
	cha_puts(buff, fp);
	str += len;
	n -= len;
    }
}

/***********************************************************************
 * printf_field
 *	by Masanao Izumo <masana-i@is.aist-nara.ac.jp>
 ***********************************************************************/
static void printf_field(width_str, word)
    char *width_str, *word;
{
    char *field = width_str;
    int field_len, word_len;

    if (width_str == NULL) {
	cha_puts(word, cha_output);
	return;
    }

    if (*field == '-')
      field++;

    word_len = -1;
    field_len = 0;
    while('0' <= *field && *field <= '9') {
	field_len = field_len * 10 + *field - '0';
	field++;
    }

    if (*field == '.') {
	int len = 0;
	word_len = strlen(word);
	field++;
	while ('0' <= *field && *field <= '9') {
	    len = len * 10 + *field - '0';
	    field++;
	}
	if (len < word_len)
	  word_len = len;
    }

    if (word_len == -1) {
	word_len = strlen(word);
	if (*width_str == '-') {
	    cha_puts(word, cha_output);
	    field_len -= word_len;
	    while(field_len-- > 0)
	      cha_putc(' ', cha_output);
	} else {
	    field_len -= word_len;
	    while(field_len-- > 0)
	      cha_putc(' ', cha_output);
	    cha_puts(word, cha_output);
	}
    } else {
	if (*width_str == '-') {
	    fputsn(word, cha_output, word_len);
	    field_len -= word_len;
	    while(field_len-- > 0)
	      cha_putc(' ', cha_output);
	} else {
	    field_len -= word_len;
	    while(field_len-- > 0)
	      cha_putc(' ', cha_output);
	    fputsn(word, cha_output, word_len);
	}
    }
}

static void printf_mrph(path_num, mrph, format)
    int path_num;
    mrph2_t *mrph;
    char *format;
{
    int  letter, value, n, c;
    char *s, *word, word_str[CHA_INPUT_SIZE];
    char *width_str;
    path_t *path;

    path = &Path[path_num];

    word = NULL;      /* string */
    letter = 0;       /* character */
    value = INT_MAX;  /* integer value */

    for (s = format; *s; s++) {
	/* escape characters */
	if (*s == '\\') {
	    switch (*++s) {
	      case 't':
		cha_putc('\t', cha_output);
		break;
	      case 'n':
		cha_putc('\n', cha_output);
		break;
#if 0
	      case '\\':
	      case '\'':
	      case '"':
#endif
	      default: /* including back slash and single/double quoation */
		cha_putc(*s, cha_output);
		break;
#if 0
		goto error_end;
#endif
	    }
	    continue;
	}

	if (*s != '%') {
	    cha_putc(*s, cha_output);
	    continue;
	}

	s++;
	if (*s == '-' || *s == '.' || (*s >= '0' && *s <= '9')) {
	    width_str = s;
	    while (*s == '-' || *s == '.' || (*s >= '0' && *s <= '9'))
	      s++;
	} else {
	    width_str = NULL;
	}

	switch (*s) {
	  case 'm': /* Ф(ѷ) */
	    memcpy(word_str, mrph->midasi, mrph->length);
	    word_str[mrph->length] = '\0';
	    word = word_str;
	    break;
	  case 'M': /* Ф(ܷ) */
	    memcpy(word_str, mrph->midasi, mrph->base_length);
	    if (mrph->ktype > 0 && mrph->kform > 0) {
		int b;
		for (b = 1;
		     strcmp(Form[mrph->ktype][b].name, BASIC_FORM);
		     b++);
		strcpy(word_str + mrph->base_length,
		       Form[mrph->ktype][b].gobi);
	    } else {
		word_str[mrph->base_length] = '\0';
	    }
#ifdef KOCHA2
	    jaso2ks(word_str);
#endif
	    word = word_str;
	    break;
	  case 'Y': /* ɤ / "̤" */
	    if (mrph->is_undef) {
		word = "̤";
		break;
	    }
	    /* FALLTHRU */
	  case 'y': /* ɤ */
	    strcpy(word_str, mrph->yomi);
	    if (mrph->ktype > 0 && mrph->kform > 0)
	      strcat(word_str, Form[mrph->ktype][mrph->kform].ygobi);
#ifdef KOCHA2
	    jaso2ks(word_str);
#endif
	    word = word_str;
	    break;
	  case 'r': /* Ф(ӤĤ) */
	    if (s[1] == '\0' || s[2] == '\0')
	      cha_putc(*s, cha_output);
	    else {
		char yomi[CHA_INPUT_SIZE];
		int a, b, pre, suf;

		strcpy(yomi, mrph->yomi);
		if (mrph->ktype > 0 && mrph->kform > 0)
		  strcat(yomi, Form[mrph->ktype][mrph->kform].ygobi);

		if (!memcmp(mrph->midasi, yomi, mrph->length)) {
		    word = yomi;
		} else {
		    /*
		     * midashi: a(>=pre) + suf
		     * yomi:    pre + b + suf
		     * pre and suf are hiraganas.
		     */
		    /* pre: hiraganas at the beginning of word */
		    for (pre = 0;
			 mrph->midasi[pre  ] == yomi[pre  ] &&
			 mrph->midasi[pre+1] == yomi[pre+1];
			 pre += 2)
		      ; /* empty */
		    /* suf: hiraganas at the end of word */
		    for (a = mrph->length, b = strlen(yomi), suf = 0;
			 mrph->midasi[a-1] == yomi[b-1] &&
			 mrph->midasi[a-2] == yomi[b-2];
			 a -= 2, b -= 2, suf += 2)
		      ; /* empty */
		    b -= pre;
		    word = word_str;
		    memcpy(word, mrph->midasi, a); word += a;
		    *word++ = s[1]; /* '(' */
		    memcpy(word, yomi + pre, b); word += b;
		    *word++ = s[2]; /* ')' */
		    memcpy(word, mrph->midasi + a, suf); word += suf;
		    *word = '\0';
		    word = word_str;
		}
		s += 2;
	    }
	    break;
	  case 'i': /* ̣ */
	    word = mrph->info;
	    break;
	  case 'I': /* ̣ */
	    if (*++s == '\0')
	      cha_putc(*--s, cha_output);
	    else if (strcmp(mrph->info, "NIL"))
	      word = mrph->info;
	    else if (*s != ' ')
	      letter = *s;
	    break;
#ifdef VGRAM
	  case 'P':
	    if (s[1] >= '1' && s[1] <= '9')
	      n = *++s - '1';
	    else
	      n = 99; /* ٤Ƥγؤɽ */
	    if (s[1] == '\0')
	      cha_putc(*s, cha_output);
	    else
	      print_nhinsi(mrph->hinsi, *++s, n);
	    break;
#endif
	  case 'h': /* ʻ(ֹ) */
	    value = mrph->hinsi;
	    break;
	  case 'H': /* ʻ(ʸ) */
#ifdef VGRAM
	    if (s[1] < '1' || s[1] > '9')
	      n = 0;
	    else {
		n = *++s - '1';
		if (Hinsi[mrph->hinsi].depth - 1 < n)
		  n = Hinsi[mrph->hinsi].depth - 1;
	    }
	    word = Hinsi[Hinsi[mrph->hinsi].path[n]].name;
#else
	    word = Class[mrph->hinsi][0].id;
#endif
	    break;
	  case 'b': /* ʻʬ(ֹ) */
#ifdef VGRAM
	    value = 0;
#else
	    value = mrph->bunrui;
#endif
	    break;
	  case 'B': /* ʻʬ(ʸ) */
	    if (s[1] == '\0')
	      cha_putc(*s, cha_output);
	    else if (*++s == 'M' && mrph->is_undef)
	      word = "̤";
#ifdef VGRAM
	    /* زʻʤֲγؤʻ̾ɽ */
	    else if (*s == 'M' || *s == 'B' || Hinsi[mrph->hinsi].depth > 1)
	      word = Hinsi[mrph->hinsi].name;
#else
	    else if (*s == 'M' || *s == 'B' || mrph->bunrui)
	      word = Class[mrph->hinsi][mrph->bunrui].id;
#endif
	    else if (*s != ' ')
	      letter = *s;
	    break;
	  case 't': /* ѷ(ֹ) */
	    value = mrph->ktype;
	    break;
	  case 'T': /* ѷ(ʸ) */
	    if (*++s == '\0')
	      cha_putc(*--s, cha_output);
	    else if (mrph->ktype)
	      word = Type[mrph->ktype].name;
	    else if (*s != ' ')
	      letter = *s;
	    break;
	  case 'f': /* ѷ(ֹ) */
	    value = mrph->kform;
	    break;
	  case 'F': /* ѷ(ʸ) */
	    if (*++s == '\0')
	      cha_putc(*--s, cha_output);
	    if (mrph->kform)
	      word = Form[mrph->ktype][mrph->kform].name;
	    else if (*s != ' ')
	      letter = *s;
	    break;
	  case 'c': /* ǤΥ */
#ifdef VGRAM
	    value = (mrph->is_undef ? Undef_cost : Hinsi[mrph->hinsi].cost)
	      * mrph->weight * Mrph_cost_weight;
#else
	    value = (mrph->is_undef ? Undef_cost : Class[mrph->hinsi][mrph->bunrui].cost)
	      * mrph->weight * Mrph_cost_weight;
#endif
	    break;
	  case 'p': /* path ˴ؤ */
	    if (s[1] == '\0') {
		cha_putc(*s, cha_output);
		break;
	    }
	    switch (*++s) {
	      case 'i':
		value = path_num;
		break;
	      case 's':
		value = path->start;
		break;
	      case 'e':
		value = path->end;
		break;
	      case 'c':
		value = path->cost;
		break;
	      case 'p':
		if ((s[1] != 'i' && s[1] != 'c') || s[2] == '\0')
		  cha_putc(*s, cha_output);
		else if (*++s == 'i') {
		    int c = *++s, j;
		    for (j = 0; path->path[j] != -1; j++) {
			if (j)
			  cha_putc(c, cha_output);
			cha_printf(cha_output, "%d", path->path[j]);
		    }
		} else {
		    int con_tbl = mrph->con_tbl;
		    int c = *++s, j;
		    for (j = 0; path->path[j] != -1; j++) {
			if (j)
			  cha_putc(c, cha_output);
			cha_printf(cha_output, "%d", Con_cost_weight *
				   check_con_cost(&Path[path->path[j]], con_tbl));
		    }
		}		
		break;
	    }
	    break;
	  default: /* includes '%' */
	    cha_putc(*s, cha_output);
	}

	if (word != NULL) {
	    printf_field(width_str, word);
	    word = NULL;
	} else if (letter) {
	    word_str[0] = letter;
	    word_str[1] = '\0';
	    printf_field(width_str, word_str);
	    letter = 0;
	} else if (value != INT_MAX) {
	    printf_field(width_str, int_to_str(value));
	    value = INT_MAX;
	}
    }

    return;

#if 0
  error_end:
    cha_exit(1, "format error near the point %d in option -- F", s - format + 1);
#endif
}

/***********************************************************************
 * print_mrph - print morpheme using option or format string
 ***********************************************************************/
void print_mrph(path_num, mrph, opt_form, format)
    int path_num;
    mrph2_t *mrph;
    int opt_form;
    char *format;
{
    /* opt_form  'F' ʤ format 򤽤Τޤ޻ */
    switch (opt_form) {
#ifdef VGRAM
      case 'd':
	format = "morph(%pi,%ps,%pe,%pc,'%m','%Y','%M',%P-,NIL,%T0,%F0,'%I0',%c,[%ppc,],[%ppi,])";
	break;
      case 'v':
	format = "%3pi %3ps %3pe %5pc %m %Y %M %P- NIL %T0 %F0 %I0 %c %ppi, %ppc,\n";
	break;
      case 'f':
	/* !VGRAM Ʊ */
	format = "%-11m %-11y %-11M %-14BM %-14T  %F \n";
	break;
      case 'e':
	format = "%m %Y %M %P- %h %T* %t %F* %f\n";
	break;
      case 'c':
	format = "%m %y %M %h %t %f\n";
	break;
#else
      case 'd':
	format = "morph(%pi,%ps,%pe,%pc,'%m','%Y','%M',%H,%BB,%T0,%F0,'%I0',%c,[%ppc,],[%ppi,])";
	break;
      case 'v':
	format = "%3pi %3ps %3pe %5pc %m %Y %M %H %BB %T0 %F0 %I0 %c %ppi, %ppc,\n";
	break;
      case 'f':
	format = "%-11m %-11y %-11M %-14BM %-14T  %F \n";
	break;
      case 'e':
	format = "%m %Y %M %H %h %B* %b %T* %t %F* %f\n";
	break;
      case 'c':
	format = "%m %y %M %h %b %t %f\n";
	break;
#endif
    }

    printf_mrph(path_num, mrph, format);
}

static void print_eos(opt_form)
    int opt_form;
{
    if (opt_form == 'W')
      cha_putc('\n', cha_output);
    else if (opt_form != 'd')
      cha_puts("EOS\n", cha_output);
}

/*
------------------------------------------------------------------------------
  PROCEDURE: <print_path_mrph>
------------------------------------------------------------------------------
*/
static void print_path_mrph(path_num, opt_form, format)
    int path_num, opt_form;
    char *format;
{
#ifdef KOCHA_DEBUG
    printf("%4d: ", Path[path_num].mrph_p);
#endif
    print_mrph(path_num, &Mrph[Path[path_num].mrph_p], opt_form, format);
}

/*
------------------------------------------------------------------------------
  PROCEDURE: <print_best_path> ǾȤΥѥΤߤɽ
------------------------------------------------------------------------------
*/
static void print_best_path(opt_form, format)
    int opt_form;
    char *format;
{
    int i, j, last, isfirst = 1;
    mrph2_t mrph, *mrph1, *mrph2;
    char yomi[CHA_INPUT_SIZE];

    j = 0;
    last = Path_num - 1;

    do {
	last = Path[last].path[0];
	path_buffer[j++] = last;
#if 0
	printf("# last,path0: %d,%d\n",last,Path[last].path[0]);fflush(stdout);
#endif
    } while (Path[last].path[0]);

    /* ϢʻñϢ뤷ɽ */
    mrph.hinsi = 0;
    mrph.yomi = yomi;
    mrph1 = &Mrph[Path[path_buffer[j - 1]].mrph_p];
    for (i = j - 1; i >= 0; i--) {
	mrph2 = i == 0 ? NULL : &Mrph[Path[path_buffer[i - 1]].mrph_p];
	if (i > 0 &&
	    !mrph1->is_undef && !mrph2->is_undef &&
#ifdef VGRAM
	    Hinsi[mrph1->hinsi].comp &&
	    Hinsi[mrph1->hinsi].comp == Hinsi[mrph2->hinsi].comp
#else
	    Class[mrph1->hinsi][mrph1->bunrui].comp &&
	    mrph1->hinsi == mrph2->hinsi &&
	    (Class[mrph1->hinsi][mrph1->bunrui].comp == 1 ||
	     mrph1->bunrui == mrph2->bunrui)
#endif
	    ) {
	    if (!mrph.hinsi) {
#ifdef VGRAM
		mrph.hinsi = Hinsi[mrph1->hinsi].comp;
#else
		mrph.hinsi = mrph1->hinsi;
		if (Class[mrph1->hinsi][mrph1->bunrui].comp == 1)
		  mrph.bunrui = 0;
		else
		  mrph.bunrui = mrph1->bunrui;
#endif
		mrph.midasi = mrph1->midasi;
		yomi[0] = '\0';
		mrph.length = mrph.weight = 0;
	    }
	    strcat(yomi, mrph1->yomi);
	    if (mrph1->ktype > 0 && mrph1->kform > 0)
	      strcat(yomi, Form[mrph1->ktype][mrph1->kform].ygobi);
	    mrph.length += mrph1->length;
	    mrph.weight += mrph1->weight;
	} else {
	    if (opt_form == 'd') {
		if (isfirst)
		  isfirst = 0;
		else
		  cha_putc(',', cha_output);
	    }
	    if (!mrph.hinsi)
	      print_mrph(path_buffer[i], mrph1, opt_form, format);
	    else {
		strcat(yomi, mrph1->yomi);
		mrph.base_length = mrph.length + mrph1->base_length;
		mrph.length += mrph1->length;
		mrph.weight += mrph1->weight;
		mrph.info = mrph1->info;
		mrph.ktype = mrph1->ktype;
		mrph.kform = mrph1->kform;
		mrph.is_undef = mrph1->is_undef;
		print_mrph(path_buffer[i], &mrph, opt_form, format);
		mrph.hinsi = 0;
	    }
	}
	mrph1 = mrph2;
    }

    print_eos(opt_form);
}

/*
 * print_all_mrph - Ϸ̤˴ޤޤƤηǤɽ
 *	-m, -d, -v ץǻ
 */
static void collect_all_mrph(path_num)
    int path_num;
{
    int i, j;

    for (i = 0; (j = Path[path_num].path[i]) && j != -1; i++) {
	if (!Path[j].do_print) {
	    Path[j].do_print = 1;
	    collect_all_mrph(j);
	}
    }
}

static void print_all_mrph(opt_form, format)
    int opt_form;
    char *format;
{
    int i, j;
    int isfirst = 1; /* ʸƬɤΥե饰 for -d option */

    for (i = 0; i < Path_num; i++)
      Path[i].do_print = 0;
    collect_all_mrph(Path_num - 1);

    /* -v ΤȤʸƬʸξɽ */
    if (opt_form == 'v') {
	Path[0].do_print = 1;
	Path[Path_num - 1].do_print = 1;
    }

    for (i = 0; i < Path_num; i++) {
	if (Path[i].do_print) {
	    if (opt_form == 'd') {
		if (isfirst)
		  isfirst = 0;
		else
		  cha_putc(',', cha_output);
	    }
	    print_path_mrph(i, opt_form, format);
	}
    }

    print_eos(opt_form);
}

/*
------------------------------------------------------------------------------
  PROCEDURE: <print_all_path> Ϸ̤˴ޤޤƤPATH
------------------------------------------------------------------------------
*/
static void print_all_path_sub(path_num, paths, opt_form, format)
    int path_num, paths, opt_form;
    char *format;
{
    int i, j;

    for (i = 0; Path[path_num].path[i] != -1; i++) {
	if (Path[path_num].path[0] == 0) {
	    for (j = paths - 1; j >= 0; j--)
	      print_path_mrph(path_buffer[j], opt_form, format);
	    cha_puts("EOP\n", cha_output);
	} else {
	    path_buffer[paths] = Path[path_num].path[i];
	    print_all_path_sub(Path[path_num].path[i], paths + 1,
			       opt_form, format);
	}
    }
}

static void print_all_path(opt_form, format)
    int opt_form;
    char *format;
{
    print_all_path_sub(Path_num - 1, 0, opt_form, format);

    print_eos(opt_form);
}

void cha_print_path(opt_show, opt_form, format)
    int opt_show, opt_form;
    char *format;
{
    if (opt_form == 'd')
      cha_putc('[', cha_output);

    switch (opt_show) {
      case 'm': print_all_mrph(opt_form, format); break;
      case 'p': print_all_path(opt_form, format); break;
      default:  print_best_path(opt_form, format); /* 'b' */
    }

    if (opt_form == 'd')
      cha_puts("].\n", cha_output);
}

