/*
 * chomocon-console 0.1
 *
 * ChoroMode (ChoroPet) controler console edition
 */
/*
 * dtmf-dial 0.1
 * (C) 1998 Itai Nahshon (nahshon@actcom.co.il)
 *
 * Use and redistribution are subject to the GNU GENERAL PUBLIC LICENSE.
 */
#include <fcntl.h>
#include <math.h>
#include <malloc.h>
#include <pthread.h>
#include <semaphore.h>
#include <stdio.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>
#include <linux/soundcard.h>

void getvalue(int *arg, int *index, int argc, char **argv, int min, int max);
void* dial_thread_main(void *arg);
void initialize_audiodev(void);
void gen_costab(void);
char* make_dialbuf(int f1, int f2);

int set_tty_raw(void);
int set_tty_cooked(void);
unsigned char kb_getc_w(void);
void kb_clear(void);

char *output = "/dev/dsp";
int bits        = 8;
int speed       = 8000;
int tone_time   = 100;
int volume      = 100;
int format	= AFMT_U8;
int use_audio   = 1;
int bufsize;

/*
 * default repeat interval is 2.5[sec]
 * Because ChoroQ continues to run about 3 seconds at a key press.
 */
int repeat_interval_sec = 2;
int repeat_interval_usec = 500000; /* 0 <= repeat_interval_usec < 1000000 */

#define MINTABSIZE 2
#define MAXTABSIZE 65536
signed short *costab;
int tabsize = 256;

int right  = 0;
int left   = 0;
int fd;

void
Usage(void) {
	fprintf(stderr, "usage: dial [options]\n"
			" Valid options with their default values are:\n"
			"   Duration options:\n"
			"     --tone-time   100\n"
			"   Audio output  options:\n"
			"     --output-dev  /dev/dsp\n"
			"     --use-audio   1\n"
			"     --speed       8000\n"
			"     --bits        8\n"
			"   Audio generation options:\n"
			"     --table-size  256\n"
			"     --volume      100\n"
			"     --left        0\n"
			"     --right       0\n"
		        "   Repeat interval options:\n"
			"     --interval    2500000\n"
			);
			
	exit(1);
}

struct dial_data {
	char ch;
	pthread_mutex_t mutex;
	sem_t sem;
} dial_data = {(char)0};

int
main(int argc, char **argv)
{
	pthread_t dial_thread;
	void *thread_return;
	int continue_loop;
	int prev_is_0x1b_0x5b;
	struct timeval time_sent;
	struct timeval time_now;
	char prev_sent_ch;
	enum GEAR_STATE {NEUTRAL, FORWARD, BACK} gear;
	int i;

	for(i = 1; i < argc; i++) {
		if(argv[i][0] != '-' ||
		   argv[i][1] != '-')
			break;

		if(!strcmp(argv[i], "--table-size")) {
			getvalue(&tabsize, &i, argc, argv,
				 MINTABSIZE, MAXTABSIZE);
		}
		else if(!strcmp(argv[i], "--tone-time")) {
			getvalue(&tone_time, &i, argc, argv,
				 10, 10000);
		}
		else if(!strcmp(argv[i], "--volume")) {
			getvalue(&volume, &i, argc, argv,
				 0, 100);
		}
		else if(!strcmp(argv[i], "--speed")) {
			getvalue(&speed, &i, argc, argv,
				 5000, 48000);
		}
		else if(!strcmp(argv[i], "--bits")) {
			getvalue(&bits, &i, argc, argv,
				 8, 16);
		}
		else if(!strcmp(argv[i], "--use-audio")) {
			getvalue(&use_audio, &i, argc, argv,
				 0, 1);
		}
		else if(!strcmp(argv[i], "--right")) {
			getvalue(&right, &i, argc, argv,
				 0, 1);
		}
		else if(!strcmp(argv[i], "--left")) {
			getvalue(&left, &i, argc, argv,
				 0, 1);
		}
		else if(!strcmp(argv[i], "--output-dev")) {
			i++;
			if(i >= argc)
				Usage();
			output = argv[i];
		}
		else if(!strcmp(argv[i], "--interval")) {
			getvalue(&repeat_interval_usec, &i, argc, argv,
				 500000, 3000000);
			repeat_interval_sec = repeat_interval_usec / 1000000;
			repeat_interval_usec %= 1000000;
		}
		else
			Usage();
	}

	if(i < argc)
		Usage();

	if(!strcmp(output, "-"))
		fd = 1;		/* stdout */
	else {
		fd = open(output, O_CREAT|O_TRUNC|O_WRONLY, 0644);
		if(fd < 0) {
			perror(output);
			exit(1);
		}
	}

	switch(bits) {
		case 8:
			format = AFMT_U8;
			break;
		case 16:
			format = AFMT_S16_LE;
			break;
		default:
			fprintf(stderr, "Value for bits should be 8 or 16\n");
			exit(1);
	}

	pthread_mutex_init(&dial_data.mutex, NULL);
	sem_init(&dial_data.sem, 0, 0);

	if (pthread_create(&dial_thread, NULL, dial_thread_main, NULL) != 0) {
		fprintf(stderr, "failed to pthread_create(3)\n");
		exit(1);
	}

	set_tty_raw(); /* set up character-at-a-time */

	continue_loop = 1;
	prev_is_0x1b_0x5b = 0;
	gear = NEUTRAL;
	prev_sent_ch = '\0';
	while (continue_loop) {
		char ch;
		int sem_value;

		ch = kb_getc_w();
		/*
		 * character code filter
		 * - controling with arrow keys
		 */
		switch (ch) {
		case 0x41: /* [up] "0x1b 0x5b 0x41" */
			if (prev_is_0x1b_0x5b == 0x5b) {
				if (gear == BACK) {
					gear = NEUTRAL;
					ch = '5';
					break;
				} else {
					gear = FORWARD;
					ch = '8';
					break;
				}
			}
			kb_clear();
			continue;
 		case 0x42: /* [down] "0x1b 0x5b 0x42" */
			if (prev_is_0x1b_0x5b == 0x5b) {
				if (gear == FORWARD) {
					gear = NEUTRAL;
					ch = '5';
					break;
				} else {
					gear = BACK;
					ch = '2';
					break;
				}
			}
			kb_clear();
			continue;
		case 0x43: /* [right] "0x1b 0x5b 0x43" */
			if (prev_is_0x1b_0x5b == 0x5b) {
				if (gear == FORWARD) {
					/* gear keeps FORWARD */
					ch = '9';
					break;
				} else if (gear == BACK) {
					/* gear keeps BACK */
					ch = '3';
					break;
				} else {
					/* don't move in NEUTRAL */
					prev_is_0x1b_0x5b = 0;
				}
			}
			kb_clear();
			continue;
		case 0x44: /* [left] "0x1b 0x5b 0x44" */
			if (prev_is_0x1b_0x5b == 0x5b) {
				if (gear == FORWARD) {
					/* gear keeps FORWARD */
					ch = '7';
					break;
				} else if (gear == BACK) {
					/* gear keeps BACK */
					ch = '1';
					break;
				} else {
					/* don't move in NEUTRAL */
					prev_is_0x1b_0x5b = 0;
				}
			}
			kb_clear();
			continue;
		case 0x1b:
			prev_is_0x1b_0x5b = 0x1b;
			continue;
		case 0x5b:
			if (prev_is_0x1b_0x5b == 0x1b) {
				prev_is_0x1b_0x5b = 0x5b;
				continue;
			}
			prev_is_0x1b_0x5b = 0;
			kb_clear();
			continue;
		case 'u':
			ch = '1';
			gear = NEUTRAL;
			break;
		case 'i':
			ch = '2';
			gear = NEUTRAL;
			break;
		case 'o':
			ch = '3';
			gear = NEUTRAL;
			break;
		case 'j':
			ch = '4';
			gear = NEUTRAL;
			break;
		case 'k':
			ch = '5';
			gear = NEUTRAL;
			break;
		case 'l':
			ch = '6';
			gear = NEUTRAL;
			break;
		case 'm':
			ch = '7';
			gear = NEUTRAL;
			break;
		case ',':
			ch = '8';
			gear = NEUTRAL;
			break;
		case '.':
			ch = '9';
			gear = NEUTRAL;
			break;
		case '0':
		case '1':
		case '2':
		case '3':
		case '4':
		case '5':
		case '6':
		case '7':
		case '8':
		case '9':
		case '*':
		case '#':
			gear = NEUTRAL;
			break;
		case 0x03:
			/* control-C */
			continue_loop = 0;
			break;
		default:
			gear = NEUTRAL;
			prev_is_0x1b_0x5b = 0;
			kb_clear();
			continue;
		}
		prev_is_0x1b_0x5b = 0;

		gettimeofday(&time_now, NULL);
		if (prev_sent_ch == ch) {
			/* prevent to repeat rush */
			int past_sec = time_now.tv_sec - time_sent.tv_sec;
			int past_usec = time_now.tv_usec - time_sent.tv_usec;
			if (past_sec * 1000000 + past_usec <
			    repeat_interval_sec * 1000000 + repeat_interval_usec) {
					kb_clear();
					continue;
			}
#ifdef DEBUG
fprintf(stderr, "%d ", past_sec * 1000000 + past_usec);
#endif /* DEBUG */
		}
		prev_sent_ch = ch;
		time_sent = time_now;

		pthread_mutex_lock(&dial_data.mutex);
		dial_data.ch = ch;
		pthread_mutex_unlock(&dial_data.mutex);
		sem_getvalue(&dial_data.sem, &sem_value);
		if (sem_value == 0) {
			sem_post(&dial_data.sem);
		}
		kb_clear();
	}
	set_tty_cooked(); /* control-C, restore normal TTY mode */
	pthread_join(dial_thread, &thread_return);
	sem_destroy(&dial_data.sem);
	pthread_mutex_destroy(&dial_data.mutex);

	return 0;
}

void *
dial_thread_main(void *arg)
{
	char *dialbuf_941_1336 = NULL; /*'0'*/
	char *dialbuf_697_1209 = NULL; /*'1'*/
	char *dialbuf_697_1336 = NULL; /*'2'*/
	char *dialbuf_697_1477 = NULL; /*'3'*/
	char *dialbuf_770_1209 = NULL; /*'4'*/
	char *dialbuf_770_1336 = NULL; /*'5'*/
	char *dialbuf_770_1477 = NULL; /*'6'*/
	char *dialbuf_852_1209 = NULL; /*'7'*/
	char *dialbuf_852_1336 = NULL; /*'8'*/
	char *dialbuf_852_1477 = NULL; /*'9'*/
	char *dialbuf_941_1209 = NULL; /*'*'*/
	char *dialbuf_941_1477 = NULL; /*'#'*/
#if 0 /* NOUSE */
	char *dialbuf_697_1633 = NULL; /*'A'*/
	char *dialbuf_770_1633 = NULL; /*'B'*/
	char *dialbuf_852_1633 = NULL; /*'C'*/
	char *dialbuf_941_1633 = NULL; /*'D'*/
#endif

	initialize_audiodev();
	gen_costab();

	dialbuf_941_1336 = make_dialbuf(941, 1336); /*'0'*/
	dialbuf_697_1209 = make_dialbuf(697, 1209); /*'1'*/
	dialbuf_697_1336 = make_dialbuf(697, 1336); /*'2'*/
	dialbuf_697_1477 = make_dialbuf(697, 1477); /*'3'*/
	dialbuf_770_1209 = make_dialbuf(770, 1209); /*'4'*/
	dialbuf_770_1336 = make_dialbuf(770, 1336); /*'5'*/
	dialbuf_770_1477 = make_dialbuf(770, 1477); /*'6'*/
	dialbuf_852_1209 = make_dialbuf(852, 1209); /*'7'*/
	dialbuf_852_1336 = make_dialbuf(852, 1336); /*'8'*/
	dialbuf_852_1477 = make_dialbuf(852, 1477); /*'9'*/
	dialbuf_941_1209 = make_dialbuf(941, 1209); /*'*'*/
	dialbuf_941_1477 = make_dialbuf(941, 1477); /*'#'*/
#if 0 /* NOUSE */
	dialbuf_697_1633 = make_dialbuf(697, 1633); /*'A'*/
	dialbuf_770_1633 = make_dialbuf(770, 1633); /*'B'*/
	dialbuf_852_1633 = make_dialbuf(852, 1633); /*'C'*/
	dialbuf_941_1633 = make_dialbuf(941, 1633); /*'D'*/
#endif

	if (dialbuf_941_1336 == NULL
	    || dialbuf_697_1209 == NULL
	    || dialbuf_697_1336 == NULL
	    || dialbuf_697_1477 == NULL
	    || dialbuf_770_1209 == NULL
	    || dialbuf_770_1336 == NULL
	    || dialbuf_770_1477 == NULL
	    || dialbuf_852_1209 == NULL
	    || dialbuf_852_1336 == NULL
	    || dialbuf_852_1477 == NULL
	    || dialbuf_941_1209 == NULL
	    || dialbuf_941_1477 == NULL
#if 0 /* NOUSE */
	    || dialbuf_697_1633 == NULL
	    || dialbuf_770_1633 == NULL
	    || dialbuf_852_1633 == NULL
	    || dialbuf_941_1633 == NULL
#endif
	    ) {
	  fprintf(stderr, "failed to malloc(2) dialbuf\n");
	  goto failed_to_malloc_dialbuf;
	}

	while(1) {
		char ch;

		sem_wait(&dial_data.sem);
		pthread_mutex_lock(&dial_data.mutex);
		ch = dial_data.ch;
		pthread_mutex_unlock(&dial_data.mutex);

		/* controling with num-pad/keyboard */
		switch (ch) {
		case '0':
			write(fd, dialbuf_941_1336, bufsize);
			break;
		case '1':
			write(fd, dialbuf_697_1209, bufsize);
			break;
		case '2':
			write(fd, dialbuf_697_1336, bufsize);
			break;
		case '3':
			write(fd, dialbuf_697_1477, bufsize);
			break;
		case '4':
			write(fd, dialbuf_770_1209, bufsize);
			break;
		case '5':
			write(fd, dialbuf_770_1336, bufsize);
			break;
		case '6':
			write(fd, dialbuf_770_1477, bufsize);
			break;
		case '7':
			write(fd, dialbuf_852_1209, bufsize);
			break;
		case '8':
			write(fd, dialbuf_852_1336, bufsize);
			break;
		case '9':
			write(fd, dialbuf_852_1477, bufsize);
			break;
		case '*':
			write(fd, dialbuf_941_1209, bufsize);
			break;
		case '#':
			write(fd, dialbuf_941_1477, bufsize);
			break;
#if 0 /* NOUSE */
		case 'A':
			write(fd, dialbuf_697_1633, bufsize);
			break;
		case 'B':
			write(fd, dialbuf_770_1633, bufsize);
			break;
		case 'C':
			write(fd, dialbuf_852_1633, bufsize);
			break;
		case 'D':
			write(fd, dialbuf_941_1633, bufsize);
			break;
#endif
		case 0x03:
			goto exit_loop;
		default:
			continue;
		}
		ioctl(fd, SNDCTL_DSP_SYNC);
	}

 exit_loop:
 failed_to_malloc_dialbuf:
	free(dialbuf_941_1336);
	free(dialbuf_697_1209);
	free(dialbuf_697_1336);
	free(dialbuf_697_1477);
	free(dialbuf_770_1209);
	free(dialbuf_770_1336);
	free(dialbuf_770_1477);
	free(dialbuf_852_1209);
	free(dialbuf_852_1336);
	free(dialbuf_852_1477);
	free(dialbuf_941_1209);
	free(dialbuf_941_1477);
#if 0 /* NOUSE */
	free(dialbuf_697_1633);
	free(dialbuf_770_1633);
	free(dialbuf_852_1633);
	free(dialbuf_941_1633);
#endif
	return NULL;
}

void
getvalue(int *arg, int *index, int argc,
	 char **argv, int min, int max) {

	if (*index >= argc-1)
		Usage();

	*arg = atoi(argv[1+*index]);

	if(*arg < min || *arg > max) {
		fprintf(stderr, "Value for %s should be in the range %d..%d\n", 
				argv[*index]+2, min, max);
		exit(1);
	}
	++*index;
}

void
initialize_audiodev(void) {
	int speed_local = speed;
	int channels = 1;
	int diff;

	if(!use_audio)
		return;

	if(right || left)
		channels = 2;

	if(ioctl(fd, SNDCTL_DSP_CHANNELS, &channels)) {
		perror("ioctl(SNDCTL_DSP_CHANNELS)");
		exit(1);
	}

	if(ioctl(fd, SNDCTL_DSP_SETFMT, &format)) {
		perror("ioctl(SNDCTL_DSP_SPEED)");
		exit(1);
	}

	if(ioctl(fd, SNDCTL_DSP_SPEED, &speed_local)) {
		perror("ioctl(SNDCTL_DSP_SPEED)");
		exit(1);
	}

	diff = speed_local - speed;
	if(diff < 0)
		diff = -diff;
	if(diff > 500) {
		fprintf(stderr,
		   "Your sound card does not support the requested speed\n");
		exit(1);
	}
	if(diff != 0) {
		fprintf(stderr,
		   "Setting speed to %d\n", speed_local);
	}
	speed = speed_local;
}

void
gen_costab(void) {
	int i;
	double d;

	costab = (signed short *)malloc(tabsize * sizeof(signed short));
	if(costab == NULL) {
		perror("malloc costab");
		exit(1);
	}

	for (i = 0; i < tabsize; i++) {
		d = 2*M_PI*i;
		costab[i] = (int)((volume/100.)*16383.*cos(d/tabsize));
	}
}

char*
make_dialbuf(int f1, int f2) {
	int i1, i2, d1, d2, e1, e2, g1, g2;
	int time;
	int val;
	int bufidx = 0;
	char *dialbuf;

	if(tone_time <= 0)
		return;

	f1 *= tabsize;
	f2 *= tabsize;
	d1 = f1 / speed;
	d2 = f2 / speed;
	g1 = f1 - d1 * speed;
	g2 = f2 - d2 * speed;
	e1 = speed/2;
	e2 = speed/2;

	i1 = i2 = 0;

	time = (tone_time * speed) / 1000;

	if (left || right) {
	  /* stereo */
	  if(format == AFMT_U8) {
	    bufsize = time * 2;
	  } else {	/* AFMT_S16_LE */
	    bufsize = time * 4;
	  }
	} else  { /* !left && !right */
	  /* monoral */
	  if(format == AFMT_U8) {
	    bufsize = time;
	  } else {	/* AFMT_S16_LE */
	    bufsize = time * 2;
	  }
	}

	dialbuf = malloc(bufsize);
	if(dialbuf == NULL) {
		perror("failed to malloc dialbuf");
		exit(1);
	}

	while(--time >= 0) {
		val = costab[i1] + costab[i2];

		if(left || !right) {
			if(format == AFMT_U8) {
				dialbuf[bufidx++] = 128 + (val >> 8); 
			}
			else {	/* AFMT_S16_LE */
				dialbuf[bufidx++] = val & 0xff;
				dialbuf[bufidx++] = (val >> 8) & 0xff;
			}
		}
		if (left != right) {
			if(format == AFMT_U8) {
				dialbuf[bufidx++] = 128;
			}
			else {	/* AFMT_S16_LE */
				dialbuf[bufidx++] = 0;
				dialbuf[bufidx++] = 0;
			}
		}
		if(right) {
			if(format == AFMT_U8) {
				dialbuf[bufidx++] = 128 + (val >> 8); 
			}
			else {	/* AFMT_S16_LE */
				dialbuf[bufidx++] = val & 0xff;
				dialbuf[bufidx++] = (val >> 8) & 0xff;
			}
		}

		i1 += d1;
		if (e1 < 0) {
			e1 += speed;
			i1 += 1;
		}
		if (i1 >= tabsize)
			i1 -= tabsize;

		i2 += d2;
		if (e2 < 0) {
			e2 += speed;
			i2 += 1;
		}
		if (i2 >= tabsize)
			i2 -= tabsize;

		e1 -= g1;
		e2 -= g2;
	}

	return dialbuf;
}
