// ============================================================================
//  $Id$
//  $Name$
// ============================================================================
#ifndef __TRING_HH
#define __TRING_HH

#include "TTypes.hh"

template <class T> class Tring
{

  protected:
    enum { tDefaultCapacity = 1, tDefaultReallocationParameter = 2 };

  protected:
    T** theObjects;

  protected:
    Tint theCapacity;

    Tint theNumberOfEntries;
    Tint thePosition;
    Tint theBeginOfPosition;
    Tint theEndOfPosition;
    Tint theReallocationParameter;

  public:
    inline Tring( Tint capacity = tDefaultCapacity, 
		    Tint reallocation = tDefaultReallocationParameter );
    inline virtual ~Tring();

  public:
    inline Tint Add( T* object );
    inline Tint Remove( Tint position );
    inline Tvoid Swap();

  public:
    inline Tvoid Clear();
    inline Tbool Resize( Tint capacity );

  public:
    inline T* Next();
    inline T* Begin() const;
    inline T* End() const;
    inline T* Get( Tint position );
    inline T* Get();
    inline T* At( Tint position );

  public:
    inline Tint GetCapacity() const;
    inline Tint GetNumberOfEntries() const;
    inline Tint GetPosition() const;
    inline Tint GetBeginOfPosition() const;
    inline Tint GetEndOfPosition() const;
    inline Tint GetReallocationParameter() const;

  public:
    inline Tvoid SetPosition( Tint position );
    inline Tvoid SetReallocationParameter( Tint reallocation );

  public:
    inline T* operator []( Tint position );

  private:
    inline Tvoid free();
    inline Tvoid allocate( Tint capacity );

};

template <class T> inline Tring<T>::Tring( Tint capacity, Tint reallocation )
{
  if ( capacity < 1 )
    theCapacity = 1;
  else
    theCapacity = capacity;
  theReallocationParameter = reallocation;
  theNumberOfEntries = 0;
  theObjects = 0;
  thePosition = 0;
  theBeginOfPosition = 0;
  theEndOfPosition = 0;

  if ( theCapacity > 0 )
    allocate( theCapacity );
}

template <class T> inline Tring<T>::~Tring()
{
  free();
}

template <class T> inline Tint Tring<T>::Add( T* object )
{
  if ( theNumberOfEntries >= theCapacity )
    Resize( theCapacity *= theReallocationParameter );

  theObjects[ theNumberOfEntries ] = object;

  theNumberOfEntries ++;
  thePosition = theNumberOfEntries - 1;
  theEndOfPosition = theNumberOfEntries - 1;

  return( theNumberOfEntries );
}

template <class T> inline Tint Tring<T>::Remove( Tint position )
{
  if ( position < 0 || position >= theNumberOfEntries )
    return( 0 );
  thePosition = position;
  delete theObjects[ thePosition ];
  for ( Tint i = thePosition; i < theNumberOfEntries; i ++ )
    theObjects[ i ] = theObjects[ i + 1 ];
  theNumberOfEntries --;
  theEndOfPosition = theNumberOfEntries - 1;

  return( theNumberOfEntries );
}

template <class T> inline Tvoid Tring<T>::Swap()
{
  if ( theNumberOfEntries < 2 )
    return;

  Tint times = theNumberOfEntries / 2;
  for ( Tint i = 0; i < times; i ++ ) {
    T* tmp = theObjects[ i ];
    theObjects[ i ] = theObjects[ theEndOfPosition - i ];
    theObjects[ theEndOfPosition - i ] = tmp;
  }

  return;
}

template <class T> inline Tvoid Tring<T>::Clear()
{
  for ( Tint i = 0; i < theNumberOfEntries; i ++ )
    if ( theObjects[ i ] )
      delete theObjects[ i ];
  theNumberOfEntries = 0;
  thePosition = 0;
  theEndOfPosition = 0;
  return;
}

template <class T> inline Tbool Tring<T>::Resize( Tint capacity )
{
  if ( theNumberOfEntries >= capacity )
    return( Tfalse );

  theCapacity = capacity;
  T** newobjects = new T* [ theCapacity ];
  for ( Tint i = 0; i < theNumberOfEntries; i ++ )
    newobjects[ i ] = theObjects[ i ];
  delete [] theObjects;
  theObjects = newobjects;

  thePosition = theNumberOfEntries - 1;
  
  return( Ttrue );
}

template <class T> inline T* Tring<T>::Next()
{
  if ( thePosition < 0 || thePosition >= theNumberOfEntries )
    return( 0 );
  return( theObjects[ thePosition ++ ] );
}

template <class T> inline T* Tring<T>::Begin() const
{
  return( theObjects[ theBeginOfPosition ] );
}

template <class T> inline T* Tring<T>::End() const
{
  return( theObjects[ theEndOfPosition ] );
}

template <class T> inline T* Tring<T>::Get( Tint position )
{
  if ( position < 0 || position >= theNumberOfEntries )
    return( 0 );
  thePosition = position;
  return( theObjects[ thePosition ] );
}

template <class T> inline T* Tring<T>::Get()
{
  return( Get( thePosition ) );
}

template <class T> inline T* Tring<T>::At( Tint position )
{
  return( Get( position ) );
}

template <class T> inline Tint Tring<T>::GetCapacity() const
{
  return( theCapacity );
}

template <class T> inline Tint Tring<T>::GetNumberOfEntries() const
{
  return( theNumberOfEntries );
}

template <class T> inline Tint Tring<T>::GetPosition() const
{
  return( thePosition );
}

template <class T> inline Tint Tring<T>::GetBeginOfPosition() const
{
  return( theBeginOfPosition );
}

template <class T> inline Tint Tring<T>::GetEndOfPosition() const
{
  return( theEndOfPosition );
}

template <class T> inline Tint Tring<T>::GetReallocationParameter() const
{
  return( theReallocationParameter );
}

template <class T> inline Tvoid Tring<T>::SetPosition( Tint position )
{
  thePosition = position;
  return;
}

template <class T> inline Tvoid Tring<T>::SetReallocationParameter( Tint reallocation )
{
  theReallocationParameter = reallocation;
  return;
}

template <class T> inline T* Tring<T>::operator []( Tint position )
{
  return( Get( position ) );
}

template <class T> inline Tvoid Tring<T>::free()
{
  for ( Tint i = 0; i < theNumberOfEntries; i ++ )
    if ( theObjects[ i ] )
      delete theObjects[ i ];
  delete [] theObjects;

  theCapacity = 0;
  theNumberOfEntries = 0;
  thePosition = 0;
  theBeginOfPosition = 0;
  theEndOfPosition = 0;
}

template <class T> inline Tvoid Tring<T>::allocate( Tint capacity )
{
  theCapacity = capacity;
  theObjects = new T* [ theCapacity ];
}

#endif
