// ============================================================================
//  $Id: TDataElement.cc,v 1.3 2002/12/06 12:07:06 iwai Exp $
//  $Name:  $
// ============================================================================
#include "TDataElement.hh"
#include "TOutputObjectStream.hh"
#include "TOutputObjectFile.hh"
#include "TOutputObjectSocket.hh"
#include "TOutputObjectSharedMemory.hh"

TDataElement::TDataElement( Telement_t type, Tint id )
  : TStreamableObject( tObjectDataElement, id ),
    theData( 0 ), theElementType( type ), theNumberOfPrimitives( 0 )
{;}

TDataElement::TDataElement( Tvoid* data, Telement_t type, Tint id, Tint ndata )
  : TStreamableObject( tObjectDataElement, id ), 
    theData( 0 ), theElementType( type ), theNumberOfPrimitives( ndata )
{
  allocateDataSpace( data );
}

TDataElement::TDataElement( const TDataElement& right )
  : TStreamableObject( right ),
    theData( 0 ), theElementType( right.theElementType ),
    theNumberOfPrimitives( right.theNumberOfPrimitives )
{
  allocateDataSpace( right.theData );
}

TDataElement::~TDataElement()
{
  freeDataSpace();
}

Tint TDataElement::WriteData( TOutputObjectStream* output )
{
  Tint datasize = 0;
  Tstream_t streamtype = output -> GetStreamType();

  switch ( streamtype ) {
    case tFileStream:
      datasize = writeToFile( (TOutputObjectFile*)output );
      break;
    case tSocketStream:
      datasize = writeToSocket( (TOutputObjectSocket*)output );
      break;
    case tSharedMemoryStream:
      datasize = writeToSharedMemory( (TOutputObjectSharedMemory*)output );
      break;
    case tUnknownStream:
    default:
      return( 0 );
      break;
  }


  return( datasize );
}

Tint TDataElement::GetDataSize()
{
  Tsize_t total = Tsizeof( theID ) + Tsizeof( theElementType ) + Tsizeof( theNumberOfPrimitives );
  switch ( theElementType ) {
    case tTypeInt:
      total += ( Tsizeof( Tint ) ) * theNumberOfPrimitives;
      break;
    case tTypeUnsignedInt:
      total += ( Tsizeof( TUint ) ) * theNumberOfPrimitives;
      break;
    case tTypeWord:
    case tTypeUnsignedShort:
      total += ( Tsizeof( TUshort ) ) * theNumberOfPrimitives;
      break;
    case tTypeDouble:
      total += ( Tsizeof( Tdouble ) ) * theNumberOfPrimitives;
      break;
    case tTypeFloat:
      total += ( Tsizeof( Tfloat ) ) * theNumberOfPrimitives;
      break;
    case tTypeShort:
      total += ( Tsizeof( Tshort ) ) * theNumberOfPrimitives;
      break;
    case tTypeLong:
      total += ( Tsizeof( Tlong ) ) * theNumberOfPrimitives;
      break;
    case tTypeUnsignedLong:
      total += ( Tsizeof( TUlong ) ) * theNumberOfPrimitives;
      break;
    case tTypeString:
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	total += ( Tsizeof( Tint ) );
	Tsize_t ss = ( ( (Tstring*)theData )[ i ] ).size() + 1;
	total += ( Tsizeof( Tchar ) ) * ss;
      }
      break;
    case tTypeUnknown:
    case tTypeObject:
    default:
      break;
  }
  return( (Tint)total );
}

const TDataElement& TDataElement::operator=( const TDataElement& right )
{
  *( (TStreamableObject*)this ) = *( (TStreamableObject*)(&right) );

  freeDataSpace();

  theElementType = right.theElementType;
  theNumberOfPrimitives = right.theNumberOfPrimitives;

  allocateDataSpace( right.theData );

  return( *this );
}

Tbool TDataElement::operator==( const TDataElement& right ) const
{
  Tbool ret = Ttrue;
  ret &= ( *((TStreamableObject*)this) == *( (TStreamableObject*)(&right) ) );
  ret &= ( theElementType == right.theElementType );
  ret &= ( theNumberOfPrimitives == right.theNumberOfPrimitives );

  if ( ret != Tfalse ) {
    switch ( theElementType ) {
      case tTypeInt:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret &= ( ((Tint*)theData)[ i ] == ((Tint*)(right.theData))[ i ] );
	  if ( ret == Tfalse )
	    break;
	}
	break;
      case tTypeUnsignedInt:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret &= ( ((TUint*)theData)[ i ] == ((TUint*)(right.theData))[ i ] );
	  if ( ret == Tfalse )
	    break;
	}
	break;
      case tTypeUnsignedShort:
      case tTypeWord:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret &= ( ((TUshort*)theData)[ i ] == ((TUshort*)(right.theData))[ i ] );
	  if ( ret == Tfalse )
	    break;
	}
	break;
      case tTypeDouble:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret &= ( ((Tdouble*)theData)[ i ] == ((Tdouble*)(right.theData))[ i ] );
	  if ( ret == Tfalse )
	    break;
	}
	break;
      case tTypeFloat:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret &= ( ((Tfloat*)theData)[ i ] == ((Tfloat*)(right.theData))[ i ] );
	  if ( ret == Tfalse )
	    break;
	}
	break;
      case tTypeShort:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret &= ( ((Tshort*)theData)[ i ] == ((Tshort*)(right.theData))[ i ] );
	  if ( ret == Tfalse )
	    break;
	}
	break;
      case tTypeLong:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret &= ( ((Tlong*)theData)[ i ] == ((Tlong*)(right.theData))[ i ] );
	  if ( ret == Tfalse )
	    break;
	}
	break;
      case tTypeUnsignedLong:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret &= ( ((TUlong*)theData)[ i ] == ((TUlong*)(right.theData))[ i ] );
	  if ( ret == Tfalse )
	    break;
	}
	break;
      case tTypeString:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret &= ( ((Tstring*)theData)[ i ] == ((Tstring*)(right.theData))[ i ] );
	  if ( ret == Tfalse )
	    break;
	}
	break;
      case tTypeUnknown:
      case tTypeObject:
      default:
	ret &= Tfalse;
	break;
    }
  }
  return( ret );
}

inline Tbool TDataElement::operator!=( const TDataElement& right ) const
{
  Tbool ret = Tfalse;
  ret |= ( *((TStreamableObject*)this) != *((TStreamableObject*)(&right)) );
  ret |= ( theElementType != right.theElementType );
  ret |= ( theNumberOfPrimitives != right.theNumberOfPrimitives );


  if ( ret == Tfalse ) {
    switch ( theElementType ) {
      case tTypeInt:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret |= ( ((Tint*)theData)[ i ] != ((Tint*)(right.theData))[ i ] );
	  if ( ret == Ttrue )
	    break;
	}
	break;
      case tTypeUnsignedInt:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret |= ( ((TUint*)theData)[ i ] != ((TUint*)(right.theData))[ i ] );
	  if ( ret == Ttrue )
	    break;
	}
	break;
      case tTypeUnsignedShort:
      case tTypeWord:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret |= ( ((TUshort*)theData)[ i ] != ((TUshort*)(right.theData))[ i ] );
	  if ( ret == Ttrue )
	    break;
	}
	break;
      case tTypeDouble:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret |= ( ((Tdouble*)theData)[ i ] != ((Tdouble*)(right.theData))[ i ] );
	  if ( ret == Ttrue )
	    break;
	}
	break;
      case tTypeFloat:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret |= ( ((Tfloat*)theData)[ i ] != ((Tfloat*)(right.theData))[ i ] );
	  if ( ret == Ttrue )
	    break;
	}
	break;
      case tTypeShort:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret |= ( ((Tshort*)theData)[ i ] != ((Tshort*)(right.theData))[ i ] );
	  if ( ret == Ttrue )
	    break;
	}
	break;
      case tTypeLong:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret |= ( ((Tlong*)theData)[ i ] != ((Tlong*)(right.theData))[ i ] );
	  if ( ret == Ttrue )
	    break;
	}
	break;
      case tTypeUnsignedLong:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret |= ( ((TUlong*)theData)[ i ] != ((TUlong*)(right.theData))[ i ] );
	  if ( ret == Ttrue )
	    break;
	}
	break;
      case tTypeString:
	for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	  ret |= ( ((Tstring*)theData)[ i ] != ((Tstring*)(right.theData))[ i ] );
	  if ( ret == Ttrue )
	    break;
	}
	break;
      case tTypeUnknown:
      case tTypeObject:
      default:
	ret |= Ttrue;
	break;
    }
  }
  return( ret );
}

Tint TDataElement::writeToFile( TOutputObjectFile* ofile )
{
  Tsize_t datasize = 0;
  static const Tsize_t nmemb = 1;
  Tsize_t ss = 0;
  Tstring strbuf;


  // write ID
  ss = fwrite( &theID, Tsizeof( theID ), nmemb, ofile -> GetFileStream() );
  if ( ss != nmemb )
    perror( "TDataElement::writeToFile" );
  else
    datasize += Tsizeof( theID );


  // write Element Type
  ss = fwrite( &theElementType, Tsizeof( theElementType ), nmemb, ofile -> GetFileStream() );
  if ( ss != nmemb )
    perror( "TDataElement::writeToFile" );
  else
    datasize += Tsizeof( theID );


  // write Number of Primitives
  ss = fwrite( &theNumberOfPrimitives, Tsizeof( theNumberOfPrimitives ), nmemb, ofile -> GetFileStream() );
  if ( ss != nmemb )
    perror( "TDataElement::writeToFile" );
  else
    datasize += Tsizeof( theNumberOfPrimitives );


  switch ( theElementType ) {

    case tTypeInt:
      Tint ii;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ii = ( (Tint*)theData )[ i ];
	ss = fwrite( &ii, Tsizeof( Tint ), nmemb, ofile -> GetFileStream() );
	if ( ss != nmemb )
	  perror( "TDataElement::writeToFile" );
	else
	  datasize += ss * Tsizeof( Tint );
      }
      break;


    case tTypeUnsignedInt:
      TUint ui;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ui = ( (TUint*)theData )[ i ];
	ss = fwrite( &ui, Tsizeof( TUint ), nmemb, ofile -> GetFileStream() );
	if ( ss != nmemb )
	  perror( "TDataElement::writeToFile" );
	else
	  datasize += ss * Tsizeof( TUint );
      }
      break;


    case tTypeUnsignedShort:
    case tTypeWord:
      TUshort us;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	us = ( (TUshort*)theData )[ i ];
	ss = fwrite( &us, Tsizeof( TUshort ), nmemb, ofile -> GetFileStream() );
	if ( ss != nmemb )
	  perror( "TDataElement::writeToFile" );
	else
	  datasize += ss * Tsizeof( TUshort );
      }
      break;


    case tTypeDouble:
      Tdouble dd;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	dd = ( (Tdouble*)theData )[ i ];
	ss = fwrite( &dd, Tsizeof( Tdouble ), nmemb, ofile -> GetFileStream() );
	if ( ss != nmemb )
	  perror( "TDataElement::writeToFile" );
	else
	  datasize += ss * Tsizeof( Tdouble );
      }
      break;


    case tTypeFloat:
      Tfloat ff;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ff = ( (Tfloat*)theData )[ i ];
	ss = fwrite( &ff, Tsizeof( Tfloat ), nmemb, ofile -> GetFileStream() );
	if ( ss != nmemb )
	  perror( "TDataElement::writeToFile" );
	else
	  datasize += ss * Tsizeof( Tfloat );
      }
      break;


    case tTypeShort:
      Tshort ww;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ww = ( (Tshort*)theData )[ i ];
	ss = fwrite( &ww, Tsizeof( Tshort ), nmemb, ofile -> GetFileStream() );
	if ( ss != nmemb )
	  perror( "TDataElement::writeToFile" );
	else
	  datasize += ss * Tsizeof( Tshort );
      }
      break;


    case tTypeLong:
      Tlong ll;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ll = ( (Tlong*)theData )[ i ];
	ss = fwrite( &ll, Tsizeof( Tlong ), nmemb, ofile -> GetFileStream() );
	if ( ss != nmemb )
	  perror( "TDataElement::writeToFile" );
	else
	  datasize += ss * Tsizeof( Tlong );
      }
      break;


    case tTypeUnsignedLong:
      TUlong ul;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ul = ( (TUlong*)theData )[ i ];
	ss = fwrite( &ul, Tsizeof( TUlong ), nmemb, ofile -> GetFileStream() );
	if ( ss != nmemb )
	  perror( "TDataElement::writeToFile" );
	else
	  datasize += ss * Tsizeof( TUlong );
      }
      break;


    case tTypeString:
      Tchar cc;
      Tint nchar;

      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	strbuf = ( (Tstring*)theData )[ i ];

	nchar = (Tint)( strbuf.size() + 1 );
	ss = fwrite( &nchar, Tsizeof( Tint ), nmemb, ofile -> GetFileStream() );
	if ( ss != nmemb )
	  perror( "TDataElement::writeToFile" );
	else
	  datasize += ss * Tsizeof( Tint );


	for ( Tint j = 0; j < nchar; j ++ ) {
	  if ( j == ( nchar - 1 ) )
	    cc = '\0';
	  else
	    cc = strbuf[ j ];
	  ss = fwrite( &cc, Tsizeof( Tchar ), nmemb, ofile -> GetFileStream() );
	  if ( ss != nmemb )
	    perror( "TDataElement::writeToFile" );
	  else
	    datasize += ss * Tsizeof( Tchar );
	}
      }
      break;


    case tTypeUnknown:
    case tTypeObject:
    default:
      break;
  }

  return( (Tint)datasize );
}

Tint TDataElement::writeToSocket( TOutputObjectSocket* osocket )
{
  Tsize_t datasize = 0;
  Tsize_t ssize;
  Tstring strbuf;

  
  // write ID
  ssize = Tsizeof( theID );
  if ( send( osocket->GetServerDescriptor(), &theID, ssize, 0 ) != (Tint)ssize )
    perror( "TDataElement::writeToSocket" );
  else
    datasize += ssize;


  // write Element Type
  ssize = Tsizeof( theElementType );
  if ( send( osocket->GetServerDescriptor(), &theElementType, ssize, 0 ) != (Tint)ssize )
    perror( "TDataElement::writeToSocket" );
  else
    datasize += ssize;


  // write Number of Primitives
  ssize = Tsizeof( theNumberOfPrimitives );
  if ( send( osocket->GetServerDescriptor(), &theNumberOfPrimitives, ssize, 0 ) != (Tint)ssize )
    perror( "TDataElement::writeToSocket" );
  else
    datasize += ssize;


  switch ( theElementType ) {


    case tTypeInt:
      Tint ii;
      ssize = Tsizeof( Tint );
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ii = ( (Tint*)theData )[ i ];
	if ( send( osocket -> GetServerDescriptor(), &ii, ssize, 0 ) != (Tint)ssize )
	  perror( "TDataElement::writeToSocket" );
	else
	  datasize += ssize;
      }
      break;


    case tTypeUnsignedInt:
      TUint ui;
      ssize = Tsizeof( TUint );
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ui = ( (TUint*)theData )[ i ];
	if ( send( osocket -> GetServerDescriptor(), &ui, ssize, 0 ) != (Tint)ssize )
	  perror( "TDataElement::writeToSocket" );
	else
	  datasize += ssize;
      }
      break;



    case tTypeUnsignedShort:
    case tTypeWord:
      TUshort us;
      ssize = Tsizeof( TUshort );
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	us = ( (TUshort*)theData )[ i ];
	if ( send( osocket -> GetServerDescriptor(), &us, ssize, 0 ) != (Tint)ssize )
	  perror( "TDataElement::writeToSocket" );
	else
	  datasize += ssize;
      }
      break;


    case tTypeDouble:
      Tdouble dd;
      ssize = Tsizeof( Tdouble );
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	dd = ( (Tdouble*)theData )[ i ];
	if ( send( osocket -> GetServerDescriptor(), &dd, ssize, 0 ) != (Tint)ssize )
	  perror( "TDataElement::writeToSocket" );
	else
	  datasize += ssize;
      }
      break;


    case tTypeFloat:
      Tfloat ff;
      ssize = Tsizeof( Tfloat );
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ff = ( (Tfloat*)theData )[ i ];
	if ( send( osocket -> GetServerDescriptor(), &ff, ssize, 0 ) != (Tint)ssize )
	  perror( "TDataElement::writeToSocket" );
	else
	  datasize += ssize;
      }
      break;


    case tTypeShort:
      Tshort ww;
      ssize = Tsizeof( Tshort );
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ww = ( (TUshort*)theData )[ i ];
	if ( send( osocket -> GetServerDescriptor(), &ww, ssize, 0 ) != (Tint)ssize )
	  perror( "TDataElement::writeToSocket" );
	else
	  datasize += ssize;
      }
      break;



    case tTypeLong:
      Tlong ll;
      ssize = Tsizeof( Tlong );
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ll = ( (Tlong*)theData )[ i ];
	if ( send( osocket -> GetServerDescriptor(), &ll, ssize, 0 ) != (Tint)ssize )
	  perror( "TDataElement::writeToSocket" );
	else
	  datasize += ssize;
      }
      break;


    case tTypeUnsignedLong:
      TUlong ul;
      ssize = Tsizeof( TUlong );
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ul = ( (TUlong*)theData )[ i ];
	if ( send( osocket -> GetServerDescriptor(), &ul, ssize, 0 ) != (Tint)ssize )
	  perror( "TDataElement::writeToSocket" );
	else
	  datasize += ssize;
      }
      break;



    case tTypeString:
      Tchar cc;
      Tint nchar;

      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	strbuf = ( (Tstring*)theData )[ i ];
	nchar = (Tint)( strbuf.size() + 1 );
	ssize = Tsizeof( Tint );
	if ( send( osocket -> GetServerDescriptor(), &nchar, ssize, 0 ) != (Tint)ssize )
	  perror( "TDataElement::writeToSocket" );
	else
	  datasize += ssize;
	
	ssize = Tsizeof( Tchar );
	for ( Tint j = 0; j < nchar; j ++ ) {
	  if ( j == ( nchar - 1 ) )
	    cc = '\0';
	  else
	    cc = strbuf[ j ];
	  if ( send( osocket -> GetServerDescriptor(), &cc, ssize, 0 ) != (Tint)ssize )
	    perror( "TDataElement::writeToSocket" );
	  else
	    datasize += ssize;
	}
      }
      break;



    case tTypeUnknown:
    case tTypeObject:
    default:
      break;
  }
  
  return( (Tint)datasize );
}

Tint TDataElement::writeToSharedMemory( TOutputObjectSharedMemory* omemory )
{
  Tsize_t datasize = 0;
  Tvoid* ptr = omemory -> GetAddress();
  Tstring strbuf;

  
  // write ID
  *( (Tint*)ptr ) = theID;
  datasize += Tsizeof( theID );
  ( (Tint*)ptr ) ++;


  // write Element Type
  *( (Telement_t*)ptr ) = theElementType;
  datasize += Tsizeof( theElementType );
  ( (Telement_t*)ptr ) ++;


  // write Number of Primitives
  *( (Tint*)ptr ) = theNumberOfPrimitives;
  datasize += Tsizeof( theNumberOfPrimitives );
  ( (Tint*)ptr ) ++;


  switch ( theElementType ) {


    case tTypeInt:
      Tint ii;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ii = ( (Tint*)theData )[ i ];
	*( (Tint*)ptr ) = ii;
	datasize += Tsizeof( Tint );
	( (Tint*)ptr ) ++;
      }
      break;


    case tTypeUnsignedInt:
      TUint ui;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ui = ( (TUint*)theData )[ i ];
	*( (TUint*)ptr ) = ui;
	datasize += Tsizeof( TUint );
	( (TUint*)ptr ) ++;
      }
      break;


    case tTypeUnsignedShort:
    case tTypeWord:
      TUshort us;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	us = ( (TUshort*)theData )[ i ];
	*( (TUshort*)ptr ) = us;
	datasize += Tsizeof( TUshort );
	( (TUshort*)ptr ) ++;
      }
      break;


    case tTypeDouble:
      Tdouble dd;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	dd = ( (Tdouble*)theData )[ i ];
	*( (Tdouble*)ptr ) = dd;
	datasize += Tsizeof( Tdouble );
	( (Tdouble*)ptr ) ++;
      }
      break;


    case tTypeFloat:
      Tfloat ff;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ff = ( (Tfloat*)theData )[ i ];
	*( (Tfloat*)ptr ) = ff;
	datasize += Tsizeof( Tfloat );
	( (Tdouble*)ptr ) ++;
      }
      break;


    case tTypeShort:
      Tshort ww;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ww = ( (Tshort*)theData )[ i ];
	*( (Tshort*)ptr ) = ww;
	datasize += Tsizeof( Tshort );
	( (Tshort*)ptr ) ++;
      }
      break;


    case tTypeLong:
      Tlong ll;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ll = ( (Tlong*)theData )[ i ];
	*( (Tlong*)ptr ) = ll;
	datasize += Tsizeof( Tlong );
	( (Tlong*)ptr ) ++;
      }
      break;


    case tTypeUnsignedLong:
      TUlong ul;
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	ul = ( (TUlong*)theData )[ i ];
	*( (TUlong*)ptr ) = ul;
	datasize += Tsizeof( TUlong );
	( (TUlong*)ptr ) ++;
      }
      break;



    case tTypeString:
      Tchar cc;
      Tint nchar;

      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	strbuf = ( (Tstring*)theData )[ i ];
	nchar = (Tint)( strbuf.size() + 1 );


	// write Number of characters
	*( (Tint*)ptr ) = nchar;
	datasize += Tsizeof( nchar );
	( (Tint*)ptr ) ++;

	for ( Tint j = 0; j < nchar; j ++ ) {
	  if ( j == ( nchar - 1 ) )
	    cc = '\0';
	  else
	    cc = strbuf[ j ];

	  // write a character
	  *( (Tchar*)ptr ) = cc;
	  datasize += Tsizeof( Tchar );
	  ( (Tchar*)ptr ) ++;
	}
      }
      break;


    case tTypeUnknown:
    case tTypeObject:
    default:
      break;
  }

  omemory -> SetAddress( ptr );
  
  return( (Tint)datasize );
}

Tostream& operator<<( Tostream& tos, const TDataElement& right )
{
  Tstring head = Twspace + Twspace + Twspace + "* Data Element, ";

  tos << head << "ID: " << right.theID;
  tos << Twspace << "Primitives: " << right.theNumberOfPrimitives;
  tos << Twspace << "Type: " << right.theElementType;

  switch ( right.theElementType ) {


    case tTypeInt:
      tos << Tslash << "Signed integer(" << Tsizeof( Tint ) << ")" << Twspace << "Data: ";
      for ( Tint i = 0; i < right.theNumberOfPrimitives; i ++ ) {
	tos << ( (Tint*)(right.theData) )[ i ];
	if ( i != right.theNumberOfPrimitives - 1 )
	  tos << "," << Tspace;
      }
      break;


    case tTypeUnsignedInt:
      tos << Tslash << "Unsigned integer(" << Tsizeof( TUint ) << ")" << Twspace << "Data: ";
      for ( Tint i = 0; i < right.theNumberOfPrimitives; i ++ ) {
	tos << ( (TUint*)(right.theData) )[ i ];
	if ( i != right.theNumberOfPrimitives - 1 )
	  tos << "," << Tspace;
      }
      break;



    case tTypeWord:
    case tTypeUnsignedShort:
      tos << Tslash << "Unsigned short integer(" << Tsizeof( TUshort ) << ")" << Twspace << "Data: ";
      for ( Tint i = 0; i < right.theNumberOfPrimitives; i ++ ) {
	tos << ( (TUshort*)(right.theData) )[ i ];
	if ( i != right.theNumberOfPrimitives - 1 )
	  tos << "," << Tspace;
      }
      break;



    case tTypeDouble:
      tos << Tslash << "Double(" << Tsizeof( Tdouble ) << ")" << Twspace << "Data: ";
      for ( Tint i = 0; i < right.theNumberOfPrimitives; i ++ ) {
	tos << ( (Tdouble*)(right.theData) )[ i ];
	if ( i != right.theNumberOfPrimitives - 1 )
	  tos << "," << Tspace;
      }
      break;



    case tTypeFloat:
      tos << Tslash << "Float(" << Tsizeof( Tfloat ) << ")" << Twspace << "Data: ";
      for ( Tint i = 0; i < right.theNumberOfPrimitives; i ++ ) {
	tos << ( (Tfloat*)(right.theData) )[ i ];
	if ( i != right.theNumberOfPrimitives - 1 )
	  tos << "," << Tspace;
      }
      break;



    case tTypeShort:
      tos << Tslash << "Signed short integer(" << Tsizeof( Tshort ) << ")" << Twspace << "Data: ";
      for ( Tint i = 0; i < right.theNumberOfPrimitives; i ++ ) {
	tos << ( (Tshort*)(right.theData) )[ i ];
	if ( i != right.theNumberOfPrimitives - 1 )
	  tos << "," << Tspace;
      }
      break;



    case tTypeLong:
      tos << Tslash << "Signed long integer(" << Tsizeof( Tlong ) << ")" << Twspace << "Data: ";
      for ( Tint i = 0; i < right.theNumberOfPrimitives; i ++ ) {
	tos << ( (Tlong*)(right.theData) )[ i ];
	if ( i != right.theNumberOfPrimitives - 1 )
	  tos << "," << Tspace;
      }
      break;


    case tTypeUnsignedLong:
      tos << Tslash << "Unsigned long integer(" << Tsizeof( TUlong ) << ")" << Twspace << "Data: ";
      for ( Tint i = 0; i < right.theNumberOfPrimitives; i ++ ) {
	tos << ( (TUlong*)(right.theData) )[ i ];
	if ( i != right.theNumberOfPrimitives - 1 )
	  tos << "," << Tspace;
      }
      break;



    case tTypeString:
      tos << Tslash << "String" << Twspace << "Data: ";
      for ( Tint i = 0; i < right.theNumberOfPrimitives; i ++ ) {
	tos << ( (Tstring*)(right.theData) )[ i ];
	if ( i != right.theNumberOfPrimitives - 1 )
	  tos << "," << Tspace;
      }
      break;


    case tTypeUnknown:
    case tTypeObject:
    default:
      tos << Tslash << "Unknown" << Twspace << "Data: ";
      tos << "unknown(or undefined) data ...";
      break;
  }

  return( tos );
}

Tvoid TDataElement::freeDataSpace()
{
  if ( theData != 0 ) {
    switch ( theElementType ) {

      case tTypeInt:
	delete [] (Tint*)theData;
	break;


      case tTypeUnsignedInt:
	delete [] (TUint*)theData;
	break;

      case tTypeWord:
      case tTypeUnsignedShort:
	delete [] (TUshort*)theData;
	break;

      case tTypeDouble:
	delete [] (Tdouble*)theData;
	break;

      case tTypeFloat:
	delete [] (Tfloat*)theData;
	break;

      case tTypeShort:
	delete [] (Tshort*)theData;
	break;

      case tTypeLong:
	delete [] (Tlong*)theData;
	break;

      case tTypeUnsignedLong:
	delete [] (TUlong*)theData;
	break;

      case tTypeString:
	delete [] (Tstring*)theData;
	break;

      case tTypeUnknown:
      case tTypeObject:
      default:
	break;
    }
  }
  theData = 0;
  theNumberOfPrimitives = 0;
  return;
}

Tvoid TDataElement::allocateDataSpace( Tvoid* data )
{
  if ( data == 0 )
    return;

  switch ( theElementType ) {

    case tTypeInt:
      theData = new Tint[ theNumberOfPrimitives ];
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	( (Tint*)(theData) )[ i ] = *( (Tint*)data );
	( (Tint*)data ) ++;
      }
      break;


    case tTypeUnsignedInt:
      theData = new TUint[ theNumberOfPrimitives ];
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	( (TUint*)(theData) )[ i ] = *( (TUint*)data );
	( (TUint*)data ) ++;
      }
      break;

    case tTypeWord:
    case tTypeUnsignedShort:
      theData = new TUshort[ theNumberOfPrimitives ];
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
  	( (TUshort*)(theData) )[ i ] = *( (TUshort*)data );
  	( (TUshort*)data ) ++;
      }
      break;

    case tTypeDouble:
      theData = new Tdouble[ theNumberOfPrimitives ];
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	( (Tdouble*)(theData) )[ i ] = *( (Tdouble*)data );
	( (Tdouble*)data ) ++;
      }
      break;

    case tTypeFloat:
      theData = new Tfloat[ theNumberOfPrimitives ];
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	( (Tfloat*)(theData) )[ i ] = *( (Tfloat*)data );
	( (Tfloat*)data ) ++;
      }
      break;

    case tTypeShort:
      theData = new Tshort[ theNumberOfPrimitives ];
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
  	( (Tshort*)(theData) )[ i ] = *( (Tshort*)data );
  	( (Tshort*)data ) ++;
      }
      break;

    case tTypeLong:
      theData = new Tlong[ theNumberOfPrimitives ];
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	( (Tlong*)(theData) )[ i ] = *( (Tlong*)data );
	( (Tlong*)data ) ++;
      }
      break;

    case tTypeUnsignedLong:
      theData = new TUlong[ theNumberOfPrimitives ];
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	( (TUlong*)(theData) )[ i ] = *( (TUlong*)data );
	( (TUlong*)data ) ++;
      }
      break;

    case tTypeString:
      theData = new Tstring[ theNumberOfPrimitives ];
      for ( Tint i = 0; i < theNumberOfPrimitives; i ++ ) {
	( (Tstring*)(theData) )[ i ] = *( (Tstring*)data );
	( (Tstring*)data ) ++;
      }
      break;

    case tTypeUnknown:
    case tTypeObject:
    default:
      break;
  }
  return;
}
