// ============================================================================
//  $Id: TSoftwareAdcModule.cc,v 1.1.1.1 2002/12/04 23:47:39 iwai Exp $
//  $Name:  $
// ============================================================================
#include "TSoftwareAdcModule.hh"
#include "TDataSegment.hh"
#include "TDataElement.hh"

Tint TSoftwareAdcModule::theSeed = (Tint)time( 0 );
TRandomEngine TSoftwareAdcModule::theRandomEngine = TRandomEngine( (Tlong)(TSoftwareAdcModule::theSeed) );

TSoftwareAdcModule::TSoftwareAdcModule( Tint nchannel, Tint scale, Trandom_t randtype )
  : TSoftwareModule( nchannel ), theScale( scale ),
    theMean( nchannel, 0 ), theSigma( nchannel, 0 ), theChannel( nchannel, 0 ),
    theRandomType( randtype )
{
  setParameters();
  Update();
}

TSoftwareAdcModule::TSoftwareAdcModule( const TSoftwareAdcModule& right )
  : TSoftwareModule( right ), theMean( right.theMean ), 
    theSigma( right.theSigma ), theChannel( right.theChannel ),
    theRandomType( right.theRandomType )
{
  SetSeed( right.GetSeed() );
  SetRandomEngine( right.GetRandomEngine() );
}

TSoftwareAdcModule::~TSoftwareAdcModule()
{;}

Tint TSoftwareAdcModule::Clear()
{
  for ( Tint i = 0; i < theNumberOfChannels; i ++ )
    theChannel[ i ] = 0;
  return( theStatus = tStatusSuccess );
}

Tint TSoftwareAdcModule::Update()
{
  switch ( theRandomType ) {
    case tRandomLandau:
    case tRandomExponential:
    case tRandomGaussian:
    case tRandomChiSquare:
    case tRandomGamma:
    case tRandomPoisson:
    case tRandomUnknown:
    default:
      fillGaussian();
      break;
  }

  return( theStatus );
}

Tint TSoftwareAdcModule::Initialize()
{
  Clear();
  for ( Tint i = 0; i < theNumberOfChannels; i ++ ) {
    theMean[ i ] = 0;
    theSigma[ i ] = 0;
  }
  setParameters();
  return( theStatus = tStatusSuccess );
}

Tvoid TSoftwareAdcModule::FillData( const TDataSegment& segment )
{
  for ( Tint i = 0; i < theNumberOfChannels; i ++ ) {
    TDataElement element( &theChannel[ i ], tTypeInt, itostr( i ), 1 );
    ((TDataElementList*)(&segment)) -> push_back( element );
  }
  return;
}

Tvoid TSoftwareAdcModule::FillData( const TDataElement& element )
{
  Tint ch = atoi( element.GetID().c_str() );
  if ( ch < 0 || ch >= theNumberOfChannels ) {
    Tcerr << "TSoftwareAdcModule::FillData: invalid ID" << Tendl;
    theStatus = -EFAULT;
    ((TDataElement*)(&element)) -> FillData( &theStatus, tTypeInt, 1 );
  } else {
    ((TDataElement*)(&element)) -> FillData( &theChannel[ ch ], tTypeInt, 1 );
  }
  return;
}

const TSoftwareAdcModule& TSoftwareAdcModule::operator=( const TSoftwareAdcModule& right )
{
  *( (TSoftwareModule*)this ) = *( (TSoftwareModule*)(&right) );
  theMean = right.theMean;
  theSigma = right.theSigma;
  theChannel = right.theChannel;
  theRandomType = right.theRandomType;
  SetSeed( right.GetSeed() );
  SetRandomEngine( right.GetRandomEngine() );
  return( *this );
}

Tbool TSoftwareAdcModule::operator==( const TSoftwareAdcModule& right ) const
{
  Tbool ret = Ttrue;
  ret &= ( *( (TSoftwareModule*)this ) == *( (TSoftwareModule*)(&right) ) );
  ret &= ( theMean == right.theMean );
  ret &= ( theSigma == right.theSigma );
  ret &= ( theChannel == right.theChannel );
  ret &= ( theRandomType == right.theRandomType );
  return( ret );
}

Tbool TSoftwareAdcModule::operator!=( const TSoftwareAdcModule& right ) const
{
  Tbool ret = Tfalse;
  ret |= ( *( (TSoftwareModule*)this ) != *( (TSoftwareModule*)(&right) ) );
  ret |= ( theMean != right.theMean );
  ret |= ( theSigma != right.theSigma );
  ret |= ( theChannel != right.theChannel );
  ret |= ( theRandomType != right.theRandomType );
  return( ret );
}

Tvoid TSoftwareAdcModule::setParameters()
{
  TRandomFlat randomFlat( TSoftwareAdcModule::theRandomEngine );

  for ( Tint i = 0; i < theNumberOfChannels; i ++ ) {
    theMean[ i ] = randomFlat.fireInt( theScale );
    Tdouble distance = (Tdouble)theMean[ i ];
    if ( (Tdouble)theScale / 2.0 < distance )
      distance = theScale - distance;

    // 3sigma = 0.997
    theSigma[ i ] = randomFlat.fireInt( (Tint)( distance / 3.0 ) );
  }

  return;
}

Tvoid TSoftwareAdcModule::fillGaussian()
{
  TRandomGaussian randomGaussian( TSoftwareAdcModule::theRandomEngine );
  for ( Tint i = 0; i < theNumberOfChannels; i ++ ) {
    Tint idata = (Tint)( randomGaussian.fire( (Tdouble)theMean[ i ], (Tdouble)theSigma[ i ] ) );
    if ( idata > theScale || idata < 0 ) {
      idata = tDataOverFlow;
      theStatus = -EFAULT;
    }
    theChannel[ i ] = idata;
  }
  return;
}
